package net.sf.vietpad;

import java.awt.datatransfer.*;
import java.awt.dnd.*;
import java.io.*;
import javax.swing.*;
import java.net.URI;

/**
 *  Text Drop Target Listener
 *
 *@author     Quan Nguyen
 *@author     Gero Herrmann
 *@version    1.0.3, 8 October 2004
 *@see        http://vietpad.sourceforge.net
 */
class TextDropTargetListener extends DropTargetAdapter
{
    private JTextArea textArea;
    private VietPad vietPad;
    private File droppedFile;

    /**
     *  Constructor for the TextDropTargetListener object
     *
     *@param  fm  instance of VietPad
     */
    public TextDropTargetListener(VietPad fm) {
        vietPad = fm;
        textArea = fm.m_editor;
    }

    /**
     *  Gives visual feedback
     *
     *@param  dtde  the DropTargetDragEvent
     */
    public void dragOver(DropTargetDragEvent dtde) {
        if (droppedFile == null) {
            DataFlavor[] flavors = dtde.getCurrentDataFlavors();
            for (int i = 0; i < flavors.length; i++) {
                if (flavors[i].isFlavorTextType()) {
                    dtde.acceptDrag(dtde.getDropAction());
                    return;
                } else if (flavors[i].isFlavorJavaFileListType()) {
                    dtde.acceptDrag(DnDConstants.ACTION_COPY);
                    return;
                }
            }
        }
        dtde.rejectDrag();
    }

    /**
     *  Handles dropped text and files
     *
     *@param  dtde  the DropTargetDropEvent
     */
    public void drop(DropTargetDropEvent dtde) {
        Transferable transferable = dtde.getTransferable();
        DataFlavor[] flavors = transferable.getTransferDataFlavors();
        
        final boolean LINUX = System.getProperty("os.name").equals("Linux");

        for (int i = 0; i < flavors.length; i++) {
//            System.out.println(flavors[i].getMimeType());
            try {
                if (flavors[i].equals(DataFlavor.stringFlavor)) {
                    final int mark = textArea.viewToModel(dtde.getLocation());
                    if (mark <= textArea.getSelectionStart() || mark > textArea.getSelectionEnd()) {
                        dtde.acceptDrop(dtde.getDropAction());
                        final String string =
                                transferable.getTransferData(DataFlavor.stringFlavor).toString().replaceAll("\r\n?", "\n");
                        
                        // This workaround is for File DnD bug on Linux (Bug ID: 4899516),
                        // in which a String rather than a List (of files) is returned when
                        // dragging file(s) from file mananger window.
                        if (LINUX) {
                            int index = string.indexOf('\n');
                            if (index != -1 && string.startsWith("file://")) {
                                URI uri = new URI(string.substring(0, index));                     
                                droppedFile = new File(uri);                     
                            }
                            if (droppedFile != null && droppedFile.exists()) {
                                // Processes one dropped file at a time in a separate thread
                                new Thread() {
                                    public void run() {
                                        if (vietPad.promptToSave()) {
                                            vietPad.openDocument(droppedFile);
                                        }
                                        droppedFile = null;
                                    }
                                }.start();                                
                            } else {
                                // If not a file, perform text copy/move
                                
                                // Fix another bug which hangs the app when highlighting selection
                                // on Linux in Java prior to 1.5
                                SwingUtilities.invokeLater(
                                    new Runnable() {
                                        public void run() {
                                            textArea.insert(string, mark);
                                            textArea.setCaretPosition(mark);
                                            textArea.setSelectionEnd(mark + string.length());
                                        }
                                    });                            
                            }
                        } else {
                            // Normal text copy/move and highlight
                            textArea.insert(string, mark);
                            textArea.setCaretPosition(mark);
                            textArea.setSelectionEnd(mark + string.length());                           
                        }

                        dtde.dropComplete(true);
                    } else {
                        textArea.setCaretPosition(mark);    // behave like Microsoft Word
                        dtde.rejectDrop();
                    }
                    return;
                } else if (flavors[i].equals(DataFlavor.javaFileListFlavor) || (LINUX && flavors[i].getPrimaryType().equals("text") && flavors[i].getSubType().equals("uri-list"))) {
                    dtde.acceptDrop(DnDConstants.ACTION_COPY);
                    
                    // Missing DataFlavor.javaFileListFlavor on Linux (Bug ID: 4899516)
                    if (flavors[i].equals(DataFlavor.javaFileListFlavor)) {
                        java.util.List fileList = (java.util.List) transferable.getTransferData(DataFlavor.javaFileListFlavor);
                        droppedFile = (File) fileList.get(0);
                    } else {
                        // This workaround is for File DnD on Linux
                        String string =
                                transferable.getTransferData(DataFlavor.stringFlavor).toString().replaceAll("\r\n?", "\n");
                        URI uri = new URI(string.substring(0, string.indexOf('\n')));                     
                        droppedFile = new File(uri);
                    }
                    
                    // Note: On Windows, Java 1.4.2 can't recognize a Unicode file name
                    // (Bug ID 4896217). Fixed in Java 1.5.
                     
                    // Processes one dropped file at a time in a separate thread
                    new Thread() {
                        public void run() {
                            if (vietPad.promptToSave()) {
                                vietPad.openDocument(droppedFile);
                            }
                            droppedFile = null;
                        }
                    }.start();
                    
                    dtde.dropComplete(true);
                    return;
                }
            } catch (Exception e) {
                e.printStackTrace();
                dtde.rejectDrop();
            }
        }
        dtde.dropComplete(false);
    }
}
