package net.sf.vietpad;

import java.awt.event.*;
import java.text.BreakIterator;
import javax.swing.text.*;
import javax.swing.JComboBox;
import java.util.*;

/**
 * Listener for Vietnamese key entries. 
 * Acts as an input preprocessor to <code>VietKey</code> module, the keyboard engine.
 * <p>
 * A listener object created from this class, when registered with a text component 
 * using the component's <code>addKeyListener</code> method,
 * gives the component the capability of Vietnamese text input. 
 * The class has numerous static methods to provide extra niceties 
 * to the registered text components. It includes support for shorthand, 
 * smart marking, and selection of popular Vietnamese input methods. A typical application is as follows:
 * <pre>
 *     JTextComponent textComp = new JTextArea();
 *     VietKeyListener keyLst = new VietKeyListener(textComp);
 *     textComp.addKeyListener(keyLst);
 *     VietKeyListener.setInputMethod(InputMethods.Telex); // default
 *     VietKeyListener.setVietModeEnabled(true); // default
 *     VietKeyListener.setSmartMark(true);
 * </pre>
 * A convenient <a href=http://prdownloads.sourceforge.net/vietpad/VietKey.JAR.zip>JAR</a> package is provided for easy integration to your program. A practical example can be found in 
 * <a href=http://vietpad.sourceforge.net>VietPad</a>, a cross-platform Vietnamese text editor.
 * 
 * @author Quan Nguyen
 * @author Quang D. Le
 *
 * @version 1.1, 3 February 2005
 *
 * @note Added Pali-Sanskrit characters, 09 Feb 02
 * @note Added Dega characters, 19 Aug 03
 */

public class VietKeyListener extends KeyAdapter {
    private JTextComponent textComponent;
    private static InputMethods inputMethod = InputMethods.Telex; // default keyboard layout
    private static boolean smartMarkOn;
    private static boolean vietModeOn = true;
    private static boolean paliSanskritDegaOn;
    private int start, end;
    private String curWord, vietWord;
    private char curChar, vietChar;
    private char keyChar, accent;
    private static Properties macroMap;
    private static boolean repeatKeyConsumed;

    private static final char ESCAPE_CHAR = '\\';
    private static final String SHIFTING_CHARS = "cmnpt"; // chars potentially cause shifting of diacritical marks
    private static final String VOWELS = "aeiouy";
    private static final String NON_ACCENTS = "!@#$%&)_={}[]|:;/>,";
    private static final int MODIFIER_MASK =
            InputEvent.META_DOWN_MASK | InputEvent.CTRL_DOWN_MASK | InputEvent.ALT_DOWN_MASK;
    private static final BreakIterator boundary = BreakIterator.getWordInstance();

    /**
     * Creates a new <CODE>VietKeyListener</CODE>.
     *
     * @param textComponent the JTextComponent to be listened
     */
    public VietKeyListener(JTextComponent textComponent)  {
        this.textComponent = textComponent;
    }
    /**
     * Creates a new <CODE>VietKeyListener</CODE>.
     *
     * @param comboBox the JComboBox to be listened
     */
    public VietKeyListener(JComboBox comboBox)  {
        JTextComponent textComponent = (JTextComponent) comboBox.getEditor().getEditorComponent();
        textComponent.addKeyListener(new VietKeyListener(textComponent));
    }
    
    /**
     * Sets Vietnamese mode for key input.
     *
     * @param mode true to enable entry of Vietnamese characters (enabled by default)
     */    
    public static void setVietModeEnabled(final boolean mode) {
        vietModeOn = mode;
    }
    /**
     * Sets Pali-Sanskrit/Dega mode for key input.
     * <p>
     * This capability is dependent on VIQR input method being selected.
     *
     * @param mode true to enable entry of Pali-Sanskrit/Dega characters in addition to Vietnamese
     */    
    public static void setPaliSanskritModeEnabled(final boolean mode) {
        paliSanskritDegaOn = mode;
    }    
    /**
     * Sets the input method.
     *
     * @param method one of the supported input methods: VNI, VIQR, or Telex (default)
     */    
    public static void setInputMethod(final InputMethods method) {
        inputMethod = method;
    }    
    /**
     * Gets the current input method.
     *
     * @return the current input method
     */    
    public static InputMethods getInputMethod() {
        return inputMethod;
    }    
    /**
     * Sets the SmartMark capability on or off.
     *
     * @param smartMark true to enable automatic placement of diacritical marks on appropriate vowels in a word;<br>
     * otherwise, they must be typed immediately after the character they qualify.
     */    
    public static void setSmartMark(final boolean smartMark) {
        smartMarkOn = smartMark;
    }

    /**
     * Sets the diacritics position to follow the classic style.
     *
     * @param classic true for classic; false for modern (default)
     */
    public static void setDiacriticsPosClassic(final boolean classic) {    
        VietKey.setDiacriticsPosClassic(classic);
    }

    /**
     * Sets map for shorthand.
     *
     * @param shortHandMap the list of shorthand sequences
     */    
    public static void setMacroMap(final Properties shortHandMap) {
        macroMap = shortHandMap;
    }

    /**
     * Invoked when a key has been typed.
     *
     * @param e invoked when a key has been typed. This event occurs when a key press is followed by a key release.
     */    
    public void keyTyped(KeyEvent e)  {
        int caretpos = textComponent.getCaretPosition();
        if (caretpos == 0 || (e.getModifiersEx() & MODIFIER_MASK) != 0) {
            return;
        }

        Document doc = textComponent.getDocument();  
        try {            
            curChar = doc.getText(caretpos-1, 1).charAt(0);
        } catch (BadLocationException exc) {
            System.err.println(exc.getMessage());
        }
        
        if (curChar != ESCAPE_CHAR && !Character.isLetter(curChar))  {
            return;
        }

        keyChar = e.getKeyChar(); // register last key entry
        
        // process shorthand entries
        if (keyChar == ' ' && macroMap != null) {
            try {
                String key = getCurrentWord(caretpos, doc.getText(0, doc.getLength()));
                if (macroMap.containsKey(key)) {
                    String value = (String) macroMap.getProperty(key);
                    textComponent.select(start, end);
                    textComponent.replaceSelection(value);
                    e.consume();
                    return;
                }
            } catch (BadLocationException exc) {
                System.err.println(exc.getMessage());
            }
        }
        
        if (!vietModeOn) {
            return; // exit processing if Viet Mode is off
        }

        if (Character.isWhitespace(keyChar) || NON_ACCENTS.indexOf(keyChar) != -1 || keyChar == '\b') {
            return; // skip over keys not used for accent marks for faster typing
        }
        
        try {
            curWord = getCurrentWord(caretpos, doc.getText(0, doc.getLength()));
        } catch (BadLocationException exc) {
            System.err.println(exc.getMessage());
        }
        
        // Shift the accent to the second vowel in a two-consecutive-vowel sequence, if applicable
        if (smartMarkOn) {
            if (curWord.length() >= 2 &&
                    (SHIFTING_CHARS.indexOf(Character.toLowerCase(keyChar)) >= 0 ||
                    VOWELS.indexOf(Character.toLowerCase(keyChar)) >= 0 )) {
                try  {
                    String newWord;
                    // special case for "qu" and "gi"
                    if (curWord.length() == 2 &&
                            VOWELS.indexOf(Character.toLowerCase(keyChar)) >= 0 &&
                            (curWord.toLowerCase().startsWith("q") || curWord.toLowerCase().startsWith("g") )) {
                        newWord = VietKey.shiftAccent(curWord+keyChar, keyChar);
                        if (!newWord.equals(curWord+keyChar)) {
                            textComponent.select(start, end);
                            textComponent.replaceSelection(newWord);
                            e.consume();
                            return;
                        }
                    }

                    newWord = VietKey.shiftAccent(curWord, keyChar);
                    if (!newWord.equals(curWord)) {
                        textComponent.select(start, end);
                        textComponent.replaceSelection(newWord);
                        curWord = newWord;
                    }
                } catch (StringIndexOutOfBoundsException exc)  {
                    System.err.println("Caret out of bound! (For Shifting Marks)");
                }
            }
        }

        accent = getAccentMark(keyChar);

        try  {
            if (Character.isDigit(accent))  {
                if (smartMarkOn)  {
                    vietWord = (curChar == ESCAPE_CHAR) ?
                            String.valueOf(keyChar) : VietKey.toVietWord(curWord, accent);
                    if (!vietWord.equals(curWord)) {
                        textComponent.select(start, end);
                        textComponent.replaceSelection(vietWord);

                        if (!VietKey.isAccentRemoved() || repeatKeyConsumed) {
                            // accent removed by repeat key, not '0' key
                            e.consume();
                        }
                    }
                } else {
                    vietChar = (curChar == ESCAPE_CHAR)? keyChar: VietKey.toVietChar(curChar, accent);
                    if (vietChar != curChar) {
                        textComponent.select(caretpos-1, caretpos);
                        textComponent.replaceSelection(String.valueOf(vietChar));

                        if (!VietKey.isAccentRemoved() || repeatKeyConsumed) {
                            // accent removed by repeat key, not '0' key
                            e.consume();
                        }
                    }
                }
            }

            // Process Pali-Sanskrit/D�ga, if applicable and enabled
            else if (accent != '\0' && paliSanskritDegaOn) {
                char phanChar = (curChar == ESCAPE_CHAR)? keyChar: accent;
                textComponent.select(caretpos-1, caretpos);
                textComponent.replaceSelection(String.valueOf(phanChar));
                e.consume();
            }
        }
        catch (StringIndexOutOfBoundsException exc)  {
            System.err.println("Caret out of bound!");
        }
    }

    /**
     * Sets to consume the accent key when it is repeated to remove the diacritical mark just entered.
     *
     * @param mode true to consume the accent key when it is used to remove the diacritical mark just entered, false otherwise
     */
    public static void consumeRepeatKey(final boolean mode) {
        repeatKeyConsumed = mode;
    }

    /**
     * Returns the accent mark (or the character if PaliSanskritDega is turned on).
     *
     * The diacritical marks follows VNI input style:<p>
     * <code>1:', 2:`, 3:?, 4:~, 5:., 6:^, 7:+, 8:(, 9:-, 0:remove diacritics<br>
     * Also, repeating the accent key removes the accent just entered.</code>
     *
     * @param   keyChar     key input
     * @return  accent      a Vietnamese diacritic mark
     *    or    phanChar    a Pali-Sanskrit/Dega character, when applicable
     */
    private char getAccentMark(char keyChar) {
        char accent = '\0';

        // VNI Input Method
        if (inputMethod == InputMethods.VNI) {        
            if (Character.isDigit(keyChar)) {
                accent = keyChar;
            }
        }
        // VIQR Input Method
        // Added Pali-Sanskrit characters, 09 Feb 02
        // Added Dega characters, 19 Aug 03
        else if (inputMethod == InputMethods.VIQR)  {        
            char phanChar = '\0'; // initialized to null character
            if (!Character.isLetterOrDigit(keyChar)) {
                switch (keyChar)  {
                    case '\'': accent = '1'; break;
                    case '`':  accent = '2'; break;
                    case '?':  accent = '3'; break;
                    case '~':  accent = '4';
                        switch (curChar) {
                            case 'R': phanChar = '\u1E5C'; break;
                            case 'r': phanChar = '\u1E5D'; break;
                            case 'L': phanChar = '\u1E38'; break;
                            case 'l': phanChar = '\u1E39'; break;
                            case 'N': phanChar = '\u00D1'; break;
                            case 'n': phanChar = '\u00F1'; break;
                        }
                        break;
                    case '.':  accent = '5';
                        switch (curChar) {
                            case 'R': phanChar = '\u1E5A'; break;
                            case 'r': phanChar = '\u1E5B'; break;
                            case 'L': phanChar = '\u1E36'; break;
                            case 'l': phanChar = '\u1E37'; break;
                            case 'M': phanChar = '\u1E42'; break;
                            case 'm': phanChar = '\u1E43'; break;
                            case 'H': phanChar = '\u1E24'; break;
                            case 'h': phanChar = '\u1E25'; break;
                            case 'T': phanChar = '\u1E6C'; break;
                            case 't': phanChar = '\u1E6D'; break;
                            case 'D': phanChar = '\u1E0C'; break;
                            case 'd': phanChar = '\u1E0D'; break;
                            case 'N': phanChar = '\u1E46'; break;
                            case 'n': phanChar = '\u1E47'; break;
                            case 'S': phanChar = '\u1E62'; break;
                            case 's': phanChar = '\u1E63'; break;
                        }
                        break;
                    case '^':  accent = '6'; break;
                    case '*':
                    case '+':  accent = '7'; break;
                    case '(':  accent = '8';
                        // Dega characters
                        switch (curChar) {
                            case 'E': phanChar = '\u0114'; break;
                            case 'e': phanChar = '\u0115'; break;
                            case 'I': phanChar = '\u012C'; break;
                            case 'i': phanChar = '\u012D'; break;
                            case 'O': phanChar = '\u014E'; break;
                            case 'o': phanChar = '\u014F'; break;
                            case 'U': phanChar = '\u016C'; break;
                            case 'u': phanChar = '\u016D'; break;
                        }
                        break;
                    case '-':  accent = '0'; break;
                    case '\"':
                        switch (curChar) {
                            case 'M': phanChar = '\u1E40'; break;
                            case 'm': phanChar = '\u1E41'; break;
                            case 'N': phanChar = '\u1E44'; break;
                            case 'n': phanChar = '\u1E45'; break;
                            case 'S': phanChar = '\u015A'; break;
                            case 's': phanChar = '\u015B'; break;
                            case ESCAPE_CHAR: phanChar = '\"'; break;
                        }
                        break;
                    case '<':
                        // Dega characters
                        switch (curChar) {
                            case 'C': phanChar = '\u010C'; break;
                            case 'c': phanChar = '\u010D'; break;
                            case 'E': phanChar = '\u011A'; break;
                            case 'e': phanChar = '\u011B'; break;
                            case ESCAPE_CHAR: phanChar = '<'; break;
                        }
                        break;
                }
            } else if (keyChar == 'D' || keyChar == 'd') {
                accent = '9';
            } else {
                // Sanskrit characters
                switch (keyChar)  {
                    case 'A': if (curChar == 'A') phanChar = '\u0100'; break;
                    case 'a': if (curChar == 'a') phanChar = '\u0101'; break;
                    case 'I': if (curChar == 'I') phanChar = '\u012A'; break;
                    case 'i': if (curChar == 'i') phanChar = '\u012B'; break;
                    case 'U': if (curChar == 'U') phanChar = '\u016A'; break;
                    case 'u': if (curChar == 'u') phanChar = '\u016B'; break;
                    case 'E': if (curChar == 'E') phanChar = '\u0112'; break;
                    case 'e': if (curChar == 'e') phanChar = '\u0113'; break;
                    case 'O': if (curChar == 'O') phanChar = '\u014C'; break;
                    case 'o': if (curChar == 'o') phanChar = '\u014D'; break;
                }
            }

            // return Sanskrit/Dega characters, if valid and enabled
            if (phanChar != '\0' && paliSanskritDegaOn)  {
                return phanChar;
            }
        }
        // Telex Input Method     
        else if (inputMethod == InputMethods.Telex)  {       
            if (Character.isLetter(keyChar)) {
                switch (keyChar)  {
                    case 'S':
                    case 's': accent = '1'; break;
                    case 'F':
                    case 'f': accent = '2'; break;
                    case 'R':
                    case 'r': accent = '3'; break;
                    case 'X':
                    case 'x': accent = '4'; break;
                    case 'J':
                    case 'j': accent = '5'; break;
                    case 'A':
                    case 'a':
                    case 'E':
                    case 'e':
                    case 'O':
                    case 'o': accent = '6'; break;
                    case 'W':
                    case 'w': accent = '7'; break;
                    case 'D':
                    case 'd': accent = '9'; break;
                    case 'Z':
                    case 'z': accent = '0'; break;
                }

                // Determine accent for common keys shared among a, e, o, w in Telex mode
                if (accent == '6' || accent == '7') {
                    accent = VietKey.getAccentInTelex(curWord, keyChar, accent);
                }
            }
        }

        return accent;
    }

    /**
     * Gets word at a position, such as at a caret.
     */
    private String getCurrentWord(int pos, String source) {
//        BreakIterator boundary = BreakIterator.getWordInstance();
        boundary.setText(source);
        end = boundary.following(pos-1);
        start = boundary.previous();
        end = pos; // fine-tune word end
        return source.substring(start, end);
    }
}
