/*
 *  Copyright 1999-2002 Matthew Robinson and Pavel Vorobiev.
 *  All Rights Reserved.
 *
 *  ===================================================
 *  This program contains code from the book "Swing"
 *  2nd Edition by Matthew Robinson and Pavel Vorobiev
 *  http://www.spindoczine.com/sbe/
 *  ===================================================
 */

package net.sf.vietpad;

import java.awt.*;
import java.awt.datatransfer.*;
import java.awt.dnd.DropTarget;
import java.awt.event.*;
import java.io.*;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.prefs.Preferences;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.*;
import javax.swing.undo.*;

/**
 *  VietPad
 *
 *@author     Quan Nguyen
 *@author     Gero Herrmann
 *@created    January 26, 2002
 *@version    1.6, 14 April 2005
 *@see        http://vietpad.sourceforge.net/
 */
public class VietPad extends JFrame
{   
    final static boolean MAC_OS_X = System.getProperty("os.name").startsWith("Mac");
    final static int TAB_SIZE = 4;
    final static Locale VIETNAM = new Locale("vi", "VN");
    private final Clipboard clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
    private final Rectangle screen =
            GraphicsEnvironment.getLocalGraphicsEnvironment().getMaximumWindowBounds();
    private final JPopupMenu popup = new JPopupMenu();
    private final UndoManager m_undo = new UndoManager();

    protected final int MENU_MASK = Toolkit.getDefaultToolkit().getMenuShortcutKeyMask();
    protected final Preferences prefs = Preferences.userRoot().node("/net/sourceforge/vietpad");
    protected final UndoableEditSupport undoSupport = new UndoableEditSupport();
    protected final static String APP_NAME = "VietPad";
    protected final File supportDir = new File(System.getProperty("user.home") +
            (MAC_OS_X ? "/Library/Application Support/" + APP_NAME : "/." + APP_NAME.toLowerCase()));

    private boolean m_textChanged = true;
    private boolean toolBarOn;
    protected boolean smartMarkOn, vietModeOn, defaultEOLOn, localeVietOn;
    private Action actionSave, actionCut, actionCopy, actionPaste, actionDelete, actionSelectAll;
    private Action m_redoAction, m_undoAction;
    private Document m_doc;
    private JMenuItem itemCut, itemCopy, itemDelete, itemPaste;
    private RawListener rawListener;
    private javax.swing.filechooser.FileFilter utf16Filter, utf8Filter, macFilter, winFilter;
    private String encoding;

    protected ResourceBundle myResources;
    protected FindDialog m_findDialog;
    protected JButton vietModeLabel;
    protected String selectedInputMethod;
    protected File m_currentFile;
    protected Font m_font;
    protected JFileChooser m_chooser;
    protected JMenuBar menuBar;
    protected JScrollPane scrollPane;
    protected JTextArea m_editor;
    protected JToolBar m_toolBar;
    protected static Locale systemDefault;
    private JFrame helptopicsFrame;


    /**
     *  Constructor for the VietPad Object
     */
    public VietPad() {
        super(APP_NAME);
        setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);

        if (MAC_OS_X && Locale.getDefault().getCountry().equals("VN")) {
            Locale.setDefault(VIETNAM);
        }
        systemDefault = Locale.getDefault();
        Locale.setDefault(
                (localeVietOn = prefs.getBoolean("localeVN", false)) || systemDefault.getLanguage().equals("vi")
                 ? VIETNAM : Locale.US);
        myResources = ResourceBundle.getBundle("Resources");

        setSize(
                snap(prefs.getInt("frameWidth", 500), 300, screen.width),
                snap(prefs.getInt("frameHeight", 360), 150, screen.height));
        setLocation(
                snap(
                prefs.getInt("frameX", (screen.width - getWidth()) / 2),
                screen.x, screen.x + screen.width - getWidth()),
                snap(
                prefs.getInt("frameY", screen.y + (screen.height - getHeight()) / 3),
                screen.y, screen.y + screen.height - getHeight()));
        m_font = new Font(
                prefs.get("fontName", MAC_OS_X ? "Lucida Grande" : "Tahoma"),
                prefs.getInt("fontStyle", Font.PLAIN),
                prefs.getInt("fontSize", 12));
        toolBarOn = prefs.getBoolean("toolBar", true);
        vietModeOn = prefs.getBoolean("vietMode", true);
        smartMarkOn = prefs.getBoolean("smartMark", true);
        defaultEOLOn = prefs.getBoolean("defaultEOL", true);
        selectedInputMethod = prefs.get("inputMethod", "Telex");// Only Telex is a National Standard
        try {
            UIManager.setLookAndFeel(prefs.get("lookAndFeel", UIManager.getSystemLookAndFeelClassName()));
        } catch (Exception e) {
            e.printStackTrace();
            // keep default LAF
        }

        // For use with VietIME Input Method (http://vietime.sf.net)
//      if (Locale.getDefault().getLanguage().equals("vi")) {
//                        Locale VIET_VNI_SM = new Locale("vi", "", "VNI SmartMark");
//                        Locale VIET_VIQR_SM = new Locale("vi", "", "VIQR SmartMark");
//                        Locale VIET_TELEX_SM = new Locale("vi", "", "Telex SmartMark");
//                        Locale VIET_VNI = new Locale("vi", "", "VNI");
//                        Locale VIET_VIQR = new Locale("vi", "", "VIQR");
//                        Locale VIET_TELEX = new Locale("vi", "", "Telex");
//                        Locale[] SUPPORTED_LOCALES = {
//                            VIET_VNI_SM, VIET_VIQR_SM, VIET_TELEX_SM, VIET_VNI, VIET_VIQR, VIET_TELEX};
//
//          String inputMethodLocale = prefs.get("inputMethodLocale", "vi__VNI SmartMark");
//          for (int i = 0; i < SUPPORTED_LOCALES.length; i++) {
//              if (inputMethodLocale.equals(SUPPORTED_LOCALES[i].toString())) {
//                  getInputContext().selectInputMethod(SUPPORTED_LOCALES[i]); // set Input method's locale
//                  break;
//              }
//          }
//      }

        rawListener = new RawListener();
        m_editor = new JTextAreaAA();// JTextAreaAA if use Anti-aliasing
        m_doc = m_editor.getDocument();
        m_doc.addUndoableEditListener(rawListener);
        undoSupport.addUndoableEditListener(new SupportListener());

        m_editor.setTabSize(TAB_SIZE);
        m_editor.setFont(m_font);
        m_editor.setWrapStyleWord(true);
        m_editor.setMargin(new Insets(3, 2, 1, 1));
        m_editor.getInputMap().put(KeyStroke.getKeyStroke("ctrl H"), "none"); // Workaround Bug ID: 6249912 & 4782077
        
        m_editor.setDragEnabled(true);
        new DropTarget(m_editor, new TextDropTargetListener(this));

        scrollPane = new JScrollPane(m_editor);
        scrollPane.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_ALWAYS);
        scrollPane.setAutoscrolls(true);
        getContentPane().add(scrollPane);

        vietModeLabel = new JButton(vietModeOn ? "V" : "E");
        vietModeLabel.setFont(vietModeLabel.getFont().deriveFont(Font.PLAIN, 24));
        vietModeLabel.setForeground(Color.GRAY);
        vietModeLabel.setContentAreaFilled(false);
        vietModeLabel.setBorderPainted(false);
        vietModeLabel.setFocusable(false);
        vietModeLabel.setToolTipText(myResources.getString("Keyboard_Mode"));

        // Add VietKeyListener to JTextArea
        m_editor.addKeyListener(new VietKeyListener(m_editor));
        VietKeyListener.setInputMethod(InputMethods.valueOf(selectedInputMethod));
//        VietKeyListener.setInputMethod(Enum.valueOf(InputMethods.Telex.getDeclaringClass(), selectedInputMethod)); // Java 1.5       
        VietKeyListener.setSmartMark(smartMarkOn);
        VietKeyListener.setVietModeEnabled(vietModeOn);
//        VietKeyListener.setPaliSanskritModeEnabled(true);

        setJMenuBar(menuBar = createMenuBar());
        m_toolBar.setVisible(toolBarOn);

        addWindowListener(
            new WindowAdapter()
            {
                public void windowClosing(WindowEvent e) {
                    quit();
                }

                public void windowActivated(WindowEvent e) {
                    m_editor.requestFocus();
                    updatePaste();
                }

                public void windowOpened(WindowEvent e) {
                    setExtendedState(prefs.getInt("windowState", Frame.NORMAL));
                }
            });

        m_editor.addCaretListener(
            new CaretListener()
            {
                public void caretUpdate(CaretEvent e) {
                    updateCutCopyDelete(e.getDot() != e.getMark());
                }
            });

        m_chooser = new JFileChooser();
        utf8Filter = new SimpleFilter("*", "Unicode (UTF-8)");
        utf16Filter = new SimpleFilter("*", "Unicode (UTF-16)");
        macFilter = new SimpleFilter("*", "Western (Mac OS Roman)");
        winFilter = new SimpleFilter("*", "Western (Windows Latin 1)");
        m_chooser.setAcceptAllFileFilterUsed(false);
        m_chooser.addChoosableFileFilter(utf8Filter);
        m_chooser.addChoosableFileFilter(utf16Filter);
        m_chooser.addChoosableFileFilter(macFilter);
        m_chooser.addChoosableFileFilter(winFilter);
        m_chooser.setFileFilter(m_chooser.getChoosableFileFilters()[prefs.getInt("filterIndex", 0)]);
        m_chooser.setCurrentDirectory(new File(
                prefs.get("currentDirectory", System.getProperty("user.home"))));

        try {
            if (!supportDir.exists()) supportDir.mkdirs();
        } catch (Exception e) {
            e.printStackTrace();
        }
        newDocument();
    }


    /**
     *  Create an icon from a graphics resource
     *
     *@param  imageName  Name of icon
     *@return            the icon created
     */
    private ImageIcon createIcon(String imageName) {
        return new ImageIcon(getClass().getResource("/org/gjt/sp/jedit/icons/" + imageName + ".png"));
    }


    /**
     *  Create Menu bar
     *
     *@return    the MenuBar Object
     */
    private JMenuBar createMenuBar() {
        final JMenuBar menuBar = new JMenuBar();
        JMenu menu;
        JMenu subMenu;
        JMenuItem item;

        // File menu

        menu = new JMenu(myResources.getString("File"));
        menu.setMnemonic('f');

        Action actionNew =
            new AbstractAction(myResources.getString("New"))
            {
                public void actionPerformed(ActionEvent e) {
                    if (promptToSave()) {
                        newDocument();
                    }
                }
            };

        item = new JMenuItem(actionNew);
        item.setMnemonic('n');
        item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N, MENU_MASK));
        menu.add(item);

        Action actionOpen =
            new AbstractAction(myResources.getString("Open") + "...")
            {
                public void actionPerformed(ActionEvent e) {
                    if (promptToSave()) {
                        openDocument(null);
                    }
                }
            };

        item = new JMenuItem(actionOpen);
        item.setMnemonic('o');
        item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, MENU_MASK));
        menu.add(item);

        actionSave =
            new AbstractAction(myResources.getString("Save"))
            {
                public void actionPerformed(ActionEvent e) {
                    saveFile(false);
                }
            };

        item = new JMenuItem(actionSave);
        item.setMnemonic('s');
        item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, MENU_MASK));
        menu.add(item);

        Action actionSaveAs =
            new AbstractAction(myResources.getString("Save_As") + "...")
            {
                public void actionPerformed(ActionEvent e) {
                    saveFile(true);
                }
            };

        item = new JMenuItem(actionSaveAs);
        item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, MENU_MASK | InputEvent.SHIFT_MASK));
        item.setMnemonic('a');
        menu.add(item);

        menu.addSeparator();

        Action actionPageSetup =
            new AbstractAction(myResources.getString("Page_Setup") + "...")
            {
                public void actionPerformed(ActionEvent e) {
                    pageSetup();
                }
            };

        item = menu.add(actionPageSetup);
        item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_P, MENU_MASK + InputEvent.SHIFT_MASK));

        Action actionPrint =
            new AbstractAction(myResources.getString("Print") + "...")
            {
                public void actionPerformed(ActionEvent e) {
                    printPage();
                }
            };

        item = menu.add(actionPrint);
        item.setMnemonic('p');
        item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_P, MENU_MASK));

        if (!MAC_OS_X) {
            menu.addSeparator();
            Action actionExit =
                new AbstractAction(myResources.getString("Exit"))
                {
                    public void actionPerformed(ActionEvent e) {
                        quit();
                    }
                };

            item = menu.add(actionExit);
            item.setMnemonic('x');
        }

        menuBar.add(menu);

        m_toolBar = new JToolBar();
        m_toolBar.add(new SmallButton(actionNew, myResources.getString("New"), createIcon("New")));
        m_toolBar.add(new SmallButton(actionOpen, myResources.getString("Open"), createIcon("Open")));
        m_toolBar.add(new SmallButton(actionSave, myResources.getString("Save"), createIcon("Save")));
        m_toolBar.add(new SmallButton(actionPrint, myResources.getString("Print"), createIcon("Print")));
        m_toolBar.addSeparator();

        // Edit menu

        menu = new JMenu(myResources.getString("Edit"));
        menu.setMnemonic('e');

        m_undoAction =
            new AbstractAction(myResources.getString("Undo"))
            {
                public void actionPerformed(ActionEvent e) {
                    try {
                        m_undo.undo();
                        updateSave(true);
                    } catch (CannotUndoException ex) {
                        System.err.println(myResources.getString("Unable_to_undo:_") + ex);
                    }
                    updateUndoRedo();
                }
            };

        item = menu.add(m_undoAction);
        item.setMnemonic('u');
        item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Z, MENU_MASK));
        popup.add(m_undoAction);

        m_redoAction =
            new AbstractAction(myResources.getString("Redo"))
            {
                public void actionPerformed(ActionEvent e) {
                    try {
                        m_undo.redo();
                        updateSave(true);
                    } catch (CannotRedoException ex) {
                        System.err.println(myResources.getString("Unable_to_redo:_") + ex);
                    }
                    updateUndoRedo();
                }
            };

        item = menu.add(m_redoAction);
        item.setMnemonic('r');
        item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Y, MENU_MASK));
        popup.add(m_redoAction);
        popup.addSeparator();

        menu.addSeparator();

        actionCut =
            new AbstractAction(myResources.getString("Cut"))
            {
                public void actionPerformed(ActionEvent e) {
                    m_editor.cut();
                    updatePaste();
                }
            };

        itemCut = menu.add(actionCut);
        itemCut.setMnemonic('t');
        itemCut.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X, MENU_MASK));
        popup.add(actionCut);

        actionCopy =
            new AbstractAction(myResources.getString("Copy"))
            {
                public void actionPerformed(ActionEvent e) {
                    m_editor.copy();
                    updatePaste();
                }
            };

        itemCopy = menu.add(actionCopy);
        itemCopy.setMnemonic('c');
        itemCopy.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_C, MENU_MASK));
        popup.add(actionCopy);

        actionPaste =
            new AbstractAction(myResources.getString("Paste"))
            {
                public void actionPerformed(ActionEvent e) {
                    undoSupport.beginUpdate();
                    m_editor.paste();
                    undoSupport.endUpdate();
                }
            };

        itemPaste = menu.add(actionPaste);
        itemPaste.setMnemonic('p');
        itemPaste.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_V, MENU_MASK));
        popup.add(actionPaste);

        actionDelete =
            new AbstractAction(myResources.getString("Delete"))
            {
                public void actionPerformed(ActionEvent e) {
                    m_editor.replaceSelection(null);
                }
            };

        itemDelete = menu.add(actionDelete);
        itemDelete.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, 0));
        popup.add(actionDelete);
        popup.addSeparator();

        menu.addSeparator();

        m_toolBar.add(new SmallButton(actionCut, myResources.getString("Cut"), createIcon("Cut")));
        m_toolBar.add(new SmallButton(actionCopy, myResources.getString("Copy"), createIcon("Copy")));
        m_toolBar.add(new SmallButton(actionPaste, myResources.getString("Paste"), createIcon("Paste")));

        m_toolBar.addSeparator();

        m_toolBar.add(new SmallButton(m_undoAction, myResources.getString("Undo"), createIcon("Undo")));
        m_toolBar.add(new SmallButton(m_redoAction, myResources.getString("Redo"), createIcon("Redo")));

        m_toolBar.addSeparator();

        Action findAction =
            new AbstractAction(myResources.getString("Find") + "...")
            {
                public void actionPerformed(ActionEvent e) {
                    if (m_findDialog == null) {
                        m_findDialog = new FindDialog(VietPad.this, 0);
                    } else {
                        m_findDialog.setSelectedIndex(0);
                    }
                    m_findDialog.setVisible(true);
                }
            };
        item = menu.add(findAction);
        item.setMnemonic('f');
        item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F, MENU_MASK));
        m_toolBar.add(new SmallButton(findAction, myResources.getString("Find"), createIcon("Find")));

        m_toolBar.addSeparator();

        Action replaceAction =
            new AbstractAction(myResources.getString("Replace") + "...")
            {
                public void actionPerformed(ActionEvent e) {
                    if (m_findDialog == null) {
                        m_findDialog = new FindDialog(VietPad.this, 1);
                    } else {
                        m_findDialog.setSelectedIndex(1);
                    }
                    m_findDialog.setVisible(true);
                }
            };

        item = menu.add(replaceAction);
        item.setMnemonic('l');
        item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_H, MENU_MASK));

        menu.addSeparator();

        actionSelectAll =
            new AbstractAction(myResources.getString("Select_All"), null)
            {
                public void actionPerformed(ActionEvent e) {
                    m_editor.selectAll();
                }
            };

        item = menu.add(actionSelectAll);
        item.setMnemonic('a');
        item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_A, MENU_MASK));
        popup.add(actionSelectAll);

        m_editor.addMouseListener(
            new MouseAdapter()
            {
                public void mousePressed(MouseEvent e) {
                    if (e.isPopupTrigger()) {
//                        int mark = m_editor.viewToModel(new Point(e.getX(), e.getY()));
//                        if (mark < m_editor.getSelectionStart() || m_editor.getSelectionEnd() < mark) {
//                            BreakIterator wb = BreakIterator.getWordInstance();
//                            wb.setText(m_editor.getText());
//                            m_editor.setSelectionStart(wb.preceding(mark));
//                            m_editor.setSelectionEnd(wb.following(mark));
//                            String word = m_editor.getSelectedText();
//                            for (int i = 0; i < 6; i++) {
//                              popup.add(VietKey.toVietWord(word, i));
//                            }
//                        }
                        popup.show(e.getComponent(), e.getX(), e.getY());
                    }
                }


                public void mouseReleased(MouseEvent e) {
                    mousePressed(e);
                }
            });

        menu.add(
            new AbstractAction(myResources.getString("Date_&_Time"), null)
            {
                DateFormat df;

                public void actionPerformed(ActionEvent e) {
                    if (df == null) {
/*                      df = DateFormat.getDateTimeInstance(DateFormat.FULL, DateFormat.SHORT, VIETNAM); */
                        df = new SimpleDateFormat(
                                "EEEE, 'ng\u00E0y' d 'th\u00E1ng' M 'n\u0103m' yyyy, HH:mm:ss Z", VIETNAM);
                    }
                    undoSupport.beginUpdate();
                    m_editor.replaceSelection(df.format(new Date()));
                    undoSupport.endUpdate();
                }
            });

        menuBar.add(menu);

        // View menu

        menu = new JMenu(myResources.getString("View"));
        menu.setMnemonic('v');

        final JMenuItem viewToolbar = new JMenuItem(toolBarOn ? myResources.getString("Hide_Toolbar") : myResources.getString("Show_Toolbar"));
        viewToolbar.addActionListener(
            new ActionListener()
            {
                public void actionPerformed(ActionEvent e) {
                    toolBarOn ^= true;
                    viewToolbar.setText(toolBarOn ? myResources.getString("Hide_Toolbar") : myResources.getString("Show_Toolbar"));
                    m_toolBar.setVisible(toolBarOn);
                }
            });
        viewToolbar.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_B, MENU_MASK));
        menu.add(viewToolbar);

        menu.addSeparator();

        subMenu = new JMenu(myResources.getString("Look_&_Feel"));

        ActionListener lafLst =
            new ActionListener()
            {
                public void actionPerformed(ActionEvent ae) {
                    updateLaF(ae.getActionCommand());
                }
            };

        ButtonGroup groupLookAndFeel = new ButtonGroup();
        UIManager.LookAndFeelInfo[] lafs = UIManager.getInstalledLookAndFeels();
        for (int i = 0; i < lafs.length; i++) {
            JRadioButtonMenuItem lafButton = new JRadioButtonMenuItem(lafs[i].getName());
            lafButton.setActionCommand(lafs[i].getClassName());
            if (UIManager.getLookAndFeel().getName().startsWith(lafs[i].getName())) {
                lafButton.setSelected(true);
            }
            lafButton.addActionListener(lafLst);
            groupLookAndFeel.add(lafButton);
            subMenu.add(lafButton);
        }

        menu.add(subMenu);

        menuBar.add(menu);

        // Help menu

        menu = new JMenu(myResources.getString("Help"));
        menu.setMnemonic('h');

        item = new JMenuItem(APP_NAME + " " + myResources.getString("Help"));

        item.addActionListener(
            new ActionListener()
            {
                public void actionPerformed(ActionEvent e) {
                    final String readme = myResources.getString("readme");
                    if (MAC_OS_X && new File(readme).exists()) {
                        try {
                            Runtime.getRuntime().exec(new String[]{"open", "-a", "Help Viewer", readme});
                        } catch (IOException x) {
                            x.printStackTrace();
                        }
                    } else {
                        if (helptopicsFrame == null) {
                            helptopicsFrame = new JFrame(VietPad.APP_NAME + " " + myResources.getString("Help"));
                            helptopicsFrame.getContentPane().setLayout(new BorderLayout());
                            HtmlPane helpPane = new HtmlPane(readme);
                            helptopicsFrame.getContentPane().add(helpPane, BorderLayout.CENTER);
                            helptopicsFrame.getContentPane().add(helpPane.getStatusBar(), BorderLayout.SOUTH);
                            helptopicsFrame.pack();
                            helptopicsFrame.setLocation((screen.width - helptopicsFrame.getWidth()) / 2, 40);
                        }
                        helptopicsFrame.setVisible(true);
                    }
                }
            });

        menu.add(item);

        if (!MAC_OS_X) {
            menu.addSeparator();
            item = new JMenuItem(myResources.getString("About_") + VietPad.APP_NAME);
            item.setMnemonic('A');
            item.addActionListener(
                new ActionListener()
                {
                    public void actionPerformed(ActionEvent e) {
                        about();
                    }
                });
            menu.add(item);
        }

        menuBar.add(menu);

        getContentPane().add(m_toolBar, BorderLayout.PAGE_START);

        return menuBar;
    }
    // end of createMenuBar()


    /**
     *  Gets the textArea attribute of the VietPad object
     *
     *@return    The textArea value
     */
    public JTextArea getTextArea() {
        return m_editor;
    }


    /**
     *  Sets the selection attribute of the VietPad object
     *
     *@param  xStart   The start of the selection
     *@param  xFinish  The end of the selection
     */
    public void setSelection(int xStart, int xFinish) {
        setSelection(xStart, xFinish, false);
    }


    /**
     *  Sets the selection of the text editor
     *
     *@param  xStart   The new selection start
     *@param  xFinish  The new selection finish
     *@param  moveUp
     */
    public void setSelection(int xStart, int xFinish, boolean moveUp) {
        if (moveUp) {
            m_editor.setCaretPosition(xFinish);
            m_editor.moveCaretPosition(xStart);
        } else {
            m_editor.select(xStart, xFinish);
        }
    }


    /**
     *  Gets the selected text of the text editor
     *
     *@return    The selection value
     */
    public String getSelection() {
        return m_editor.getSelectedText();
    }


    /**
     *  Creates a new document
     */
    protected void newDocument() {
        m_editor.setText(null);
        m_currentFile = null;
        m_doc.putProperty(DefaultEditorKit.EndOfLineStringProperty, System.getProperty("line.separator"));
        setTitle(myResources.getString("Untitled") + " - " + APP_NAME);
        m_editor.scrollRectToVisible(new Rectangle(0, 0, 1, 1));
        m_chooser.setFileFilter(utf8Filter);
        updateSave(false);
        m_undo.discardAllEdits();
        updateUndoRedo();
        updateCutCopyDelete(false);
    }


    /**
     *  Opens a document
     *
     *@param  file  A file to be opened, null for Open dialog
     */
    public void openDocument(File file) {
        if (file == null) {
            if (m_chooser.showOpenDialog(VietPad.this) != JFileChooser.APPROVE_OPTION) {
                return;
            }

            File f = m_chooser.getSelectedFile();

            if ((f == null) || !f.isFile()) {
                return;
            }

            m_currentFile = f;
        } else if (!file.exists()) {
            JOptionPane.showMessageDialog(this, myResources.getString("Cannot_find_file_") + file, APP_NAME, JOptionPane.WARNING_MESSAGE);
            return;
        } else if (!file.isFile()) {
            JOptionPane.showMessageDialog(this, file + myResources.getString("_is_not_a_file"), APP_NAME, JOptionPane.WARNING_MESSAGE);
            return;
        } else {
            m_currentFile = file;
        }

        encoding = detectEncoding(m_currentFile);// get file encoding
        if (encoding.startsWith("UTF-16")) {
            m_chooser.setFileFilter(utf16Filter);
        } else if (encoding == "UTF8") {
            m_chooser.setFileFilter(utf8Filter);
        } else if (m_chooser.getFileFilter() == macFilter) {
            encoding = "MacRoman";
        } else if (m_chooser.getFileFilter() == winFilter) {
            encoding = "Cp1252";
        } else {
            encoding = System.getProperty("file.encoding");
            m_chooser.setFileFilter(MAC_OS_X ? macFilter : winFilter);
        }

        getGlassPane().setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
        getGlassPane().setVisible(true);

        try {
            BufferedReader in = new BufferedReader(new InputStreamReader(
                    new FileInputStream(m_currentFile), encoding));
            m_editor.read(in, null);
            in.close();
            m_doc = m_editor.getDocument();
            if (m_doc.getText(0, 1).equals("\uFEFF")) {
                m_doc.remove(0, 1); // remove BOM
            }
            m_doc.addUndoableEditListener(rawListener);
            setTitle(m_currentFile.getName() + " - " + APP_NAME);
        } catch (BadLocationException e) {
            // do nothing
            e.printStackTrace();
        } catch (OutOfMemoryError e) {
            e.printStackTrace();
            JOptionPane.showMessageDialog(this, APP_NAME
                     + myResources.getString("_has_run_out_of_memory.\nPlease_restart_") + APP_NAME
                     + myResources.getString("_and_try_again."), myResources.getString("Out_of_Memory"), JOptionPane.ERROR_MESSAGE);
        } catch (FileNotFoundException ex) {
            showError(ex, myResources.getString("Cannot_find_file_") + m_currentFile);
        } catch (Exception ex) {
            showError(ex, myResources.getString("Error_reading_file_") + m_currentFile);
        } finally {
            SwingUtilities.invokeLater(
                new Runnable()
                {
                    public void run() {
                        m_editor.setCaretPosition(0);
                        m_editor.scrollRectToVisible(new Rectangle(0, 0, 1, 1));
                        updateSave(false);
                        m_undo.discardAllEdits();
                        updateUndoRedo();
                        updateCutCopyDelete(false);
                        getGlassPane().setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
                        getGlassPane().setVisible(false);
                    }
                });
        }
    }


    /**
     *  Saves document to file
     *
     *@param  saveAs  True for saving to a new file
     *@return         True for success
     */
    protected boolean saveFile(boolean saveAs) {
        if (!saveAs && !m_textChanged) {
            return true;
        }

        if (saveAs || (m_currentFile == null)) {
            if (m_currentFile != null) {
                //  m_chooser.setCurrentDirectory(m_currentFile.getAbsoluteFile());
                m_chooser.setSelectedFile(m_currentFile);
            }

            if (m_chooser.showSaveDialog(VietPad.this) != JFileChooser.APPROVE_OPTION) {
                return false;
            }

            File f = m_chooser.getSelectedFile();

            if (f == null) {
                return false;
            }

            m_currentFile = f.getName().indexOf('.') > -1 ? f :
                    new File(m_chooser.getCurrentDirectory(), f.getName() + ".txt");
            setTitle(m_currentFile.getName() + " - " + APP_NAME);
        }

        if (m_chooser.getFileFilter() == utf8Filter) {
            encoding = "UTF8";
        } else if (m_chooser.getFileFilter() == utf16Filter) {
            encoding = "UTF-16";
        } else if (m_chooser.getFileFilter() == macFilter) {
            encoding = "MacRoman";
        } else {
            encoding = "Cp1252";
        }

        if (isUnicode(m_editor.getText()) && !encoding.startsWith("UTF")) {
            if (JOptionPane.CANCEL_OPTION == JOptionPane.showConfirmDialog(
                    VietPad.this,
                    myResources.getString("The_file_appears_to_contain_Unicode_characters.")
                     + myResources.getString("\nSaving_as_Windows_Latin_1_will_result_in_loss_of_those_characters.")
                     + myResources.getString("\nDo_you_still_want_to_proceed?")
                     + myResources.getString("\n\nTo_preserve_all_characters,_cancel_now")
                     + myResources.getString("\nand_save_as_a_new_file_in_Unicode_UTF-8_or_UTF-16_format.")
                    , myResources.getString("Save_as_Windows_Latin_1"), JOptionPane.OK_CANCEL_OPTION,
                    JOptionPane.WARNING_MESSAGE)) {
                return false;
            }
        }

        getGlassPane().setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
        getGlassPane().setVisible(true);
        if (defaultEOLOn) {
            m_doc.putProperty(DefaultEditorKit.EndOfLineStringProperty, System.getProperty("line.separator"));
        }
        try {
            BufferedWriter out = new BufferedWriter(new OutputStreamWriter(
                    new FileOutputStream(m_currentFile), encoding));
            m_editor.write(out);
            out.close();
            updateSave(false);
        } catch (OutOfMemoryError e) {
            e.printStackTrace();
            JOptionPane.showMessageDialog(this, APP_NAME
                     + myResources.getString("_has_run_out_of_memory.\nPlease_restart_") + APP_NAME
                     + myResources.getString("_and_try_again."), myResources.getString("Out_of_Memory"), JOptionPane.ERROR_MESSAGE);
        } catch (FileNotFoundException fnfe) {
            showError(fnfe, myResources.getString("Error_saving_file_") + m_currentFile + myResources.getString(".\nFile_is_inaccessible."));
        } catch (Exception ex) {
            showError(ex, myResources.getString("Error_saving_file_") + m_currentFile);
        } finally {
            SwingUtilities.invokeLater(
                new Runnable()
                {
                    public void run() {
                        getGlassPane().setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
                        getGlassPane().setVisible(false);
//                      updateSave(false);
                    }
                });
        }

        return true;
    }


    /**
     *  Determines whether a string contains Unicode characters
     *
     *@param  str  The string to investigate
     *@return      true if Unicode characters detected
     */
    protected boolean isUnicode(String str) {
        return !str.matches("[\u0000-\u00FF\u2019]*");
        // ISO Latin 1 plus right single quotation mark
    }


    /**
     *  Auto-detects file encoding.
     *
     *  Specifically tailored for detecting Vietnamese Unicode characters. Detect file encodings:
     *  Windows Latin 1, UTF-8, -16, -16BE, -16LE. Vietnamese legacy encodings are returned
     *  as Windows Latin 1.
     *
     *@param  file  Description of the Parameter
     *@return       the encoding detected
     */
    protected String detectEncoding(File file) {

        String encoding = "UTF8"; // UTF-8 as default encoding
        long filesize = file.length();

        if (filesize == 0) {
            return encoding;
        }
        if (filesize > 1024) {
            filesize = 1024;
            // read in 1024 bytes, increase if necessary
        }
        int[] b = new int[(int) filesize];

        try {
            DataInputStream in = new DataInputStream(new FileInputStream(file));
            for (int i = 0; i < filesize; i++) {
                b[i] = in.readUnsignedByte();
            }
            in.close();

            if (b[0] == 0xFE && b[1] == 0xFF) {
                // UTF-16 and -16BE BOM FEFF
                encoding = "UTF-16";
            } else if (b[0] == 0xFF && b[1] == 0xFE) {
                // UTF-16LE BOM FFFE
                encoding = "UTF-16LE";
            } else if (b[0] == 0xEF && b[1] == 0xBB && b[2] == 0xBF) {
                encoding = "UTF8";
            } else {
                for (int i = 0; i < b.length; i++) {
                    if ((b[i] & 0xF0) == 0xC0) {
                        if ((b[i + 1] & 0x80) == 0x80) {
                            encoding = "UTF8";
                            break;
                        }
                    } else if ((b[i] & 0xE1) == 0xE1) {
                        if ((b[i + 1] & 0x80) == 0x80 && (b[i + 2] & 0x80) == 0x80) {
                            encoding = "UTF8";
                            break;
                        }
                    } else if ((b[i] & 0x80) != 0x00) {
                        encoding = "Cp1252";
                    }
                }
            }
        } catch (IOException e) {
            System.err.println(myResources.getString("Can't_open_file."));
        } catch (Exception e) {
            System.err.println(e.getMessage());
        }

        return encoding;
    }


    /**
     *  Display a dialog to save changes
     *
     *@return    false if user canceled, true else
     */
    protected boolean promptToSave() {
        if (!m_textChanged) {
            return true;
        }
        switch (JOptionPane.showConfirmDialog(this,
                myResources.getString("Do_you_want_to_save_the_changes_to")
                 + "\n\"" + (m_currentFile == null ? myResources.getString("Untitled") : m_currentFile.getName()) + "\"?",
                APP_NAME, JOptionPane.YES_NO_CANCEL_OPTION, JOptionPane.INFORMATION_MESSAGE)) {
                        case JOptionPane.YES_OPTION:
                            return saveFile(false);
                        case JOptionPane.NO_OPTION:
                            return true;
                        default:
                            return false;
        }
    }


    /**
     *  Shows a warning message
     *
     *@param  e        the exception to warn about
     *@param  message  the message to display
     */
    public void showError(Exception e, String message) {
        e.printStackTrace();
        JOptionPane.showMessageDialog(this, message, APP_NAME, JOptionPane.WARNING_MESSAGE);
    }


    /**
     *  Updates the Save action
     *
     *@param  modified  whether file has been modified
     */
    private void updateSave(boolean modified) {
        if (m_textChanged != modified) {
            m_textChanged = modified;
            actionSave.setEnabled(modified);
            rootPane.putClientProperty("windowModified", Boolean.valueOf(modified));
            // see http://developer.apple.com/qa/qa2001/qa1146.html
        }
    }


    /**
     *  Updates the Undo and Redo actions
     */
    private void updateUndoRedo() {
        m_undoAction.setEnabled(m_undo.canUndo());
        m_redoAction.setEnabled(m_undo.canRedo());
    }


    /**
     *  Updates the Cut, Copy, and Delete actions
     *
     *@param  isTextSelected  whether any text currently selected
     */
    private void updateCutCopyDelete(boolean isTextSelected) {
        actionCut.setEnabled(isTextSelected);
        actionCopy.setEnabled(isTextSelected);
        actionDelete.setEnabled(isTextSelected);
    }


    /**
     *  Updates the Paste action
     */
    private void updatePaste() {
        Transferable clipData = clipboard.getContents(clipboard);
        if (clipData != null) {
            actionPaste.setEnabled(clipData.isDataFlavorSupported(DataFlavor.stringFlavor));
        }
    }


    /**
     *  Updates UI component if changes in LAF
     *
     *@param  laf  the look and feel class name
     */
    protected void updateLaF(String laf) {
        try {
            UIManager.setLookAndFeel(laf);
        } catch (Exception exc) {
            // do nothing
            exc.printStackTrace();
        }
        SwingUtilities.updateComponentTreeUI(this);
        SwingUtilities.updateComponentTreeUI(popup);
        SwingUtilities.updateComponentTreeUI(m_chooser);

        if (m_findDialog != null) {
            SwingUtilities.updateComponentTreeUI(m_findDialog);
        }
    }


    /**
     *  Displays Page Setup dialog
     */
    protected void pageSetup() {
        // to be implemented in subclass
    }


    /**
     *  Displays Print dialog
     */
    protected void printPage() {
        // to be implemented in subclass
    }


    /**
     *  Displays the About box
     */
    protected void about() {
        JOptionPane.showMessageDialog(this,
                APP_NAME + ", v1.2.1 \u00a9 2002\n" +
                myResources.getString("Vietnamese_Unicode_Text_Editor") +
//                myResources.getString("Pali") +
                DateFormat.getDateInstance(DateFormat.LONG)
                        .format(new GregorianCalendar(2005, Calendar.MAY, 30).getTime()) +
                "\nhttp://vietpad.sourceforge.net",
                myResources.getString("About_") + APP_NAME,
                JOptionPane.INFORMATION_MESSAGE);
    }


    /**
     *  Quits the application
     */
    protected void quit() {
        if (promptToSave()) {
            prefs.put("currentDirectory", m_chooser.getCurrentDirectory().getPath());
            prefs.put("fontName", m_font.getName());
            prefs.put("inputMethod", selectedInputMethod);
            prefs.put("lookAndFeel", UIManager.getLookAndFeel().getClass().getName());
            prefs.put("inputMethodLocale", getInputContext().getLocale().toString());
            prefs.putBoolean("smartMark", smartMarkOn);
            prefs.putBoolean("toolBar", toolBarOn);
            prefs.putBoolean("vietMode", vietModeOn);
            prefs.putBoolean("defaultEOL", defaultEOLOn);
            prefs.putInt("fontSize", m_font.getSize());
            prefs.putInt("fontStyle", m_font.getStyle());
            prefs.putInt("windowState", getExtendedState());

            if (getExtendedState() == NORMAL) {
                prefs.putInt("frameHeight", getHeight());
                prefs.putInt("frameWidth", getWidth());
                prefs.putInt("frameX", getX());
                prefs.putInt("frameY", getY());
            }

            javax.swing.filechooser.FileFilter[] filters = m_chooser.getChoosableFileFilters();
            for (int i = 0; i < filters.length; i++) {
                if (filters[i] == m_chooser.getFileFilter()) {
                    prefs.putInt("filterIndex", i);
                    break;
                }
            }
 
            System.exit(0);
        }
    }


    /**
     *  Listens to raw undoable edits
     *
     *@author     Gero Herrmann
     *@created    October 26, 2003
     */
    private class RawListener implements UndoableEditListener
    {
        /**
         *  Description of the Method
         *
         *@param  e  Description of the Parameter
         */
        public void undoableEditHappened(UndoableEditEvent e) {
            undoSupport.postEdit(e.getEdit());
        }
    }


    /**
     *  Listens to undoable edits filtered by undoSupport
     *
     *@author     Gero Herrmann
     *@created    October 26, 2003
     */
    private class SupportListener implements UndoableEditListener
    {
        /**
         *  Description of the Method
         *
         *@param  e  Description of the Parameter
         */
        public void undoableEditHappened(UndoableEditEvent e) {
            updateSave(true);
            m_undo.addEdit(e.getEdit());
            updateUndoRedo();
        }
    }


    /**
     *  JTextArea with Anti-Aliasing for font size greater than 18. AA in Java can
     *  be problematic, however, as the caret and text selection appear not to be
     *  at the same exact position.
     *
     *@author     Quan Nguyen
     *@created    October 26, 2003
     */
    private class JTextAreaAA extends JTextArea
    {
        protected void paintComponent(Graphics g) {
            Graphics2D g2d = (Graphics2D) g;

            // Anti-alias for font size > 18
            if (getFont().getSize() > 18) {
                g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
                g2d.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
                g2d.setRenderingHint(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_ON);
            }
            super.paintComponent(g2d);
        }
    }


    private int snap(final int ideal, final int min, final int max) {
        final int TOLERANCE = 0;
        return ideal < min + TOLERANCE ? min : (ideal > max - TOLERANCE ? max : ideal);
    }

}
