package net.sf.vietpad;

import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.print.*;
import java.util.StringTokenizer;
import java.util.Vector;
import javax.swing.RepaintManager;

/**
 * VietPad extension implementing printing functions
 *
 * @author   Quan Nguyen
 * @version  1.0.9, 7 April 2004
 */
public class VietPadWithPrinting extends VietPad implements Printable {

    private Vector m_lines;
    private String jreVersion = System.getProperty("java.version");
    
    // Printer setup
    private final PrinterJob printerJob = PrinterJob.getPrinterJob();
    private PageFormat pageFormat = printerJob.defaultPage();
    
    public int print(Graphics g, PageFormat pageFormat, int pageIndex)
        throws PrinterException {
        Graphics2D g2d = (Graphics2D) g;
        g2d.translate(pageFormat.getImageableX(), pageFormat.getImageableY());

        double wPage = pageFormat.getImageableWidth();
        double hPage = pageFormat.getImageableHeight();
        g2d.setClip(0, 0, (int) wPage, (int) hPage);
        
        // Anti-alias for font size > 18
//        if (m_editor.getFont().getSize() > 18) {
//            g2d.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
//            g2d.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
//            g2d.setRenderingHint(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_ON);
//        }
        
//      g2d.setColor(m_editor.getBackground());
//      g2d.fillRect(0, 0, wPage, hPage);
//      g2d.setColor(m_editor.getForeground());
        g2d.setFont(m_editor.getFont());

        FontMetrics fm = g2d.getFontMetrics();
        int hLine = fm.getHeight();

        if (m_lines == null) {
            m_lines = getLines(fm, wPage);
        }

        int numLines = m_lines.size();
        int linesPerPage = (int) Math.max(hPage / hLine, 1);
        int numPages = (int) Math.ceil((double) numLines / (double) linesPerPage);

        if (pageIndex >= numPages) {
            m_lines = null;
            return NO_SUCH_PAGE;
        }

        int x = 0;
        int y = fm.getAscent();
        int lineIndex = linesPerPage * pageIndex;

        while ((lineIndex < m_lines.size()) && (y < hPage)) {
            String str = (String) m_lines.get(lineIndex);
            g2d.drawString(str, x, y);
            y += hLine;
            lineIndex++;
        }

        return PAGE_EXISTS;
    }


    private Vector getLines(FontMetrics fm, double wPage) {
        Vector v = new Vector();

        String text = m_editor.getText();
        String prevToken = "";
        StringTokenizer st = new StringTokenizer(text, "\n\r", true);
        int MAX_ADVANCE = fm.getMaxAdvance() - 4;
        
        while (st.hasMoreTokens()) {
            String line = st.nextToken();

            if (line.equals("\r")) {
                continue;
            }

            // StringTokenizer will ignore empty lines, 
            // so it's a bit tricky to get them...
            if (line.equals("\n") && prevToken.equals("\n")) {
                v.add("");
            }

            prevToken = line;

            if (line.equals("\n")) {
                continue;
            }

            StringTokenizer st2 = new StringTokenizer(line, " \t", true);
            String line2 = "";

            while (st2.hasMoreTokens()) {
                String token = st2.nextToken();

                if (token.equals("\t")) {
                    int numSpaces = TAB_SIZE - (line2.length() % TAB_SIZE);
                    token = "";

                    for (int k = 0; k < numSpaces; k++) {
                        token += " ";
                    }
                }
                               
                int lineLength = fm.stringWidth(line2 + token);
                
                // Adds MAX_ADVANCE term to compensate for text clipping at the right margin.
                // The problem is most conspicuous when selected fontsize is 11. 
                // The error may be due to inaccuracies in Java's FontMetrics.
                //
                // Discovered that it's indeed a bug (Bug Id 4352983). Fixed in Java 1.5.
                if (!(jreVersion.compareTo("1.5") < 0)) {
                    MAX_ADVANCE = 0; // clear the offset value if J2SE 1.5 or greater
                }
                
                if ((lineLength + MAX_ADVANCE >= wPage) && (line2.length() > 0)) {
                    v.add(line2);
                    line2 = token.trim();

                    continue;
                }

                line2 += token;
            }

            v.add(line2);
        }

        return v;
    }


    /**
     * implement Page Setup
     */
    protected void pageSetup() {
        pageFormat = printerJob.pageDialog(pageFormat);
    }

    
    /**
     * implement Print
     */
    protected void printPage() {
        if (printerJob.printDialog()) {
            RepaintManager.currentManager(this).setDoubleBufferingEnabled(false);

            Thread runner = new Thread() {
                    public void run() {
                        try {
                            printerJob.setPrintable(VietPadWithPrinting.this, pageFormat);
                            printerJob.print();
                        } catch (PrinterException exception) {
                            System.err.println("Printing error: " +
                                exception);
                        }
                    }
                };

            runner.start();
            RepaintManager.currentManager(this).setDoubleBufferingEnabled(true);
        }
    }

}
