package net.sf.vietpad;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.io.*;
import java.nio.channels.*;
import java.text.BreakIterator;
import java.util.*;
import javax.swing.*;
import javax.swing.text.BadLocationException;
import org.unicode.Normalizer;
import com.swabunga.spell.engine.*;
import com.swabunga.spell.swing.JTextComponentSpellChecker;

/**
 *  Implementation of conversion and sort functions
 *
 *@author     Quan Nguyen
 *@author     Gero Herrmann
 *@created    February 11, 2002
 *@version    1.6, 25 May 2005
 */
public class VietPadWithTools extends VietPadWithFormat
{
    private ConvertDialog convertDlg;
    private SortDialog sortDlg;
    private PreferencesDialog preferencesDlg;
    private boolean diacriticsPosClassicOn;
    private boolean repeatKeyConsumed;
    private Map map;
    private long mapLastModified = Long.MIN_VALUE;
    
    /**
     *  Creates a new instance of VietPadWithTools
     */
    public VietPadWithTools() {
        super();
        diacriticsPosClassicOn = prefs.getBoolean("diacriticsPosClassic", true);
        repeatKeyConsumed = prefs.getBoolean("repeatKeyConsumed", false);
        VietKeyListener.consumeRepeatKey(repeatKeyConsumed);

        // try-catch block to work around a JRE 1.4.2 bug that sometimes throws
        // NoSuchMethodError exception on certain Windows 2000 and Linux systems
        // It's been determined that JRE 1.4.2 cannot properly recognize certain package name
        try {
            VietKey.setDiacriticsPosClassic(diacriticsPosClassicOn);
        } catch (java.lang.NoSuchMethodError e) {
            e.printStackTrace();
        }

        menuBar.add(createToolsMenu(), menuBar.getMenuCount() - 1);

       

        setVisible(true);
    }

    /**
     *  Creates the Tools menu
     *
     *@return    The menu
     */
    private JMenu createToolsMenu() {
        JMenu mTools = new JMenu(myResources.getString("Tools"));
        mTools.setMnemonic('T');
        
        Action spellAction =
            new AbstractAction(myResources.getString("Spell_Check") + "...")
            {
                public void actionPerformed(ActionEvent e) {                                     
                    try {
                        File customDictFile = new File(supportDir, "custom.dic");
                        if (!customDictFile.exists()) {
                            customDictFile.createNewFile();
                        }

                        final Reader phonetReader = new InputStreamReader(ClassLoader.getSystemResourceAsStream("dict/phonet.vi"), "UTF-8");
                        final Reader mainDictReader = new InputStreamReader(ClassLoader.getSystemResourceAsStream("dict/vi_VN.dic"), "UTF-8");
                        final SpellDictionary dictionary = new SpellDictionaryHashMap(mainDictReader, phonetReader);
                        final SpellDictionary customDictionary = new SpellDictionaryHashMap(customDictFile, phonetReader);                        
                 
                        undoSupport.beginUpdate();

                        new Thread() {
                            public void run() {
                                try {
                                    JTextComponentSpellChecker sc = new JTextComponentSpellChecker(dictionary, customDictionary, myResources.getString("Spell_Check"));
                                    sc.spellCheck(m_editor);
                                } catch (Exception ex) {
                                    ex.printStackTrace();
                                }
                            }
                        }.start();

                        undoSupport.endUpdate();
                        phonetReader.close();
                        mainDictReader.close();
                    } catch (Exception ex) {
                        ex.printStackTrace();
                }
            }
        };

        JMenuItem item = mTools.add(spellAction);
//        item.setMnemonic('s');
        item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_F7, 0));

        mTools.addSeparator();

        Action convertAction =
            new AbstractAction(myResources.getString("Convert_to_Unicode") + "...")
            {
                public void actionPerformed(ActionEvent e) {
                    if (convertDlg == null) {
                        convertDlg = new ConvertDialog(VietPadWithTools.this, false);// non-modal
                        convertDlg.setSelectedIndex(prefs.getInt("convertIndex", 0));
                        convertDlg.setLocation(
                                prefs.getInt("convertX", convertDlg.getX()),
                                prefs.getInt("convertY", convertDlg.getY()));
                    }
                    if (m_currentFile != null) {
                        convertDlg.checkHTML(
                                m_currentFile.getName().toLowerCase().matches(".+\\.html?"));
                    }
                    if (m_editor.getSelectedText() == null) {
                        m_editor.selectAll();
                    }
                    convertDlg.setVisible(true);
                }
            };
        item = mTools.add(convertAction);
        item.setMnemonic('u');
        item.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_U, MENU_MASK));

        Action viqrAction =
            new AbstractAction(myResources.getString("Convert_to_VIQR"))
            {
                public void actionPerformed(ActionEvent e) {
                    if (m_editor.getSelectedText() == null) {
                        m_editor.selectAll();
                        if (m_editor.getSelectedText() == null) {
                            return;
                        }
                    }
                    if (!isUnicode(m_editor.getSelectedText())) {
                        if (JOptionPane.CANCEL_OPTION == JOptionPane.showConfirmDialog(
                                VietPadWithTools.this,
                                myResources.getString("The_text_appears_to_be_already_in_VIQR_format.\nDo_you_still_want_to_proceed?"),
                                myResources.getString("Convert_to_VIQR"),
                                JOptionPane.OK_CANCEL_OPTION,
                                JOptionPane.INFORMATION_MESSAGE)) {
                            return;
                        }
                    }
                    UnicodeConversion unicon = new UnicodeConversion("Unicode");
                    String result = unicon.convert(m_editor.getSelectedText(), false);

                    undoSupport.beginUpdate();
                    int start = m_editor.getSelectionStart();
                    m_editor.replaceSelection(result);
                    setSelection(start, start + result.length());
                    undoSupport.endUpdate();
                }
            };
        item = mTools.add(viqrAction);
        item.setMnemonic('q');

        mTools.addSeparator();

        Action stripAction =
            new AbstractAction(myResources.getString("Strip_Diacritics"))
            {
                public void actionPerformed(ActionEvent e) {
                    if (m_editor.getSelectedText() == null) {
                        m_editor.selectAll();
                        if (m_editor.getSelectedText() == null) {
                            return;
                        }
                    }
                    Normalizer decomposer = new Normalizer(Normalizer.D, false);
                    String result = decomposer.normalize(m_editor.getSelectedText());
                    result = result.replaceAll("\\p{InCombiningDiacriticalMarks}+",
                            "").replace('\u0111', 'd').replace('\u0110', 'D');

                    undoSupport.beginUpdate();
                    int start = m_editor.getSelectionStart();
                    m_editor.replaceSelection(result);
                    setSelection(start, start + result.length());
                    undoSupport.endUpdate();
                }
            };
        item = mTools.add(stripAction);
        item.setMnemonic('d');

        Action addAction =
            new AbstractAction(myResources.getString("Add_Diacritics"))
            {
                public void actionPerformed(ActionEvent event) {
                    if (m_editor.getSelectedText() == null) {
                        m_editor.selectAll();
                        if (m_editor.getSelectedText() == null) {
                            return;
                        }
                    }

                    getGlassPane().setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
                    getGlassPane().setVisible(true);
                    try {
                        String result = addDiacritics(m_editor.getSelectedText());
                        undoSupport.beginUpdate();
                        int start = m_editor.getSelectionStart();
                        m_editor.replaceSelection(result);
                        setSelection(start, start + result.length());
                        undoSupport.endUpdate();
                    } catch (Exception e) {
                        e.printStackTrace();
                    } finally {
                        SwingUtilities.invokeLater(
                            new Runnable()
                            {
                                public void run() {
                                    getGlassPane().setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
                                    getGlassPane().setVisible(false);
                                }
                            });
                    }
                }
            };
        item = mTools.add(addAction);
        item.setMnemonic('a');

        Action normalizeAction =
            new AbstractAction(myResources.getString("Normalize_Diacritics"))
            {
                public void actionPerformed(ActionEvent e) {
                    if (m_editor.getSelectedText() == null) {
                        m_editor.selectAll();
                        if (m_editor.getSelectedText() == null) {
                            return;
                        }
                    }
                    String result = normalizeDiacritics(m_editor.getSelectedText());
                    undoSupport.beginUpdate();
                    int start = m_editor.getSelectionStart();
                    m_editor.replaceSelection(result);
                    setSelection(start, start + result.length());
                    undoSupport.endUpdate();
                }
            };
        item = mTools.add(normalizeAction);
        item.setMnemonic('n');

        mTools.addSeparator();

        Action sortAction =
            new AbstractAction(myResources.getString("Sort_Lines") + "...")
            {
                public void actionPerformed(ActionEvent e) {
                    if (sortDlg == null) {
                        sortDlg = new SortDialog(VietPadWithTools.this, false);// non-modal
                        sortDlg.setReverse(prefs.getBoolean("sortReverse", false));
                        sortDlg.setLocation(
                                prefs.getInt("sortX", sortDlg.getX()),
                                prefs.getInt("sortY", sortDlg.getY()));
                    }
                    if (m_editor.getSelectedText() == null) {
                        m_editor.selectAll();
                    }
                    sortDlg.setVisible(true);
                }
            };
        item = mTools.add(sortAction);
        item.setMnemonic('s');

        if (!MAC_OS_X) {
            mTools.addSeparator();

            Action optionsAction =
                new AbstractAction(myResources.getString("Preferences"))
                {
                    public void actionPerformed(ActionEvent e) {
                        preferences();
                    }
                };
            item = mTools.add(optionsAction);
            item.setMnemonic('r');
        }

        return mTools;
    }


    /**
     *  Adds diacritics to unmarked Viet text
     *
     *@param  source  Plain text to be marked
     *@return         Text with diacritics added
     */
    public String addDiacritics(String source) {
        loadMap();

        StringBuffer strB = new StringBuffer(source.toLowerCase());

        // Break text into words
        BreakIterator boundary = BreakIterator.getWordInstance();
        boundary.setText(source);
        int length = source.length();
        int start = boundary.first();

        // try first to process three words; if no match, try two, then one
        for (int end = boundary.next(); end != BreakIterator.DONE; start = end, end = boundary.next()) {
            if (!Character.isLetter(strB.charAt(start))) {
                continue;
            }

            // read next 1 or 2 words, if possible
            for (int i = 0; i < 4; i++) {
                if (end < length) {
                    end = boundary.next();
                } else {
                    break;
                }

                char ch = strB.charAt(end - 1);
                if (!Character.isLetter(ch) && ch != ' ') {
                    break;
                }
            }

            String key = strB.substring(start, end);

            // trim Key off non-letter characters
            while (!Character.isLetter(key.charAt(key.length() - 1))) {
                end = boundary.previous();// step back last word

                if (start == end) {
                    break;// break out of while loop
                }
                key = strB.substring(start, end);
            }

            if (map.containsKey(key)) {
                // match 3-, 2-, or 1-word sequence
                strB.replace(start, end, (String) map.get(key));
            } else {
                if (end > start) {
                    end = boundary.previous();// step back last word
                }

                if (end > start) {
                    end = boundary.previous();
                }
                key = strB.substring(start, end);

                if (map.containsKey(key)) {
                    // match 2- or 1-word sequence
                    strB.replace(start, end, (String) map.get(key));
                } else {
                    if (end > start) {
                        end = boundary.previous();// step back last word
                    }

                    if (end > start) {
                        end = boundary.previous();
                    }
                    key = strB.substring(start, end);
                    if (map.containsKey(key)) {
                        // match 1-word sequence
                        strB.replace(start, end, (String) map.get(key));
                    }
                }
            }

            end = boundary.next();
        }

        // Capitalize letters that are capital in source
        for (int i = 0; i < length; i++) {
            if (Character.isUpperCase(source.charAt(i))) {
                strB.setCharAt(i, Character.toUpperCase(strB.charAt(i)));
            }
        }
        return normalizeDiacritics(strB.toString());
    }


    /**
     *  Reads Vietnamese wordlist for Add Diacritics function
     */
    void loadMap() {
        final String wordList = "vietwordlist.txt";
        try {
            File dataFile = new File(supportDir, wordList);
            if (!dataFile.exists()) {
                final ReadableByteChannel input =
                        Channels.newChannel(ClassLoader.getSystemResourceAsStream("dict/" + dataFile.getName()));
                final FileChannel output =
                        new FileOutputStream(dataFile).getChannel();
                output.transferFrom(input, 0, 1000000L);
                input.close();
                output.close();
            }
            long fileLastModified = dataFile.lastModified();
            if (map == null) {
                map = new HashMap();
            } else {
                if (fileLastModified <= mapLastModified) {
                    return;// no need to reload map
                }
                map.clear();
            }
            mapLastModified = fileLastModified;
            BufferedReader bs = new BufferedReader(new InputStreamReader(new FileInputStream(dataFile), "UTF-8"));
            String accented;
            String plain;
            Normalizer decomposer = new Normalizer(Normalizer.D, false);
            while ((accented = bs.readLine()) != null) {
                plain = decomposer.normalize(accented)
                        .replaceAll("\\p{InCombiningDiacriticalMarks}+", "")
                        .replace('\u0111', 'd')
                        .replace('\u0110', 'D');
                map.put(plain.toLowerCase(), accented);
            }
            bs.close();
        } catch (IOException e) {
            map = null;
            e.printStackTrace();
            JOptionPane.showMessageDialog(this,
                    myResources.getString("Cannot_find_\"") + wordList + myResources.getString("\"_in\n") + supportDir.toString(),
                    VietPad.APP_NAME,
                    JOptionPane.ERROR_MESSAGE);
        }
    }

    /**
     *  Normalizes diacritics
     *
     *@author     Gero Herrmann
     *@param  source  Text to be normalized diacritics (diacritics in improper positions)
     *@return         Text with diacritics normalized (diacritics shifted to proper positions)
     */
    String normalizeDiacritics(String source) {
        final String TONE = "([\u0300\u0309\u0303\u0301\u0323])";
        final boolean isJAVA_1_5 = System.getProperty("java.version").compareTo("1.5") >= 0;
        
        // Works around an obscure Normalization bug which 
        // erroneously converts D with stroke and d with stroke to D and d,
        // respectively, on certain Windows systems,
        // by substituting them with \00DO and \00F0, respectively,
        // prior to normalization and then reverting them in post-processing.
        // It seems that some other installed applications have interfered with Java.

        String result = new Normalizer(Normalizer.D, false).normalize(source.replace('\u0110', '\u00D0').replace('\u0111', '\u00F0'))
                .replaceAll("(?i)" + TONE + "([aeiouy\u0306\u0302\u031B]+)", "$2$1")  // quy tac 1+3+4
                .replaceAll("(?i)(?<=[\u0306\u0302\u031B])(.)" + TONE + (isJAVA_1_5 ? "\\b" : "\\B"), "$2$1")  // quy tac 2
                .replaceAll("(?i)(?<=[ae])([iouy])" + TONE, "$2$1")  // quy tac 5
                .replaceAll("(?i)(?<=[oy])([iuy])" + TONE, "$2$1")
                .replaceAll("(?i)(?<!q)(u)([aeiou])" + TONE, "$1$3$2")
                .replaceAll("(?i)(?<!g)(i)([aeiouy])" + TONE, "$1$3$2");
        if (diacriticsPosClassicOn) {
            result = result.replaceAll("(?i)(?<!q)([ou])([aeoy])" + TONE + "(?!\\w)", "$1$3$2");
        }
        return new Normalizer(Normalizer.C, false).normalize(result).replace('\u00D0', '\u0110').replace('\u00F0', '\u0111');
    }


    /**
     *  Converts to Unicode
     *
     *@param  encoding  Source encoding
     *@param  html      True for HTML source
     */
    protected void convert(final VietEncodings sourceEncoding, final boolean html) {
        if (m_editor.getSelectedText() == null) {
            m_editor.selectAll();
            return;
        }

        if (sourceEncoding == VietEncodings.VIQR && isUnicode(m_editor.getSelectedText())) {
            if (JOptionPane.CANCEL_OPTION == JOptionPane.showConfirmDialog(
                    VietPadWithTools.this,
                    myResources.getString("The_text_appears_to_be_already_in_Unicode_format.\nDo_you_still_want_to_proceed?"),
                    myResources.getString("Convert_to_Unicode"),
                    JOptionPane.OK_CANCEL_OPTION,
                    JOptionPane.INFORMATION_MESSAGE)) {
                return;
            }
        }

        UnicodeConversion unicon = new UnicodeConversion(sourceEncoding);
        String result = unicon.convert(m_editor.getSelectedText(), html);

        undoSupport.beginUpdate();
        int start = m_editor.getSelectionStart();
        m_editor.replaceSelection(result);
        setSelection(start, start + result.length());
        undoSupport.endUpdate();
    }


    /**
     *  Sorts lines
     *
     *@param  reverse    True for reverse sorting
     *@param  delimiter  Delimiter for Left-to-Right sorting; empty if not needed
     */
    protected void sort(final boolean reverse, final String delimiter) {
        if (m_editor.getSelectedText() == null) {
            m_editor.selectAll();
            return;
        }

        // locate the beginning of a paragraph
        int start = m_editor.getSelectionStart();
        if (start != 0 && m_editor.getText().charAt(start - 1) != '\n') {
            try {
                int lineStart = m_editor.getLineStartOffset(m_editor.getLineOfOffset(start));
                start = lineStart;
                m_editor.setSelectionStart(start);
            } catch (BadLocationException e) {
                System.err.println(e);
            }
        }

        // locate the end of a paragraph
        int end = m_editor.getSelectionEnd();
        if (end != m_editor.getDocument().getLength() && m_editor.getText().charAt(end) != '\n') {
            try {
                int lineEnd = m_editor.getLineEndOffset((m_editor.getLineOfOffset(end)));
                if (m_editor.getDocument().getLength() == lineEnd) {
                    end = lineEnd;// at EOF, no '\n'
                } else {
                    end = lineEnd - 1;// exclude '\n'
                }

                m_editor.setSelectionEnd(end);
            } catch (BadLocationException e) {
                System.err.println(e);
            }
        }

        String[] words = m_editor.getSelectedText().split("\n");

        VietComparator vietComparator = new VietComparator();
        StringBuffer result = new StringBuffer();

        // for Left-to-Right sorting
        if (!delimiter.equals("")) {
            for (int i = 0; i < words.length; i++) {
                result.setLength(0);
                String[] entries = words[i].split("\\Q" + delimiter);
                java.util.Arrays.sort(entries, vietComparator);
                for (int j = 0; j < entries.length; j++) {
                    result.append(entries[j]).append(delimiter);
                }
                result.setLength(result.length() - delimiter.length());
                words[i] = result.toString();
            }
        }
        result.setLength(0);

        // sort lines (paragraphs)
        java.util.Arrays.sort(words, vietComparator);

        if (reverse) {
            for (int i = words.length - 1; i >= 0; i--) {
                result.append(words[i]).append("\n");
            }
        } else {
            for (int i = 0; i < words.length; i++) {
                result.append(words[i]).append("\n");
            }
        }
        result.setLength(result.length() - 1);// delete the last '\n'

        undoSupport.beginUpdate();
        m_editor.replaceSelection(result.toString());
        setSelection(start, start + result.length());
        undoSupport.endUpdate();
    }


    /**
     *  Retrieves options values from Preferences Dialog
     */
    protected void retrieveOptions() {
        defaultEOLOn = preferencesDlg.isDefaultEOL();
        diacriticsPosClassicOn = preferencesDlg.isDiacriticsPosClassic();
        repeatKeyConsumed = preferencesDlg.isRepeatKeyConsumed();
        localeVietOn = preferencesDlg.isLocaleVN();

        // try-catch block to work around a JRE 1.4.2 bug that sometimes throws
        // NoSuchMethodError exception on certain Windows 2000 and Linux systems
        // It's been determined that JRE 1.4.2 cannot properly recognize certain package name
        try {
            VietKey.setDiacriticsPosClassic(diacriticsPosClassicOn);
        } catch (java.lang.NoSuchMethodError e) {
            e.printStackTrace();
        }
        VietKeyListener.consumeRepeatKey(repeatKeyConsumed);
    }


    /**
     *  Updates UI component if changes in LAF
     *
     *@param  laf  The look and feel class name
     */
    protected void updateLaF(String laf) {
        super.updateLaF(laf);

        if (convertDlg != null) {
            SwingUtilities.updateComponentTreeUI(convertDlg);
            convertDlg.pack();
        }
        if (sortDlg != null) {
            SwingUtilities.updateComponentTreeUI(sortDlg);
            sortDlg.pack();
        }
        if (preferencesDlg != null) {
            SwingUtilities.updateComponentTreeUI(preferencesDlg);
            preferencesDlg.pack();
        }
    }


    /**
     *  Preferences (Options) settings
     */
    public void preferences() {
        if (preferencesDlg == null) {
            preferencesDlg = new PreferencesDialog(VietPadWithTools.this, true);// modal

            preferencesDlg.setDiacriticsPosition(diacriticsPosClassicOn);
            preferencesDlg.setRepeatKeyConsumed(repeatKeyConsumed);
            preferencesDlg.setDefaultEOL(defaultEOLOn);
            preferencesDlg.setLocaleVN(localeVietOn);

            preferencesDlg.setLocation(
                    prefs.getInt("optionsX", preferencesDlg.getX()),
                    prefs.getInt("optionsY", preferencesDlg.getY()));
        }
        preferencesDlg.setVisible(true);
    }


    /**
     *  Remembers settings and dialog locations, then quits
     */
    protected void quit() {
        if (convertDlg != null) {
            prefs.putInt("convertIndex", convertDlg.getSelectedIndex());
            prefs.putInt("convertX", convertDlg.getX());
            prefs.putInt("convertY", convertDlg.getY());
        }
        if (sortDlg != null) {
            prefs.putBoolean("sortReverse", sortDlg.isReverse());
            prefs.putInt("sortX", sortDlg.getX());
            prefs.putInt("sortY", sortDlg.getY());
        }
        if (preferencesDlg != null) {
            prefs.putBoolean("diacriticsPosClassic", diacriticsPosClassicOn);
            prefs.putBoolean("repeatKeyConsumed", repeatKeyConsumed);
            prefs.putBoolean("localeVN", localeVietOn);
            prefs.putInt("optionsX", preferencesDlg.getX());
            prefs.putInt("optionsY", preferencesDlg.getY());
        }
        super.quit();
    }


    /**
     *  Starts VietPad when called from a Mac OS X application bundle
     *
     *@param  args  The command line arguments
     */
    public static void main(String[] args) {
        new VietPadWithTools();
    }

}
