/*
 *  This file was modified for VietPad from the original supplied by Unicode, Inc.,
 *  adding a normalize method accepting a char parameter and returning a char
 *  and static compose and decompose methods matching com.ibm.icu.text.Normalizer.
 */
package org.unicode;

/**
 *  Implements Unicode Normalization Forms C, D, KC, KD.<br>
 *  See UTR#15 for details.<br>
 *  Copyright (c) 1998-1999 Unicode, Inc. All Rights Reserved.<br>
 *  The Unicode Consortium makes no expressed or implied warranty of any kind,
 *  and assumes no liability for errors or omissions. No liability is assumed
 *  for incidental and consequential damages in connection with or arising out
 *  of the use of the information here.
 *
 *@author     Mark Davis
 *@created    October 31, 2003
 */

public class Normalizer
{
    final static String copyright = "Copyright (c) 1998-1999 Unicode, Inc.";


    /**
     *  Create a normalizer for a given form.
     *
     *@param  form      Description of the Parameter
     *@param  fullData  Description of the Parameter
     */
    public Normalizer(byte form, boolean fullData) {
        this.form = form;
        if (data == null) {
            data = NormalizerBuilder.build(fullData);
        }// load 1st time
    }


    /**
     *  Masks for the form selector
     */
    final static byte
            COMPATIBILITY_MASK = 1,
            COMPOSITION_MASK = 2;

    /**
     *  Normalization Form Selector
     */
    public final static byte
            D = 0,
            C = COMPOSITION_MASK,
            KD = COMPATIBILITY_MASK,
            KC = (byte) (COMPATIBILITY_MASK + COMPOSITION_MASK);


    /**
     *  Normalizes text according to the chosen form, replacing contents of the
     *  target buffer.
     *
     *@param  source  the original text, unnormalized
     *@param  target  the resulting normalized text
     *@return         Description of the Return Value
     */
    public StringBuffer normalize(String source, StringBuffer target) {

        // First decompose the source into target,
        // then compose if the form requires.

        if (source.length() != 0) {
            internalDecompose(source, target);
            if ((form & COMPOSITION_MASK) != 0) {
                internalCompose(target);
            }
        }
        return target;
    }


    /**
     *  Normalizes text according to the chosen form
     *
     *@param  source  the original text, unnormalized
     *@return         target the resulting normalized text
     */
    public String normalize(String source) {
        return normalize(source, new StringBuffer()).toString();
    }


    /**
     *  Normalizes text according to the chosen form
     *
     *@param  source  the original char, unnormalized
     *@return         target the resulting normalized char
     */
    public char normalize(char source) {
        return normalize(String.valueOf(source)).charAt(0);
    }


    /**
     *  Composes a string. The string will be composed to according to the
     *  specified mode.
     *
     *@param  str     The string to compose.
     *@param  compat  If true the string will be composed accoding to NFKC rules
     *      and if false will be composed according to NFC rules.
     *@return         String The composed string
     *@draft          ICU 2.6
     */
    public static String compose(String str, boolean compat) {

        if (normalizer == null) {
            normalizer = new Normalizer(compat ? KC : C, false);
        } else {
            normalizer.form = compat ? KC : C;
        }
        return normalizer.normalize(str);
    }


    /**
     *  Decomposes a string. The string will be decomposed to according to the
     *  specified mode.
     *
     *@param  str     The string to decompose.
     *@param  compat  If true the string will be decomposed accoding to NFKD rules
     *      and if false will be decomposed according to NFD rules.
     *@return         String The decomposed string
     *@draft          ICU 2.6
     */
    public static String decompose(String str, boolean compat) {

        if (normalizer == null) {
            normalizer = new Normalizer(compat ? KD : D, false);
        } else {
            normalizer.form = compat ? KD : D;
        }
        return normalizer.normalize(str);
    }


    // ======================================
    //                  PRIVATES
    // ======================================


    static private Normalizer normalizer;
    
    /**
     *  The current form.
     */
    private byte form;


    /**
     *  Decomposes text, either canonical or compatibility, replacing contents of
     *  the target buffer.
     *
     *@param  source  the original text, unnormalized
     *@param  target  the resulting normalized text
     */
    private void internalDecompose(String source, StringBuffer target) {
        StringBuffer buffer = new StringBuffer();
        boolean canonical = (form & COMPATIBILITY_MASK) == 0;
        for (int i = 0; i < source.length(); ++i) {
            buffer.setLength(0);
            data.getRecursiveDecomposition(canonical, source.charAt(i), buffer);

            // add all of the characters in the decomposition.
            // (may be just the original character, if there was
            // no decomposition mapping)

            for (int j = 0; j < buffer.length(); ++j) {
                char ch = buffer.charAt(j);
                int chClass = data.getCanonicalClass(ch);
                int k = target.length();// insertion point
                if (chClass != 0) {

                    // bubble-sort combining marks as necessary

                    for (; k > 0; --k) {
                        if (data.getCanonicalClass(target.charAt(k - 1)) <= chClass) {
                            break;
                        }
                    }
                }
                target.insert(k, ch);
            }
        }
    }


    /**
     *  Composes text in place. Target must already have been decomposed.
     *
     *@param  target  input: decomposed text. output: the resulting normalized
     *      text.
     */
    private void internalCompose(StringBuffer target) {

        int starterPos = 0;

        int compPos = 1;
        char starterCh = target.charAt(0);
        int lastClass = data.getCanonicalClass(starterCh);
        if (lastClass != 0) {
            lastClass = 256;
        }// fix for irregular combining sequence

        // Loop on the decomposed characters, combining where possible

        for (int decompPos = 1; decompPos < target.length(); ++decompPos) {
            char ch = target.charAt(decompPos);
            int chClass = data.getCanonicalClass(ch);
            char composite = data.getPairwiseComposition(starterCh, ch);
            if (composite != NormalizerData.NOT_COMPOSITE
                     && (lastClass < chClass || lastClass == 0)) {
                target.setCharAt(starterPos, composite);
                starterCh = composite;
            } else {
                if (chClass == 0) {
                    starterPos = compPos;
                    starterCh = ch;
                }
                lastClass = chClass;
                target.setCharAt(compPos++, ch);
            }
        }
        target.setLength(compPos);
    }


    /**
     *  Contains normalization data from the Unicode Character Database. use false
     *  for the minimal set, true for the real set.
     */
    private static NormalizerData data = null;


    /**
     *  Just accessible for testing.
     *
     *@param  ch  Description of the Parameter
     *@return     The excluded value
     */
    boolean getExcluded(char ch) {
        return data.getExcluded(ch);
    }


    /**
     *  Just accessible for testing.
     *
     *@param  ch  Description of the Parameter
     *@return     The rawDecompositionMapping value
     */
    String getRawDecompositionMapping(char ch) {
        return data.getRawDecompositionMapping(ch);
    }
}
