/*
 * Licensed to Elasticsearch under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.elasticsearch.gradle.doc

import org.elasticsearch.gradle.doc.SnippetsTask.Snippet
import org.gradle.api.InvalidUserDataException
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.OutputDirectory

import java.nio.file.Files
import java.nio.file.Path
import java.util.regex.Matcher

/**
 * Generates REST tests for each snippet marked // TEST.
 */
public class RestTestsFromSnippetsTask extends SnippetsTask {
    @Input
    Map<String, String> setups = new HashMap()

    /**
     * Root directory of the tests being generated. To make rest tests happy
     * we generate them in a testRoot() which is contained in this directory.
     */
    @OutputDirectory
    File testRoot = project.file('build/rest')

    public RestTestsFromSnippetsTask() {
        project.afterEvaluate {
            // Wait to set this so testRoot can be customized
            project.sourceSets.test.output.dir(testRoot, builtBy: this)
        }
        TestBuilder builder = new TestBuilder()
        doFirst { outputRoot().delete() }
        perSnippet builder.&handleSnippet
        doLast builder.&finishLastTest
    }

    /**
     * Root directory containing all the files generated by this task. It is
     * contained withing testRoot.
     */
    File outputRoot() {
        return new File(testRoot, '/rest-api-spec/test')
    }

    private class TestBuilder {
        private static final String SYNTAX = {
            String method = /(?<method>GET|PUT|POST|HEAD|OPTIONS|DELETE)/
            String pathAndQuery = /(?<pathAndQuery>[^\n]+)/
            String badBody = /GET|PUT|POST|HEAD|OPTIONS|DELETE|#/
            String body = /(?<body>(?:\n(?!$badBody)[^\n]+)+)/
            String nonComment = /$method\s+$pathAndQuery$body?/
            String comment = /(?<comment>#.+)/
            /(?:$comment|$nonComment)\n+/
        }()

        /**
         * The file in which we saw the last snippet that made a test.
         */
        Path lastDocsPath

        /**
         * The file we're building.
         */
        PrintWriter current

        /**
         * Called each time a snippet is encountered. Tracks the snippets and
         * calls buildTest to actually build the test.
         */
        void handleSnippet(Snippet snippet) {
            if (snippet.language == 'json') {
                throw new InvalidUserDataException(
                        "$snippet: Use `js` instead of `json`.")
            }
            if (snippet.testSetup) {
                setup(snippet)
                return
            }
            if (snippet.testResponse) {
                response(snippet)
                return
            }
            if (snippet.test || snippet.console) {
                test(snippet)
                return
            }
            // Must be an unmarked snippet....
        }

        private void test(Snippet test) {
            setupCurrent(test)

            if (false == test.continued) {
                current.println('---')
                current.println("\"$test.start\":")
            }
            if (test.skipTest) {
                current.println("  - skip:")
                current.println("      features: always_skip")
                current.println("      reason: $test.skipTest")
            }
            if (test.setup != null) {
                String setup = setups[test.setup]
                if (setup == null) {
                    throw new InvalidUserDataException("Couldn't find setup "
                        + "for $test")
                }
                current.println(setup)
            }

            body(test, false)
        }

        private void response(Snippet response) {
            current.println("  - response_body: |")
            response.contents.eachLine { current.println("      $it") }
        }

        void emitDo(String method, String pathAndQuery,
                String body, String catchPart, boolean inSetup) {
            def (String path, String query) = pathAndQuery.tokenize('?')
            current.println("  - do:")
            if (catchPart != null) {
                current.println("      catch: $catchPart")
            }
            current.println("      raw:")
            current.println("        method: $method")
            current.println("        path: \"$path\"")
            if (query != null) {
                for (String param: query.tokenize('&')) {
                    def (String name, String value) = param.tokenize('=')
                    if (value == null) {
                        value = ''
                    }
                    current.println("        $name: \"$value\"")
                }
            }
            if (body != null) {
                // Throw out the leading newline we get from parsing the body
                body = body.substring(1)
                current.println("        body: |")
                body.eachLine { current.println("          $it") }
            }
            /* Catch any shard failures. These only cause a non-200 response if
             * no shard succeeds. But we need to fail the tests on all of these
             * because they mean invalid syntax or broken queries or something
             * else that we don't want to teach people to do. The REST test
             * framework doesn't allow us to has assertions in the setup
             * section so we have to skip it there. We also have to skip _cat
             * actions because they don't return json so we can't is_false
             * them. That is ok because they don't have this
             * partial-success-is-success thing.
             */
            if (false == inSetup && false == path.startsWith('_cat')) {
                current.println("  - is_false: _shards.failures")
            }
        }

        private void setup(Snippet setup) {
            if (lastDocsPath == setup.path) {
                throw new InvalidUserDataException("$setup: wasn't first")
            }
            setupCurrent(setup)
            current.println('---')
            current.println("setup:")
            body(setup, true)
            // always wait for yellow before anything is executed
            current.println(
                    "  - do:\n" +
                    "      raw:\n" +
                    "        method: GET\n" +
                    "        path: \"_cluster/health\"\n" +
                    "        wait_for_status: \"yellow\"")
        }

        private void body(Snippet snippet, boolean inSetup) {
            parse("$snippet", snippet.contents, SYNTAX) { matcher, last ->
                if (matcher.group("comment") != null) {
                    // Comment
                    return
                }
                String method = matcher.group("method")
                String pathAndQuery = matcher.group("pathAndQuery")
                String body = matcher.group("body")
                String catchPart = last ? snippet.catchPart : null
                if (pathAndQuery.startsWith('/')) {
                    // Leading '/'s break the generated paths
                    pathAndQuery = pathAndQuery.substring(1)
                }
                emitDo(method, pathAndQuery, body, catchPart, inSetup)
            }
        }

        private PrintWriter setupCurrent(Snippet test) {
            if (lastDocsPath == test.path) {
                return
            }
            finishLastTest()
            lastDocsPath = test.path

            // Make the destination file:
            // Shift the path into the destination directory tree
            Path dest = outputRoot().toPath().resolve(test.path)
            // Replace the extension
            String fileName = dest.getName(dest.nameCount - 1)
            dest = dest.parent.resolve(fileName.replace('.asciidoc', '.yaml'))

            // Now setup the writer
            Files.createDirectories(dest.parent)
            current = dest.newPrintWriter('UTF-8')
        }

        void finishLastTest() {
            if (current != null) {
                current.close()
                current = null
            }
        }
    }
}
