/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Other licenses:
 * -----------------------------------------------------------------------------
 * Commercial licenses for this work are available. These replace the above
 * ASL 2.0 and offer limited warranties, support, maintenance, and commercial
 * database integrations.
 *
 * For more information, please visit: http://www.jooq.org/licenses
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */
package org.jooq.impl;

import java.sql.CallableStatement;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;

import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.Parser;
import org.jooq.tools.JooqLogger;
import org.jooq.tools.jdbc.DefaultConnection;

/**
 * @author Lukas Eder
 */
@SuppressWarnings("deprecation")
final class ParsingConnection extends DefaultConnection {

    private static final JooqLogger log = JooqLogger.getLogger(ParsingConnection.class);

    private final Configuration     configuration;
    private final DSLContext        ctx;
    private final Parser            parser;

    ParsingConnection(Configuration configuration) {
        super(configuration.connectionProvider().acquire());

        this.configuration = configuration;
        this.ctx = DSL.using(configuration);
        this.parser = ctx.parser();
    }

    final String translate(String sql) {
        log.debug("Translating from", sql);
        String result = ctx.render(parser.parseQuery(sql));
        log.debug("Translating to", result);
        return result;
    }

    @Override
    public final Statement createStatement() throws SQLException {
        return new ParsingStatement(this, getDelegate().createStatement());
    }

    @Override
    public final Statement createStatement(int resultSetType, int resultSetConcurrency) throws SQLException {
        return new ParsingStatement(this, getDelegate().createStatement(resultSetType, resultSetConcurrency));
    }

    @Override
    public final Statement createStatement(int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
        return new ParsingStatement(this, getDelegate().createStatement(resultSetType, resultSetConcurrency, resultSetHoldability));
    }

    @Override
    public final PreparedStatement prepareStatement(String sql) throws SQLException {
        return new ParsingStatement(this, getDelegate().prepareStatement(translate(sql)));
    }

    @Override
    public final PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
        return new ParsingStatement(this, getDelegate().prepareStatement(translate(sql), resultSetType, resultSetConcurrency));
    }

    @Override
    public final PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
        return new ParsingStatement(this, getDelegate().prepareStatement(translate(sql), resultSetType, resultSetConcurrency, resultSetHoldability));
    }

    @Override
    public final PreparedStatement prepareStatement(String sql, int autoGeneratedKeys) throws SQLException {
        return new ParsingStatement(this, getDelegate().prepareStatement(translate(sql), autoGeneratedKeys));
    }

    @Override
    public final PreparedStatement prepareStatement(String sql, int[] columnIndexes) throws SQLException {
        return new ParsingStatement(this, getDelegate().prepareStatement(translate(sql), columnIndexes));
    }

    @Override
    public final PreparedStatement prepareStatement(String sql, String[] columnNames) throws SQLException {
        return new ParsingStatement(this, getDelegate().prepareStatement(translate(sql), columnNames));
    }

    @Override
    public final CallableStatement prepareCall(String sql) throws SQLException {
        return new ParsingStatement(this, getDelegate().prepareCall(translate(sql)));
    }

    @Override
    public final CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
        return new ParsingStatement(this, getDelegate().prepareCall(sql, resultSetType, resultSetConcurrency));
    }

    @Override
    public final CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
        return new ParsingStatement(this, getDelegate().prepareCall(sql, resultSetType, resultSetConcurrency, resultSetHoldability));
    }

    @Override
    public final void close() throws SQLException {
        configuration.connectionProvider().release(getDelegate());
    }
}
