/**
 * The MIT License
 * Copyright (c) 2014 Ilkka Seppälä
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package com.iluwatar.poison.pill;

/**
 * One of the possible approaches to terminate Producer-Consumer pattern is using the Poison Pill
 * idiom. If you use Poison Pill as the termination signal then Producer is responsible to notify
 * Consumer that the exchange is over and reject any further messages. The Consumer receiving Poison
 * Pill will stop reading messages from the queue. You must also ensure that the Poison Pill will be
 * the last message that will be read from the queue (if you have prioritized queue then this can be
 * tricky).
 * <p>
 * In simple cases the Poison Pill can be just a null-reference, but holding a unique separate
 * shared object-marker (with name "Poison" or "Poison Pill") is more clear and self describing.
 * 
 */
public class App {

  /**
   * Program entry point
   * 
   * @param args command line args
   */
  @edu.cuny.hunter.streamrefactoring.annotations.EntryPoint public static void main(String[] args) {
    MessageQueue queue = new SimpleMessageQueue(10000);

    final Producer producer = new Producer("PRODUCER_1", queue);
    final Consumer consumer = new Consumer("CONSUMER_1", queue);

    new Thread() {
      @Override
      public void run() {
        consumer.consume();
      }
    }.start();

    new Thread() {
      @Override
      public void run() {
        producer.send("hand shake");
        producer.send("some very important information");
        producer.send("bye!");
        producer.stop();
      }
    }.start();
  }
}
