/*
 * Copyright 2009-2010 Markus Karg
 *
 * This file is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * For a copy of the GNU Lesser General Public License
 * see http://www.gnu.org/licenses/lgpl.html.
 */
package civvi.common;

import java.util.Collections;
import java.util.Enumeration;
import java.util.Iterator;

/**
 * Enumeration utility methods.
 * 
 * @author <a href="mailto:markus@headcrashing.eu">Markus Karg</a>
 * @see <a href="http://weblogs.java.net/blog/mkarg/archive/2010/07/04/using-enumerations-each-statements-five-times-faster-jre-without-ram-l">Using Enumerations in for-each statements</a>
 * @since v1.0.0 [4 Jul 2010]
 */
public class Enumerations {
	public Enumerations() {
		// prevent init!
	}

	/**
	 * Allows using of {@link Enumeration} with the for-each statement. The
	 * implementation is not using any heap space and such is able to serve
	 * virtually endless Enumerations, while
	 * {@link Collections#list(Enumeration)} is limited by available RAM. As a
	 * result, this implementation is much faster than
	 * {@link Collections#list(Enumeration)}.
	 * </p>
	 * <strong>Note:</strong> {@link Iterator#remove()} will throw a
	 * {@link UnsupportedOperationException} as there is no equivalent in
	 * {@link Enumeration}.
	 * 
	 * @param <E>
	 * @param enumeration the source enumeration.
	 * @return an {@link Iterable} directly calling the original Enumeration.
	 */
	public static final <E> Iterable<E> toIterable(
			final Enumeration<E> enumeration)
	{
		return new Iterable<E>() {
			public final Iterator<E> iterator() {
				return new EnumerationIterator<E>(enumeration);
			}
		};
	}


	// --- Inner Classes ---

	/**
	 * A {@link Iterator} implementation that takes a {@link Enumeration}.
	 * 
	 * @author <a href="mailto:markus@headcrashing.eu">Markus Karg</a>
	 * @since v1.0.0 [4 Jul 2010]
	 * @param <E>
	 */
	private static class EnumerationIterator<E> implements Iterator<E> {
		private final Enumeration<E> delegate;

		public EnumerationIterator(Enumeration<E> delegate) {
			this.delegate = delegate;
		}

		@Override
		public final boolean hasNext() {
			return this.delegate.hasMoreElements();
		}

		@Override
		public final E next() {
			return this.delegate.nextElement();
		}

		@Override
		public final void remove() {
			throw new UnsupportedOperationException(
					"Cannot remove from an Enumeration!");
		}
	}
}
