package civvi.common;

import java.lang.reflect.ParameterizedType;

/**
 * Utilities for handling Generics.
 * 
 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
 * @since v1.0.0 [10 Jul 2010]
 */
public class GenericsUtil {
	public GenericsUtil() {
		// prevent init!
	}

	/**
	 * Returns the parameter type from the given class. {@code i}
	 * references the index or the parameters in the order they're declared.
	 * For example, for a class that has two parameters:
	 * <pre>
	 * public static class MyMap implements List&lt;String, Double&gt; { ... }
	 * 
	 * assert String.class == GenericsUtil.getParamType(MyMap.class, 0); // assert successful
	 * final MyMap myMapInstance = new MyMap();
	 * assert Double.class == GenericsUtil.getParamType(myMapInstance.getClass(), 1); // assert successful
	 * </pre>
	 * <strong>Note:</strong> due to the nature of the generics implementation
	 * frustratingly the following will not work as it requires compiled
	 * generics information and throws away runtime structures:
	 * <pre>
	 * 	Map&lt;BigDecimal, Double&gt; myMap = new MyFooClass&lt;BigDecimal, Double&gt;();
	 * 	assert BigDecimal.class == GenericsUtil.getParamType(myMap.getClass(), 0); // assert fails
	 * 	assert Double.class == GenericsUtil.getParamType(myMap.getClass(), 1); // assert fails
	 * </pre>
	 * 
	 * @param cls the class to inspect.
	 * @param i the index of the parameter starting at zero.
	 * @return the parameter type.
	 */
	public static Class<?> getParamType(Class<?> cls, int i) {
		final ParameterizedType pType = (ParameterizedType) cls.getGenericSuperclass();
		return (Class<?>) pType.getActualTypeArguments()[i];
	}
}
