package civvi.common;  

import static javax.naming.directory.SearchControls.SUBTREE_SCOPE;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.naming.Context;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;

/**
 * A set of utilities for performing lookup of LDAP services.
 *   
 * @author <a href="mailto=dansiviter@gmail.com">Daniel Siviter</a>
 * @since v1.0.0 [23 May 2010]
 */
public class LdapUtil {  
	public static final String MEMBER_OF_ATTRIBUTE = "memberOf";

	private static final Logger log =
		Logger.getLogger(LdapUtil.class.getName());

	/**
	 * Retrieves the user given by the username and password. This will attempt
	 * to resolve the domain automatically using the locate network address.
	 *
	 * @param username the username of the user.
	 * @param password the password of the user.
	 * @return the found user.
	 * @throws UnknownHostException if no IP address for the local host could
	 * be found.
	 * @throws NamingException
	 * @see {@link #retrieveUser(String, String, String)}
	 */
	public static SearchResult retrieveUser(String username, String password)
	throws UnknownHostException, NamingException
	{
		final String domain = getDomain();
		final DirContext context = getLdapContext(domain, username, password);
		try {
			return retrieveUser(context, domain, username);
		} finally {
			if (context != null) { 
				context.close();
			}
		}
	}

	/**
	 * 
	 * TODO
	 * @return
	 * @throws UnknownHostException
	 */
	public static String getDomain() throws UnknownHostException {
		final InetAddress localaddr = InetAddress.getLocalHost();
		final String canocicalHost = localaddr.getCanonicalHostName();
		return canocicalHost.substring(canocicalHost.indexOf('.') + 1);
	}

	/**
	 * 
	 * TODO
	 * @param domain the domain for the user.
	 * @param username the username of the user.
	 * @param password the password of the user.
	 * @return
	 * @throws NamingException
	 */
	public static DirContext getLdapContext(
			String domain,
			String username,
			String password)
	throws NamingException
	{
		if (log.isLoggable(Level.INFO)) {
			log.info(String.format(
					"Attempting to resolve user. [domainName=%s,username=%s,password=***ommited***]",
					domain,
					username)); 
		}

		final Hashtable<String, Object> props = new Hashtable<String, Object>();
		props.put(Context.SECURITY_PRINCIPAL, toPrincipleName(username, domain));  
		props.put(Context.SECURITY_CREDENTIALS, password);
		props.put(Context.REFERRAL, "follow");
		props.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");

		try {
			props.put(Context.PROVIDER_URL, "ldap://" + obtainLDAPServer(domain) + '/');	
			return new InitialDirContext(props);
		} catch (NamingException e) {  
			log.log(Level.WARNING,"Failed to bind to LDAP",e);  
			throw new RuntimeException(String.format(
					"Either no such user or incorrect password! [user=%s,domain=%s,password=***ommited***]",
					username, domain),e);  
		}  
	}

	/**
	 * Retrieves the user given by the username and password. This will
	 * automatically resolve the LDAP server via DNS lookup.
	 *
	 * @param domain the domain for the user.
	 * @param username the username of the user.
	 * @return the found user.
	 */
	public static SearchResult retrieveUser(
			DirContext context,
			String domain,
			String username)
	throws NamingException
	{
		// locate this user's record  
		final SearchControls controls = new SearchControls();  
		controls.setSearchScope(SUBTREE_SCOPE);
		final String principalName = toPrincipleName(username, domain);
		NamingEnumeration<SearchResult> renum = context.search(
				toDC(domain),
				String.format("(& (userPrincipalName=%s)(objectClass=user))", principalName),
				controls);  
		if(!renum.hasMore()) {  
			// failed to find it. Fall back to sAMAccountName.  
			// see http://www.nabble.com/Re%3A-Hudson-AD-plug-in-td21428668.html  
			renum = context.search(toDC(domain),"(& (sAMAccountName=" + username + ")(objectClass=user))", controls);  
			if(!renum.hasMore())  
				throw new RuntimeException(String.format(
						"Unable to locate user information! [username=%s]",
						username));  
		}  
		final SearchResult result = renum.next();  

		final List<String> groups = new ArrayList<String>();  
		final Attribute memberOf = result.getAttributes().get(MEMBER_OF_ATTRIBUTE);  
		if (memberOf != null) {// null if this user belongs to no group at all  
			for (int i = 0; i < memberOf.size(); i++) {
				final String group = (String) memberOf.get(i);
				groups.add(group.substring(3, group.indexOf(',')));
//				Attributes atts = context.getAttributes("\"" + group.substring(3, group.indexOf(',')) + '"', new String[]{"CN"});  
//				Attribute att = atts.get("CN");
//				groups.add(att.get().toString());  
			}  
		}  
		return result;  
	}  

	/**
	 * 
	 * TODO
	 *
	 * @param domainName
	 * @return
	 */
	static String toDC(String domainName) {  
		final StringBuilder buf = new StringBuilder();  
		for (String token : domainName.split("\\.")) {  
			if (token.isEmpty()) { // defensive check
				continue;
			}
			if (buf.length() > 0) {
				buf.append(",");
			}
			buf.append("DC=").append(token);  
		}  
		return buf.toString();  
	}  

	/**
	 * 
	 * TODO
	 * @param username
	 * @param domain
	 * @return
	 */
	static String toPrincipleName(String username, String domain) {
		return username + '@' + domain;
	}

	/**
	 * Creates {@link DirContext} for accessing DNS.
	 *
	 * @return
	 * @throws NamingException
	 */
	public static DirContext createDNSLookupContext() throws NamingException {
		final Hashtable<String, Object> env = new Hashtable<String, Object>();
		env.put(
				Context.INITIAL_CONTEXT_FACTORY,
		"com.sun.jndi.dns.DnsContextFactory");
		return new InitialDirContext(env);
	}

	/**
	 * Obtains the LDAP server's host name by automatically creating DNS
	 * lookup.
	 *
	 * @param domainName
	 * @return
	 * @throws NamingException
	 */
	public static String obtainLDAPServer(String domainName)
	throws NamingException
	{
		return obtainLDAPServer(createDNSLookupContext(), domainName);
	}

	/**
	 * Use DNS and obtains the LDAP server's host name.
	 *
	 * @param ictx
	 * @param domainName
	 * @return
	 * @throws NamingException
	 */
	public static String obtainLDAPServer(DirContext ictx, String domainName)
	throws NamingException
	{
		final String ldapServer = "_ldap._tcp." + domainName;

		if (log.isLoggable(Level.INFO)) {
			log.info(String.format(
					"Attempting to resolve LDAP server host from SRV record. [%s]",
					ldapServer));
		}
		Attributes attrs = ictx.getAttributes(ldapServer, new String[]{ "SRV" });
		Attribute attr = attrs.get("SRV");
		if (attr == null) {
			throw new NamingException();
		}

		// get LDAP host with highest priority
		int priority = -1;
		String result = null;
		for (NamingEnumeration<?> ne = attr.getAll(); ne.hasMoreElements(); ) {
			String[] fields = ne.next().toString().split(" ");
			int p = Integer.parseInt(fields[0]);
			if (priority == -1 || p < priority) {
				priority = p;
				result = fields[3];
				// cut off trailing '.'
				//result = result.replace("\\.$","");
				if (result.endsWith(".")) {
					result = result.substring(0, result.length() - 1);
				}
			}
		}

		if (log.isLoggable(Level.INFO)) {
			log.info(String.format("Resolved LDAP server host. [%s]", result));
		}
		return result;
	}
}  
