package civvi.common;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.net.URL;
import java.util.logging.Filter;
import java.util.logging.Formatter;
import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.logging.Logger;

/**
 * A Class to configure the <code>LogManager</code> from a file on the
 * classpath.
 *
 * @author <a href="mailto=dansiviter@gmail.com">Daniel Siviter</a>
 * @since 13-Nov-2005
 */
public final class LogUtil {
	/** Default logging config. file. */
	public static final String DEFAULT_CONFIG = "logging.properties";

	private LogUtil() {
		// prevent construction!
	}
	
	/**
	 * Configures the <code>LogManager</code> from the file path given. The
	 * path is based on the classpath.
	 *
	 * @param path The path to the configuration file in the classpath.
	 */
	public static void configureFromFile(String path) {
		final Logger log = Logger.getLogger(LogUtil.class.getName());
		try {
			if (path == null) {
				path = DEFAULT_CONFIG;
			}
			
			// default level is INFO so should log first message
			log.config("Loading \"" + path + "\" from classpath.");
			URL url = Thread.currentThread().getContextClassLoader().getResource(path);
			log.config("URL: \"" + url + "\"");
			
			if (url == null) {
				log.warning("Could not find logging file: \"" + path + "\", using default!");
				return;
			}
			
			LogManager.getLogManager().readConfiguration(
					new BufferedInputStream(url.openStream()));
		} catch (IOException ioe) {
			log.log(
					Level.WARNING,
					"Could not configure logging from file \"" + path + "\", using default!",
					ioe);
		}
	}
	
	/**
	 * Package private method to get a String property.
	 * If the property is not defined we return the given
	 * default value.
	 * 
	 * @param name the name of the property.
	 * @param defaultValue the default value of the property.
	 * @return the 
	 */
	public static String getStringProperty(String name, String defaultValue) {
		String val = LogManager.getLogManager().getProperty(name);
		if (val == null)
			return defaultValue;
		return val.trim();
	}

	/**
	 * Static method to get a Level property.
	 * If the property is not defined or cannot be parsed
	 * we return the given default value.
	 *
	 * @param name
	 * @param defaultValue
	 * @return
	 */
	public static  Level getLevelProperty(String name, Level defaultValue) {
		String val = LogManager.getLogManager().getProperty(name);
		if (val == null)
			return defaultValue;
		
		try {
			return Level.parse(val.trim());
		} catch (Exception ex) {
			return defaultValue;
		}
	}

	/**
	 * Static method to get a filter property.
	 * We return an instance of the class named by the "name" 
	 * property. If the property is not defined or has problems
	 * we return the defaultValue.
	 *
	 * @param name
	 * @param defaultValue
	 * @return
	 */
	public static Filter getFilterProperty(String name, Filter defaultValue) {
		String val = LogManager.getLogManager().getProperty(name);
		try {
			if (val != null) {
				Class<?> clz = ClassLoader.getSystemClassLoader().loadClass(val);
				return (Filter) clz.newInstance();
			}
		} catch (Exception ex) {
			// We got one of a variety of exceptions in creating the
			// class or creating an instance.
			// Drop through.
		}
		// We got an exception.  Return the defaultValue.
		return defaultValue;
	}

	/**
	 * Package private method to get a formatter property.
	 * We return an instance of the class named by the "name" 
	 * property. If the property is not defined or has problems
	 * we return the defaultValue.
	 *
	 * @param name
	 * @param defaultValue
	 * @return
	 */
	public static Formatter getFormatterProperty(
			String name, Formatter defaultValue)
	{
		String val = LogManager.getLogManager().getProperty(name);
		try {
			if (val != null) {
				Class<?> clz = ClassLoader.getSystemClassLoader().loadClass(val);
				return (Formatter) clz.newInstance();
			}
		} catch (Exception ex) {
			// We got one of a variety of exceptions in creating the
			// class or creating an instance.
			// Drop through.
		}
		// We got an exception.  Return the defaultValue.
		return defaultValue;
	}
}
