package civvi.common;

import java.math.BigDecimal;

import javax.xml.transform.Source;

/**
 * TODO
 * 
 * @author <a href="mailto=dansiviter@gmail.com">Daniel Siviter</a>
 * @since v1.0.0 [10th October 2008]
 */
public abstract class XmlBuilder {

	/**
	 * @return the builder node representing the root element of the XML
	 * document. In other words, the same builder node returned by the 
	 * {@link #create(String)} method.
	 */
	public abstract XmlBuilder root();

	/**
	 * Add a named XML element to the document as a child of this builder node,
	 * and return the builder node representing the new child. 
	 * 
	 * @param name the name of the XML element.
	 * @return a builder node representing the new child.
	 * @throws IllegalStateException if you attempt to add a child element to
	 * an XML node that already contains a text node value.
	 */
	public abstract XmlBuilder element(String name);

	/**
	 * Synonym for {@link #element(String)}. 
	 * 
	 * @param name the name of the XML element.
	 * @return a builder node representing the new child.
	 * @throws IllegalStateException if you attempt to add a child element to
	 * an XML node that already contains a text node value.
	 */
	public XmlBuilder elem(String name) {
		return element(name);
	}

	/**
	 * Synonym for {@link #element(String)}. 
	 * 
	 * @param name the name of the XML element.
	 * @return a builder node representing the new child.
	 * @throws IllegalStateException if you attempt to add a child element to
	 * an XML node that already contains a text node value.
	 */
	public XmlBuilder e(String name) {
		return element(name);
	}

	/**
	 * Add a named attribute value to the element represented by this builder
	 * node, and return the node representing the element to which the 
	 * attribute was added (<strong>not</strong> the new attribute node).
	 * 
	 * @param name the attribute's name.
	 * @param value the attribute's value.
	 * @return the builder node representing the element to which the attribute
	 * was added.
	 */
	public abstract XmlBuilder attribute(String name, String value);

	/**
	 * 
	 * @param name
	 * @param value
	 * @return
	 */
	public XmlBuilder a(String name, BigDecimal value) {
		return attribute(name, value);
	}

	/**
	 * 
	 * @param name
	 * @param value
	 * @return
	 */
	public XmlBuilder attr(String name, BigDecimal value) {
		return attribute(name, value);
	}

	/**
	 * 
	 * @param name
	 * @param value
	 * @return
	 */
	public XmlBuilder attribute(String name, BigDecimal value) {
		return attribute(name, value);
	}

	/**
	 * 
	 * @param name
	 * @param value
	 * @return
	 */
	public XmlBuilder a(String name, Object value) {
		return attribute(name, value);
	}

	/**
	 * 
	 * @param name
	 * @param value
	 * @return
	 */
	public XmlBuilder attr(String name, Object value) {
		return attribute(name, value);
	}

	/**
	 * 
	 * @param name
	 * @param value
	 * @return
	 */
	public	XmlBuilder attribute(String name, Object value) {
		if (value != null) {
			attribute(name, value.toString());
		}
		return this;
	}

	/**
	 * Synonym for {@link #attribute(String, String)}.
	 * 
	 * @param name the attribute's name.
	 * @param value the attribute's value.
	 * @return the builder node representing the element to which the attribute
	 * was added.
	 */
	public XmlBuilder attr(String name, String value) {
		return attribute(name, value);
	}

	/**
	 * Synonym for {@link #attribute(String, String)}.
	 * 
	 * @param name the attribute's name.
	 * @param value the attribute's value.
	 * @return the builder node representing the element to which the attribute
	 * was added.
	 */
	public XmlBuilder a(String name, String value) {
		return attribute(name, value);
	}

	/**
	 * Add a text value to the element represented by this builder node, and 
	 * return the node representing the element to which the text 
	 * was added (<strong>not</strong> the new text node).
	 * 
	 * @param value the text value to add to the element.
	 * @return the builder node representing the element to which the text was
	 * added.
	 */
	public abstract XmlBuilder text(String value);

	/**
	 * Add a new Element and add the given text to it
	 * 
	 * @param name the name of the element to add
	 * @param value the text to add the element
	 * @return the builder node representing the element to which the element
	 * was added.
	 */
	public XmlBuilder te(String name, Object value) {
		return textElement(name, value);
	}

	/**
	 * Add a new Element and add the given text to it
	 * 
	 * @param name the name of the element to add
	 * @param value the text to add the element
	 * @return the builder node representing the element to which the element
	 * was added.
	 */
	public abstract XmlBuilder textElement(String name, Object value);

	/**
	 * Add a new Element and add the given text to it
	 * 
	 * @param name the name of the element to add
	 * @param value the text to add the element
	 * @return the builder node representing the element to which the element
	 * was added.
	 */
	public XmlBuilder te(String name, BigDecimal value) {
		return textElement(name, value);
	}

	/**
	 * Add a new Element and add the given text to it
	 * 
	 * @param name the name of the element to add
	 * @param value the text to add the element
	 * @return the builder node representing the element to which the element
	 * was added.
	 */
	public XmlBuilder textElement(String name, BigDecimal value) {
		if (value != null) {
    		textElement(name, value.toPlainString());
    	}
        return this;
	}

	/**
	 * Add a text value to the element represented by this builder node, and 
	 * return the node representing the element to which the text 
	 * was added (<strong>not</strong> the new text node).
	 * 
	 * @param value the text value to add to the element.
	 * @return the builder node representing the element to which the text was
	 * added.
	 */
	public abstract XmlBuilder text(BigDecimal value);

	/**
	 * Synonmy for {@link #text(String)}.
	 * 
	 * @param value the text value to add to the element.
	 * @return the builder node representing the element to which the text was
	 * added.
	 */
	public XmlBuilder t(String value) {
		return text(value);
	}

	/**
	 * Add a CDATA value to the element represented by this builder node, and 
	 * return the node representing the element to which the data 
	 * was added (<strong>not</strong> the new CDATA node).
	 * 
	 * @param data the data value that will be Base64-encoded and added to a
	 * {@code CDATA} element.
	 * @return the builder node representing the element to which the data was
	 * added.
	 */
	public abstract XmlBuilder cdata(String data);

	/**
	 * Synonym for {@link #cdata(String)}.
	 * 
	 * @param data the data value to add to the element.
	 * @return the builder node representing the element to which the data was
	 * added.
	 */
	public XmlBuilder data(String data) {
		return cdata(data);
	}

	/**
	 * Synonym for {@link #cdata(String)}.
	 * 
	 * @param data the data value to add to the element.
	 * @return the builder node representing the element to which the data was
	 * added.
	 */
	public XmlBuilder d(String data) {
		return cdata(data);
	}

	/**
	 * Add a comment to the element represented by this builder node, and 
	 * return the node representing the element to which the comment 
	 * was added (<strong>not</strong> the new comment node).
	 * 
	 * @param comment the comment to add to the element.
	 * @return the builder node representing the element to which the comment
	 * was added.
	 */
	public abstract XmlBuilder comment(String comment);

	/**
	 * Synonym for {@link #comment(String)}.
	 * 
	 * @param comment the comment to add to the element.
	 * @return the builder node representing the element to which the comment
	 * was added.
	 */
	public XmlBuilder cmnt(String comment) {
		return comment(comment);
	}

	/**
	 * Synonym for {@link #comment(String)}.
	 * 
	 * @param comment the comment to add to the element.
	 * @return the builder node representing the element to which the comment
	 * was added.
	 */
	public XmlBuilder c(String comment) {
		return comment(comment);
	}

	/**
	 * Add an instruction to the element represented by this builder node, and 
	 * return the node representing the element to which the instruction 
	 * was added (<strong>not</strong> the new instruction node).
	 * 
	 * @param target the target value for the instruction.
	 * @param data the data value for the instruction
	 * @return the builder node representing the element to which the
	 * instruction was added.
	 */
	public abstract XmlBuilder instruction(String target, String data);

	/**
	 * Synonym for {@link #instruction(String, String)}.
	 * 
	 * @param target the target value for the instruction.
	 * @param data the data value for the instruction
	 * @return the builder node representing the element to which the
	 * instruction was added.
	 */
	public XmlBuilder inst(String target, String data) {
		return instruction(target, data);
	}

	/**
	 * Synonym for {@link #instruction(String, String)}.
	 * 
	 * @param target the target value for the instruction.
	 * @param data the data value for the instruction
	 * @return the builder node representing the element to which the
	 * instruction was added.
	 */
	public XmlBuilder i(String target, String data) {
		return instruction(target, data);
	}

	/**
	 * Add a reference to the element represented by this builder node, and 
	 * return the node representing the element to which the reference 
	 * was added (<strong>not</strong> the new reference node).
	 * 
	 * @param name the name value for the reference.
	 * @return the builder node representing the element to which the
	 * reference was added.
	 */
	public abstract XmlBuilder reference(String name);

	/**
	 * Synonym for {@link #reference(String)}.
	 * 
	 * @param name the name value for the reference.
	 * @return the builder node representing the element to which the
	 * reference was added.
	 */
	public XmlBuilder ref(String name) {
		return reference(name);
	}

	/**
	 * Synonym for {@link #reference(String)}.
	 * 
	 * @param name the name value for the reference.
	 * @return the builder node representing the element to which the
	 * reference was added.
	 */
	public XmlBuilder r(String name) {
		return reference(name);
	}

	/**
	 * Return the builder node representing the n<em>th</em> ancestor element 
	 * of this node, or the root node if n exceeds the document's depth.
	 *   
	 * @param steps the number of parent elements to step over while navigating
	 * up the chain of node ancestors. A steps value of 1 will find a node's
	 * parent, 2 will find its grandparent etc.
	 * @return the n<em>th</em> ancestor of this node, or the root node if this
	 * is reached before the n<em>th</em> parent is found.
	 */
	public abstract XmlBuilder up(int steps);

	/**
	 * Return the builder node representing the parent of the current node.
	 * 
	 * @return the parent of this node, or the root node if this method is
	 * called on the root node. 
	 */
	public XmlBuilder up() {
		return up(1);
	}

	/**
	 * Inserts the given XML document as a sub element of the current element.
	 * 
	 * @param source
	 * @return
	 */
	public abstract XmlBuilder insertSubTree(Source source);
}