package civvi.common;

import static javax.xml.datatype.DatatypeConstants.DATE;
import static javax.xml.datatype.DatatypeConstants.DATETIME;
import static javax.xml.datatype.DatatypeConstants.TIME;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.io.Writer;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Iterator;
import java.util.Map;

import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.datatype.DatatypeConstants;
import javax.xml.datatype.DatatypeFactory;
import javax.xml.datatype.XMLGregorianCalendar;
import javax.xml.namespace.NamespaceContext;
import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamResult;
import javax.xml.validation.Schema;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.apache.commons.lang.StringUtils;
import org.w3c.dom.Attr;
import org.w3c.dom.CharacterData;
import org.w3c.dom.DOMConfiguration;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * TODO
 * 
 * @author <a href="mailto=dansiviter@gmail.com">Daniel Siviter</a>
 * @since 10th October 2007
 */
public class XmlUtil {
	private static final DocumentBuilderFactory DOM_BUILDER_FACTORY =
		DocumentBuilderFactory.newInstance();
	
	private static final DatatypeFactory datatypeFactory;
	private static final XPathFactory XPATH_FACTORY = XPathFactory.newInstance();
	
	static {
		try {
			datatypeFactory = DatatypeFactory.newInstance();
		} catch (DatatypeConfigurationException dce) {
			throw new RuntimeException(dce);
		}
	}
	
	private XmlUtil() {
		// prevent construction.
	}

	/**
	 * Recursively removes all text nodes containing whitespace only from a document element. Also
	 * trims leading and trailing whitespace from text nodes.
	 * 
	 * @param e The root document element.
	 */
	public static void removeWhitespaceNodes(Element e) {
		NodeList children = e.getChildNodes();
		for (int i = children.getLength() - 1; i >= 0; i--) {
			Node child = children.item(i);
			if (child instanceof Text && ((Text) child).getData().trim().length() == 0)
				e.removeChild(child);
			else if (child instanceof Text)
				child.setTextContent(((Text) child).getData().trim());
			else if (child instanceof Element)
				removeWhitespaceNodes((Element) child);
		}
	}

	/**
	 * Removes all attributes named 'xsi:nil' from an {@link InputSource}.
	 * 
	 * @param inputSource InputSource holding the XML entity.
	 * @param namespace The namespace mapping to the 'xsi' prefix.
	 * @return A document with 'xsi:nil' attributes stripped.
	 * @throws RuntimeException if a {@link ParserConfigurationException}, {@link SAXException},
	 *             {@link IOException} or {@link XPathExpressionException} is thrown whilst
	 *             performing this operation.
	 */
	public static Document removeXsiNilAttributes(
			final InputSource inputSource,
			final String namespace)
	{
		// Generate a document.
		final DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
		factory.setNamespaceAware(true);
		Document doc = null;
		try {
			final DocumentBuilder builder = factory.newDocumentBuilder();
			doc = builder.parse(inputSource);
		} catch (ParserConfigurationException ex) {
			throw new RuntimeException("Could not generate document from input source", ex);
		} catch (SAXException ex) {
			throw new RuntimeException("Could not generate document from input source", ex);
		} catch (IOException ex) {
			throw new RuntimeException("Could not generate document from input source", ex);
		}

		// Create a namespace context.
		NamespaceContext nsContext = new NamespaceContext() {
			@SuppressWarnings("rawtypes")
			@Override
			public Iterator getPrefixes(String namespaceURI) {
				return null;
			}

			@Override
			public String getPrefix(String namespaceURI) {
				return null;
			}

			@Override
			public String getNamespaceURI(String prefix) {
				if (prefix.equals("xsi")) {
					return namespace;
				}
				return "";
			}
		};

		// Use an XPath expression to find all nodes containing xsi:nil attributes.
		final XPath xPath = XPathFactory.newInstance().newXPath();
		xPath.setNamespaceContext(nsContext);
		NodeList nodes = null;
		final String xPathExpression = "//*[@xsi:nil]";
		try {
			nodes = (NodeList) xPath.evaluate(xPathExpression, doc, XPathConstants.NODESET);
		} catch (XPathExpressionException ex) {
			throw new RuntimeException("Could not evaluate XPath expression=" + xPathExpression, ex);
		}

		// Remove the xsi:nil attribute nodes.
		if (null != nodes) {
			for (int i = 0; i < nodes.getLength(); i++) {
				if (nodes.item(i) instanceof Element) {
					((Element) nodes.item(i)).removeAttribute("xsi:nil");
				}
			}
		}

		// Reload the document, ensuring namespace declarations are removed where possible.
		final DOMConfiguration docConfig = doc.getDomConfig();
		docConfig.setParameter("namespace-declarations", false);
		doc.normalizeDocument();
		return doc;
	}

	/**
	 * Finds an attribute value from an initial node. If the attribute does not exist it will
	 * traverse the ancestor nodes to find it.
	 * 
	 * @param doc The document.
	 * @param startNodeXPath XPath expression for the starting node.
	 * @param attributeName The attribute name.
	 * @return The attribute name or an empty string if it wasn't evaluated.
	 */
	public static String getElementAttributeFromAncestors(
			final Document doc,
			final String startNodeXPath,
			final String attributeName)
	{
		// First check whether the attribute exists on the initial node.
		final String attributeValue = (String) evaluateXPathExpression(doc, startNodeXPath
				+ "/@" + attributeName, XPathConstants.STRING);

		// Now check the ancestors if necessary.
		if ("".equals(attributeValue)) {
			return (String) evaluateXPathExpression(doc, startNodeXPath
					+ "/ancestor::*[@" + attributeName + " != '']/@" + attributeName,
					XPathConstants.STRING);
		}
		return attributeValue;
	}
	
	/**
	 * Finds an attribute value from an initial attribute XPath. If the attribute does not exist it will
	 * traverse the ancestor nodes to find it.
	 * 
	 * @param doc The document.
	 * @param attributeXPath XPath expression for the starting attribute.
	 * @return The attribute name or an empty string if it wasn't evaluated.
	 */
	public static String getElementAttributeFromAncestors(
			final Document doc,
			final String attributeXPath)
	{
		final String nodeXPath = StringUtils.substringBefore(attributeXPath, "/@");
		final String attributeName = StringUtils.substringAfter(attributeXPath, "/@");
		return getElementAttributeFromAncestors(doc, nodeXPath, attributeName);
	}
		
	
	/**
	 * Creates a new instance of {@link Document}.
	 * 
	 * @return a new document.
	 * @throws ParserConfigurationException
	 */
	public static Document createDocument() {
		try {
			return createBuilder(null, false).newDocument();
		} catch (ParserConfigurationException pce) {
			throw new RuntimeException("Unable to create Document Builder Factory.", pce);
		}
	}

	/**
	 * Parses the XML {@link String} into a {@link Document}.
	 * 
	 * @param xml the XML contents to parse.
	 * @param schema the schema for the XML document or {@code null}.
	 * @param validating if {@code true} it will validate the document.
	 * @return
	 * @throws ParserConfigurationException
	 * @throws IOException
	 * @throws SAXException
	 */
	public static Document parse(String xml, Schema schema, boolean validating)
	throws ParserConfigurationException, IOException, SAXException
	{
		final InputStream is = new ByteArrayInputStream(xml.getBytes());
		return createBuilder(schema, validating).parse(is);
	}
	
	/**
	 * Parses the file into a {@link Document}.
	 * 
	 * @param file the file to read.
	 * @param schema the schema for the XML document or {@code null}.
	 * @param validating if {@code true} it will validate the document.
	 * @return the new document.
	 * @throws ParserConfigurationException
	 * @throws IOException
	 * @throws SAXException
	 */
	public static Document parse(File file, Schema schema, boolean validating)
	throws ParserConfigurationException, IOException, SAXException
	{
		return createBuilder(schema, validating).parse(file);
	}
	
	/**
	 * Persists the XML to the file system.
	 *
	 * @param src the source XML.
	 * @param file the file to persist to.
	 * @throws TransformerException
	 */
	public static void persist(Source src, File file)
	throws TransformerException
	{
		final TransformerFactory tranFactory = TransformerFactory.newInstance();
		final Transformer aTransformer = tranFactory.newTransformer();
		final Result dest = new StreamResult(file);
		aTransformer.transform(src, dest);
	}
	
	/**
	 * Returns a new instance of {@link DocumentBuilder}.
	 * 
	 * @param schema the schema for the XML document or {@code null}.
	 * @param validating if {@code true} it will validate the document.
	 * @return a new instance.
	 * @throws ParserConfigurationException
	 */
	private static DocumentBuilder createBuilder(
			Schema schema,
			boolean validating)
	throws ParserConfigurationException
	{
		synchronized (DOM_BUILDER_FACTORY) {
			DOM_BUILDER_FACTORY.setSchema(schema);
			DOM_BUILDER_FACTORY.setValidating(validating);
			return DOM_BUILDER_FACTORY.newDocumentBuilder();
		}
	}
	
	/**
	 * Returns a new instance of {@link XPath}.
	 * 
	 * @return a new instance.
	 */
	private static XPath createXPath() {
		synchronized (XPATH_FACTORY) {
			return XPATH_FACTORY.newXPath();
		}
	}
	
	/**
	 * Evaluates an XPath expression against a {@link Document}.
	 * 
	 * @param <T> the type of result
	 * @param expression the XPath expression
	 * @param source the document to execute the XPath against.
	 * @param returnType 
	 * @return
	 */
	public static <T extends QName> Object evaluateXPathExpression(
			final Document source,
			final String expression,
			final T returnType)
	{
		try {
			return createXPath().evaluate(expression, source, returnType);
		} catch (XPathExpressionException xee) {
			throw new RuntimeException(String.format(
					"Could not evalute the XPath Expression : %s", expression), xee);
		}
	}

	/**
	 * Adds a new empty {@link Element} to the given parent {@link Element}.
	 * 
	 * @param doc the document to add to.
	 * @param parent the parent element. If {@code null} then the new child
	 * will be added directly to the document.  
	 * @param name the name of the new element.
	 * @return the created element.
	 */
	public static Element addElement(
			Document doc,
			Node parent,
			String name)
	{
		if (doc == null) {
			doc = parent.getOwnerDocument();
		}
		final Element elem = doc.createElement(name);
		if (parent == null) {
			parent = doc;
		}
		parent.appendChild(elem);
		return elem;
	}
	
	

	/**
	 * Adds a new {@link Element} with the given text to the given parent
	 * {@link Element}.
	 * 
	 * @param doc the document to add to.
	 * @param parent the parent element.
	 * @param name the name of the new element.
	 * @param value the text value.
	 * @return the created element.
	 */
	public static Element addTextElement(
			Document doc,
			Node parent,
			String name,
			String value)
	{
		if (doc == null) {
			doc = parent.getOwnerDocument();
		}
		final Element elem = addElement(doc, parent, name);
		if (value != null) {
			elem.appendChild(doc.createTextNode(value));
		}
		return elem;
	}

	/**
	 * Adds a new {@link Element} with multiple attributes to the given parent
	 * {@link Element}. The attribute data should be passed in via a {@link Map},
	 * with each entry containing an attribute name as key and attribute value as the value.
	 * 
	 * @param doc
	 * @param parent
	 * @param name
	 * @param attributeData
	 * @return
	 */
	public static Element addElement(
			Document doc,
			Node parent,
			String name,
			Map<String, String> attributeData) {
		if (doc == null) {
			doc = parent.getOwnerDocument();
		}
		final Element elem = addElement(doc, parent, name);
		if (null != attributeData && !attributeData.isEmpty()) {
			Iterator<Map.Entry<String, String>> iter = attributeData.entrySet().iterator(); 
			while (iter.hasNext()) {
				Map.Entry<String, String> entry = iter.next();
				elem.setAttribute(entry.getKey(), entry.getValue());
			}
		}
		return elem;
	}

	/**
	 * Adds a new {@link Element} with multiple attributes to the given parent
	 * {@link Element}. The attribute data should be passed in pairs of key and
	 * attribute value as the value.
	 * 
	 * @param doc
	 * @param parent
	 * @param name
	 * @param attributeData
	 * @return
	 */
	public static Element addElement(
			Document doc,
			Node parent,
			String name,
			Object ... attributeData) {
		if (doc == null) {
			doc = parent.getOwnerDocument();
		}
		final Element elem = addElement(doc, parent, name);
		
		// check we have an even number of attribute pairs
		if ((attributeData.length & 1) == 1) {
			throw new RuntimeException("Need an even number attribute args to make attribute name/value pairs.");
		}
		addAttributes(elem, attributeData);
		return elem;
	}

	public static Element addAttributes(Element elem, Object... attributeData) {
		return addAttributes(elem, false, attributeData);
	}

	public static Element addAttributes(Element elem, boolean dropEmptyValues, Object... attributeData) {
		for (int i = 0; i < attributeData.length - 1; i += 2) {
			Object attribName = attributeData[i];
			Object attribValue = attributeData[i + 1];
			if (attribName != null && attribValue != null) {
				String value = attribValue.toString();
				if (!(dropEmptyValues && value.isEmpty())) {
					elem.setAttribute(attribName.toString(), value);
				}
			}
		}
		return elem;
	}

	/**
	 * Adds a {@link Attr} to the parent {@link Element}. 
	 *
	 * @param doc
	 * @param parent
	 * @param name
	 * @return
	 */
	public static Attr addAttr(Document doc, Element parent, String name) {
		final Attr attr = doc.createAttribute(name);
		parent.setAttributeNode(attr);
		return attr;
	}
	
	/**
	 * Extracts the {@link Element} for the given name from the parent.
	 * 
	 * @param parent
	 * @param name
	 * @return
	 */
	public  static <T extends Node> T findNode(
			Node parent,
			Class<T> type,
			String name)
	{
		final NodeList childNodes = parent.getChildNodes();
		
		for (int i = 0; i < childNodes.getLength(); i++) {
			final Node child = childNodes.item(i);
			if (type.isAssignableFrom(child.getClass()) &&
					name.equals(child.getNodeName()))
			{
				return type.cast(child);
			}
		}
		return null;
	}
	
	/**
	 * Extracts the {@link Element} for the given name from the parent.
	 * 
	 * @param parent
	 * @param name
	 * @return
	 */
	public static Element findElement(Node parent, String name) {
		return findNode(parent, Element.class, name);
	}
	
	/**
	 * Convenience method to copy the contents of the old {@link Node} into the
	 * new one.
	 * 
	 * @param newDoc
	 * @param newNode
	 * @param oldParent
	 */
	public static void copyContents(
			Document newDoc,
			Node newNode,
			Node oldNode)
	{
		// FIXME we should be able to achieve this with much less code (e.g.
		// the code commented out) but for some reason there are
		// incompatibility issues being spat out by the tests.
//		final NodeList childNodes = oldNode.getChildNodes();
//	  	for (int i = 0; i < childNodes.getLength(); i++) {
//	  		final Node child = newDoc.importNode(childNodes.item(i), true);
//	  		newNode.appendChild(child);
//	  	}
		
		final NodeList childs = oldNode.getChildNodes();

		for (int i = 0; i < childs.getLength(); i++) {
			final Node child = childs.item(i);
			Element newElem = null;
			switch (child.getNodeType()) {
			case Node.ELEMENT_NODE:
				//Get all the attributes of an element in a map
				final NamedNodeMap attrs = child.getAttributes();

				// Process each attribute
				newElem = newDoc.createElement(child.getNodeName());
				for (int j = 0; j < attrs.getLength(); j++) {
					final Attr attr = (Attr) attrs.item(j);
					// add attribute name and value to the new element
					newElem.setAttribute(
							attr.getNodeName(),
							attr.getNodeValue());
				}	
				newNode.appendChild(newElem);  
				break;

			case Node.TEXT_NODE:
				newNode.appendChild(newDoc.createTextNode(getString(child)));
			}
			copyContents(newDoc, newElem, child);
		}
	}

	@Deprecated
	private static String getString(Node child) {
		String s = "";
		if (child instanceof CharacterData) {
			CharacterData cd = (CharacterData) child;
			s = cd.getData();
		}
		while (s.startsWith("\n")) {
			s = s.substring(1);
		}

		return s; 
	}
	
	/**
	 * Transforms a {@link Source} object into a {@link String} representation.
	 * 
	 * @param xml the source input.
	 * @return the string output.
	 * @throws TransformerException
	 */
	public static String toString(Source xml) throws TransformerException {
		return toString(xml, false);
	}

	/**
	 * Transforms a {@link Source} object into a {@link String} representation.
	 * 
	 * @param xml the source input.
	 * @param pretty if {@code true} then pretty print with white space.
	 * @return the string output.
	 * @throws TransformerException
	 */
	public static String toString(Source xml, boolean pretty)
	throws TransformerException
	{
		StringWriter writer = new StringWriter();
		toString(xml, pretty, writer);
		return writer.toString();
	}

	/**
	 * Transforms a {@link Source} object into a {@link String} representation.
	 * 
	 * @param xml the source input.
	 * @param pretty if {@code true} then pretty print with white space.
	 * @param writer write the string to this {@link Writer}.
	 * @throws TransformerException
	 */
	public static void toString(Source xml, boolean pretty, Writer writer)
	throws TransformerException
	{
		final TransformerFactory factory = TransformerFactory.newInstance();
		
		try {
			factory.setAttribute("{http://xml.apache.org/xalan}indent-amount", 2);
		} catch (IllegalArgumentException iae) {
			// try a different ident amount
			try {
				factory.setAttribute("indent-number", 2);
			} catch (IllegalArgumentException iae2) {
				// ignore
			}
		}

		final Transformer transformer = factory.newTransformer();
		if (pretty) {
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
			transformer.setOutputProperty(OutputKeys.ENCODING, "utf-8");
		}
		transformer.transform(xml, new StreamResult(writer));
	}

	/**
	 * Transforms a {@link Source} object into a {@link String} representation.
	 * 
	 * @param xml the source input.
	 * @param pretty if {@code true} then pretty print with white space.
	 * @param writer write the string to this {@link Writer}.
	 */
	public static void xmlToString(Source xml, boolean pretty, Writer writer) {
		try {
			toString(xml, pretty, writer);
		} catch (TransformerException te) {
			throw new RuntimeException(te);
		}
	}

	/**
	 * Transforms a {@link Source} object into a {@link String} representation.
	 * 
	 * @param xml the source input.
	 * @return the string output.
	 */
	public static String xmlToString(Source xml, boolean pretty) {
		try {
			return toString(xml, pretty);
		} catch (TransformerException te) {
			throw new RuntimeException(te);
		}
	}

	/**
	 * Sets a date attribute onto the element. This uses ISO 8601 Date and Time
	 * formats as performed by {@link XMLGregorianCalendar}. The different type
	 * formats can be specified by the {@code type} field.
	 *
	 * @param elem the element to set on.
	 * @param attrName the attribute name.
	 * @param date the date value.
	 * @param type the type to format as. If {@code null} then it will default
	 * to {@link DatatypeConstants#DATETIME}.
	 * @see DatatypeConstants
	 * @see #setDateAttribute(Element, String, Calendar, QName)
	 */
	public static void setDateAttribute(
			Element elem,
			String attrName,
			Date date,
			QName type)
	{
		final XMLGregorianCalendar xmlDate = createXMLDate(date, type);
		setDateAttribute(elem, attrName, xmlDate);
	}

	/**
	 * Sets a date attribute onto the element. This uses ISO 8601 Date and Time
	 * formats as performed by {@link XMLGregorianCalendar}. The different type
	 * formats can be specified by the {@code type} field. Currently, only
	 * {@link DatatypeConstants#DATETIME}, {@link DatatypeConstants#DATE} and
	 * {@link DatatypeConstants#TIME} are supported.
	 *
	 * @param elem the element to set on.
	 * @param attrName the attribute name.
	 * @param calendar the date value.
	 * @param type the type to format as. If {@code null} then it will default
	 * to {@link DatatypeConstants#DATETIME}.
	 * @see DatatypeConstants
	 */
	public static void setDateAttribute(
			Element elem,
			String attrName,
			Calendar calendar,
			QName type)
	{
		final XMLGregorianCalendar xmlDate = createXMLDate(calendar, type);
		setDateAttribute(elem, attrName, xmlDate);
	}

	/**
	 * Sets the date onto the element.
	 *
	 * @param elem the element to set on.
	 * @param attrName the attribute name.
	 * @param calendar the date value.
	 * @see #setDateAttribute(Element, String, Calendar, QName)
	 */
	public static void setDateAttribute(
			Element elem,
			String attrName,
			XMLGregorianCalendar calendar)
	{
		elem.setAttribute(attrName, calendar.toXMLFormat());
	}
	
	/**
	 * Create XML date object. This uses ISO 8601 Date and Time formats as
	 * performed by {@link XMLGregorianCalendar}. The different type formats
	 * can be specified by the {@code type} field. Currently, only
	 * {@link DatatypeConstants#DATETIME}, {@link DatatypeConstants#DATE} and
	 * {@link DatatypeConstants#TIME} are supported.
	 *
	 * @param calendar the date value.
	 * @param type the type to format as. If {@code null} then it will default
	 * to {@link DatatypeConstants#DATETIME}.
	 * @see DatatypeConstants
	 */
	public static XMLGregorianCalendar createXMLDate(
			Date date,
			QName type)
	{
		final Calendar calendar = (GregorianCalendar) Calendar.getInstance();
		calendar.setTime(date);
		return createXMLDate(calendar, type);
	}
	
	/**
	 * Create XML date object. This uses ISO 8601 Date and Time formats as
	 * performed by {@link XMLGregorianCalendar}. The different type formats
	 * can be specified by the {@code type} field. Currently, only
	 * {@link DatatypeConstants#DATETIME}, {@link DatatypeConstants#DATE} and
	 * {@link DatatypeConstants#TIME} are supported.
	 *
	 * @param calendar the date value.
	 * @param type the type to format as. If {@code null} then it will default
	 * to {@link DatatypeConstants#DATETIME}.
	 * @see DatatypeConstants
	 */
	public static XMLGregorianCalendar createXMLDate(
			Calendar calendar,
			QName type)
	{
		final XMLGregorianCalendar xmlDate;
		if (type == null || type == DATETIME) {
			xmlDate = datatypeFactory.newXMLGregorianCalendar((GregorianCalendar) calendar);

		} else if (type == DATE) {
			xmlDate = datatypeFactory.newXMLGregorianCalendarDate(
					calendar.get(Calendar.YEAR),
					calendar.get(Calendar.MONTH) + 1,
					calendar.get(Calendar.DAY_OF_MONTH),
					calendar.getTimeZone().getRawOffset() == 0 ?
							DatatypeConstants.FIELD_UNDEFINED :
								calendar.getTimeZone().getRawOffset());

		} else if (type == TIME) {
			xmlDate = datatypeFactory.newXMLGregorianCalendarTime(
					calendar.get(Calendar.HOUR_OF_DAY),
					calendar.get(Calendar.MINUTE),
					calendar.get(Calendar.SECOND),
					calendar.get(Calendar.MILLISECOND),
					calendar.getTimeZone().getRawOffset() == 0 ?
							DatatypeConstants.FIELD_UNDEFINED :
								calendar.getTimeZone().getRawOffset());
		} else {
			throw new IllegalArgumentException(String.format(
					"Unknown date type! [type=%1$s]", type));
		}

		return xmlDate;
	}

	/**
	 * Returns a {@link Calendar} from the XML representation of a Gregorian
	 * calendar. (See 
	 * <a href="http://www.w3.org/TR/xmlschema-2/#dateTime-order">
	 * XML Schema 1.0 Part 2, Section 3.2.[7-14].1, <em>Lexical 
	 * Representation</em>.</a>).
	 *
	 * @param lexicalRepresentation
	 * @return a instance of calendar.
	 * @throws IllegalArgumentException If the {@code lexicalRepresentation}
	 * is not a valid <code>XMLGregorianCalendar</code>.
	 */
	public static Calendar parseDate(String lexicalRepresentation) {
		final XMLGregorianCalendar xmlDate =
			datatypeFactory.newXMLGregorianCalendar(lexicalRepresentation);
		return xmlDate.toGregorianCalendar();
	}

	/**
	 * Formats the date as in an XML style dictated by
	 * {@link XMLGregorianCalendar} and {@link DatatypeConstants}.
	 *
	 * @param calendar the date value.
	 * @param type the type to format as. If {@code null} then it will default
	 * to {@link DatatypeConstants#DATETIME}.
	 * @return a formatted {@link String}.
	 * @see #createXMLDate(Date, QName)
	 */
	public static String format(Date date, QName type) {
		return createXMLDate(date, type).toXMLFormat();
	}

	/**
	 * Formats the date as in an XML style dictated by
	 * {@link XMLGregorianCalendar} and {@link DatatypeConstants}.
	 *
	 * @param calendar the date value.
	 * @param type the type to format as. If {@code null} then it will default
	 * to {@link DatatypeConstants#DATETIME}.
	 * @return a formatted {@link String}.
	 * @see #createXMLDate(Calendar, QName)
	 */
	public static String format(Calendar calendar, QName type) {
		return createXMLDate(calendar, type).toXMLFormat();
	}

	/**
	 * Locates the node (regardless of a attribute or element) and sets the
	 * given value on it. If {@code preventOverwrite} is {@code true}
	 * if an existing value is already set then no value is set. This utility
	 * method can create the parent element chain including usage of filters.
	 *
	 * @param context
	 * @param xPathExpression
	 * @param value
	 * @param preventOverwrite
	 */
	public static String setValueViaXPath(
			Document doc,
			String xPathExpression,
			String value,
			boolean preventOverwrite)
	{
		return setValueViaXPath(
				doc,
				createXPath(),
				xPathExpression,
				value,
				preventOverwrite);
	}
	
	/**
	 * Locates the node (regardless of a attribute or element) and sets the
	 * given value on it. If {@code preventOverwrite} is {@code true}
	 * if an existing value is already set then no value is set. This utility
	 * method can create the parent element chain including usage of filters.
	 *
	 * @param context
	 * @param xPath
	 * @param xPathExpression
	 * @param value
	 * @param preventOverwrite
	 */
	public static String setValueViaXPath(
			Document doc,
			XPath xPath,
			String xPathExpression,
			String value,
			boolean preventOverwrite)
	{
		if (value == null) {
			return null;
		}

		Node node = evalXPath(
				doc,
				xPath,
				xPathExpression,
				Node.class);
		
		if (node == null) {
			node = addNode(doc, xPathExpression);
		}

		final String currentValue = node.getTextContent();
		
		if (currentValue.isEmpty() || !preventOverwrite) {
			node.setTextContent(value);
			return value;
		}
		return currentValue;
	}
	
	/**
	 * Adds the node chain defined by the {@code xPath}. If the {@link Node}
	 * at the end of the path exists then it will just return it.
	 *
	 * @param doc
	 * @param xPathExpression
	 * @return a {@link Attr} or a {@link Element}.
	 */
	public static Node addNode(Document doc, final String xPathExpression) {
		Element parent;
		Node currentNode = doc.getDocumentElement();
		
		final String[] names = xPathExpression.split("/");
		
		for (String name : names) {
			if (name.isEmpty()) {
				continue;
			}
			
			String[] filters = new String[0];
			
			if (name.contains("[")) {
				final String nodeString = name.substring(
						name.indexOf("[") + 1,
						name.lastIndexOf("]"));
				filters = nodeString.split("\\]\\[");
				name = StringUtils.substringBefore(name, "[");
			}
			if (filters.length == 0 && currentNode.getNodeName().equals(name)) {
				continue;
			}
			
			parent = (Element) currentNode;
			currentNode = getChildElement(parent, name, filters);

			if (currentNode == null) {
				if (name.startsWith("@")) {
					currentNode = addAttr(
							doc,
							parent,
							StringUtils.substringAfter(name, "@"));
				} else {
					final Object[] attributeData = new String[filters.length * 2];
					for (int i = 0; i < filters.length; i++) {
						attributeData[i * 2] = extractFilterName(filters[i]);
						attributeData[i * 2 + 1] = extractFilterValue(filters[i]);					
					}
					currentNode = addElement(
							doc,
							parent,
							name,
							attributeData);
				}
			}
		}
		return currentNode;
	}
	
	/**
	 * Get the first child element with the given name
	 *
	 * @param parentElement
	 * @param name
	 * @return
	 */
	public static Node getChildElement(Element parentElement, String name, String... filters) {
		final NodeList nodeList = parentElement.getElementsByTagName(name);

		if (nodeList.getLength() > 0) {
			for (int i = 0; i < nodeList.getLength(); i++) {
				final Node node = (Node) nodeList.item(i);
				
				if (node instanceof Element &&
						matchesFilters((Element) node, filters))
				{
					return node;
				} else if (node instanceof Attr) { 
					return node;
				}
			}
		}
		return null;
	}

	/**
	 * Returns {@code true} if the element matches the filters.
	 *
	 * @param elem
	 * @param filters
	 * @return
	 */
	private static boolean matchesFilters(Element elem, String... filters) {
		for (String filter : filters) {
			final String attribute = extractFilterName(filter);
			final String value = extractFilterValue(filter);

			if (!elem.getAttribute(attribute).equals(value)) {
				return false;		
			}
		}
		return true;
	}
	
	/**
	 * Convenience method to extract name from {@code @fooAttribute="blagh"}
	 * style filter.
	 *
	 * @param filter
	 * @return
	 */
	private static String extractFilterName(String filter) {
		if (filter.startsWith("@")) {
			return filter.substring(1, filter.indexOf("="));
		} else if (filter.startsWith(".")) {
			// means this element, so return null
			return null;
		} else {
			return filter.substring(0, filter.indexOf("="));
		}
	}

	/**
	 * Convenience method to extract value from {@code @fooAttribute="blagh"}
	 * style filter.
	 *
	 * @param filter
	 * @return
	 */
	private static String extractFilterValue(String filter) {
		return filter.substring(filter.indexOf("=\"") + 2, filter.lastIndexOf("\""));
	}

	/**
	 * Evaluates the xPath expression.
	 *
	 * @param <T>
	 * @param doc
	 * @param expression
	 * @param type either a {@link String}, {@link Node}, {@link Attr},
	 * {@link Element}, {@link Double}, {@link Boolean} or a {@link NodeList}.
	 * @return the found object.
	 * @see XPathConstants
	 */
	public static <T> T evalXPath(
			Document doc,
			String expression,
			Class<T> type)
	{
		return evalXPath(doc, createXPath(), expression, type);
	}
	
	/**
	 * Evaluates the xPath expression.
	 *
	 * @param <T>
	 * @param doc
	 * @param xPath
	 * @param expression
	 * @param type either a {@link String}, {@link Node}, {@link Attr},
	 * {@link Element}, {@link Double}, {@link Boolean}, {@link NodeList},
	 * {@link Calendar} or a {@link Date}.
	 * @return the found object.
	 * @see XPathConstants
	 */
	public static <T> T evalXPath(
			Document doc,
			XPath xPath,
			String expression,
			Class<T> type)
	{
		final QName returnType;

		if (type == String.class) {
			returnType = XPathConstants.STRING;
		} else if (Node.class.isAssignableFrom(type)) {
			returnType = XPathConstants.NODE;
		} else if (type == NodeList.class) {
			returnType = XPathConstants.NODESET;
		} else if (type == Boolean.class) {
			returnType = XPathConstants.BOOLEAN;
		} else if (Number.class.isAssignableFrom(type)) {
			returnType = XPathConstants.NUMBER;
		} else if (type == Date.class || type == Calendar.class) {
			returnType = null; // special case, not used!
		} else {
			throw new IllegalArgumentException(String.format(
					"Unknown type! [type=%s]", type));
		}

		try {
			Object result;

			if (type == Calendar.class) {
				// if a calendar is needed get string value
				final String lexicalRepresentation = evalXPath(doc, expression, String.class);
				result = parseDate(lexicalRepresentation);
			} else if (type == Date.class) {
				// just get calendar, but then extract a Date via #getTime()
				final Calendar date = evalXPath(doc, expression, Calendar.class);
				result = date.getTime();
			} else {
				result = xPath.evaluate(expression, doc, returnType);
			}

			return type.cast(result);
		} catch (XPathExpressionException ex) {
			throw new RuntimeException(String.format(
					"Could not evaluate XPath expression! [xPath=%s]",
					expression), ex);
		}
	}
}
