package civvi.common.builder;

import java.io.StringWriter;
import java.io.Writer;
import java.math.BigDecimal;
import java.util.Properties;
import java.util.Map.Entry;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import civvi.common.Base64;
import civvi.common.XmlBuilder;

/**
 * XML Builder is a utility that creates simple XML documents using relatively 
 * sparse Java code. It is intended to allow for quick and painless creation of 
 * XML documents where you might otherwise be tempted to use concatenated 
 * strings, rather than face the tedium and verbosity of coding with 
 * JAXP (http://jaxp.dev.java.net/).
 * <p>
 * Internally, XML Builder uses JAXP to build a standard W3C 
 * {@link org.w3c.dom.Document} model (DOM) that you can easily export as a 
 * string, or access and manipulate further if you have special requirements.
 * </p> 
 * <p>
 * The XMLBuilder class serves as a wrapper of {@link org.w3c.dom.Element} nodes,
 * and provides a number of utility methods that make it simple to
 * manipulate the underlying element and the document to which it belongs. 
 * In essence, this class performs dual roles: it represents a specific XML 
 * node, and also allows manipulation of the entire underlying XML document.
 * The platform's default {@link DocumentBuilderFactory} and 
 * {@link DocumentBuilder} classes are used to build the document.
 * </p>
 *  
 * @author James Murty
 */
public class XmlBuilderDom extends XmlBuilder {
	/*
	 * A DOM Document that stores the underlying XML document operated on by 
	 * XMLBuilder instances. This document object belongs to the root node
	 * of a document, and is shared by this node with all other XMLBuilder 
	 * instances via the {@link #getDocument()} method. 
	 * This instance variable must only be created once, by the root node for 
	 * any given document. 
	 */
	private Document xmlDocument = null;

	/*
	 * This node's parent XML builder node, if any. All element nodes except 
	 * the docuemnt's root will have a parent.
	 */
	private XmlBuilderDom myParent = null;

	/*
	 * The underlying element represented by this builder node. 
	 */
	private Element xmlElement = null;

	/**
	 * Construct a new builder object that wraps the given XML element, which
	 * in turn will belong to an underlying XML document.  
	 * This constructor is for internal use only.
	 * 
	 * @param xmlDocument a new and empty XML document which the builder will
	 * manage and manipulate.
	 * @param myElement the XML element that this builder node will wrap. This
	 * element will be added as the root of the underlying XML document. 
	 */
	protected XmlBuilderDom(Document xmlDocument, Element myElement) {
		this.myParent = null;
		this.xmlElement = myElement;
		this.xmlDocument = xmlDocument;
		this.xmlDocument.appendChild(myElement);
	}

	/**
	 * Construct a new builder object that wraps the given XML element, and
	 * is the child of an existing XML builder node.
	 * This constructor is for internal use only.
	 * 
	 * @param parent
	 * the builder node that will contain (be the parent of) the new 
	 * XML element.
	 * @param myElement
	 * the XML element that this builder node will wrap. This element will be
	 * added as child node of the parent's XML element.
	 */
	protected XmlBuilderDom(XmlBuilderDom parent, Element myElement) {
		this.myParent = parent;
		this.xmlElement = myElement;
		this.xmlDocument = parent.xmlDocument;
		parent.xmlElement.appendChild(myElement);
	}

	/**
	 * Construct a builder for new XML document. The document will be created
	 * with the given root element, and the builder returned by this method
	 * will serve as the starting-point for any further document additions.
	 * 
	 * @param name
	 * the name of the document's root element.
	 * @return
	 * a builder node that can be used to add more nodes to the XML document.
	 * 
	 * @throws FactoryConfigurationError 
	 * @throws ParserConfigurationException 
	 */
	public static XmlBuilderDom create(String name) throws XmlBuilderException 
	{
		// Init DOM builder and Document.
		DocumentBuilder builder;
		try {
			builder = DocumentBuilderFactory.newInstance().newDocumentBuilder();
		} catch (ParserConfigurationException e) {
			throw new XmlBuilderException(e);
		}
		Document document = builder.newDocument();
		return new XmlBuilderDom(document, document.createElement(name));
	}

	/**
	 * @return
	 * the XML element that this builder node will manipulate.
	 */
	public Element getElement() {
		return xmlElement;
	}

	@Override
	public XmlBuilderDom root() {
		// Navigate back through all parents to find the document's root node.
		XmlBuilderDom curr = this;
		while (curr.myParent != null) {
			curr = curr.myParent;
		}
		return curr;
	}

	/**
	 * @return
	 * the XML document constructed by all builder nodes.
	 */
	public Document getDocument() {
		return root().xmlDocument;
	}

	@Override
	public XmlBuilderDom element(String name) {
		// Ensure we don't create sub-elements in Elements that already have text node values.
		Node textNode = null;
		NodeList childNodes = xmlElement.getChildNodes();
		for (int i = 0; i < childNodes.getLength(); i++) {
			if (Element.TEXT_NODE == childNodes.item(i).getNodeType()) {
				textNode = childNodes.item(i);
				break;
			}
		}
		if (textNode != null) {
			throw new IllegalStateException("Cannot add sub-element <" +
					name + "> to element <" + xmlElement.getNodeName() 
					+ "> that already contains the Text node: " + textNode);
		}

		XmlBuilderDom child = new XmlBuilderDom(this, getDocument().createElement(name));        
		return child;
	}

	@Override
	public XmlBuilderDom attribute(String name, String value) {
		xmlElement.setAttribute(name, value);
		return this;
	}

	@Override
	public XmlBuilderDom text(String value) {
		if (value != null) {
			xmlElement.appendChild(getDocument().createTextNode(value));
		}
		return this;
	}

	@Override
	public XmlBuilder text(BigDecimal value) {
		if (value != null) {
			text(value.toPlainString());
		}
		return this;
	}

	@Override
	public XmlBuilderDom textElement(String name, Object value) {
		if (value != null) {
			XmlBuilderDom elem = element(name);
			elem.text(value.toString());
			elem.up();
		}
		return this;
	}

	@Override
	public XmlBuilderDom cdata(String data) {
		xmlElement.appendChild(getDocument().createCDATASection(
				new String(Base64.encode(data.getBytes()))));
		return this;
	}

	@Override
	public XmlBuilderDom comment(String comment) {
		xmlElement.appendChild(getDocument().createComment(comment));
		return this;
	}

	@Override
	public XmlBuilderDom instruction(String target, String data) {
		xmlElement.appendChild(getDocument().createProcessingInstruction(target, data));
		return this;
	}

	@Override
	public XmlBuilderDom reference(String name) {
		xmlElement.appendChild(getDocument().createEntityReference(name));
		return this;
	}

	@Override
	public XmlBuilderDom up(int steps) {
		XmlBuilderDom curr = this;
		int stepCount = 0;
		while (curr.myParent != null && stepCount < steps) {
			curr = curr.myParent;            
			stepCount++;
		}        
		return curr;
	}

	@Override
	public XmlBuilderDom up() {
		this.xmlElement = null;
		this.xmlDocument = null;
		return (XmlBuilderDom) super.up();
	}

	@Override
	public XmlBuilderDom insertSubTree(Source source) {
		if (source instanceof DOMSource) {
			DOMSource domSource = (DOMSource) source;
			xmlElement.appendChild(xmlDocument.adoptNode(domSource.getNode()));
		} else {
			throw new XmlBuilderException("Unsuported source type: " + source.getClass());
		}

		return this;
	}

	/**
	 * Serialize the XML document to the given writer using the default 
	 * {@link TransformerFactory} and {@link Transformer} classes. If output
	 * options are provided, these options are provided to the 
	 * {@link Transformer} serializer. 
	 * 
	 * @param writer
	 * a writer to which the serialized document is written.
	 * @param outputProperties
	 * settings for the {@link Transformer} serializer. This parameter may be
	 * null or an empty Properties object, in which case the default output
	 * properties will be applied.
	 * 
	 * @throws TransformerException
	 */
	public void toWriter(Writer writer, Properties outputProperties) 
	throws TransformerException 
	{
		final StreamResult streamResult = new StreamResult(writer);
		final DOMSource domSource = new DOMSource(getDocument());
		final TransformerFactory tf = TransformerFactory.newInstance();
		final Transformer serializer = tf.newTransformer();

		if (outputProperties != null) {
			for (Entry<Object, Object> entry : outputProperties.entrySet()) {
				serializer.setOutputProperty(
						(String) entry.getKey(),
						(String) entry.getValue());
			}
		}
		serializer.transform(domSource, streamResult);
	}

	/**
	 * 
	 * @param outputProperties
	 * @return
	 * @throws TransformerException
	 */
	public String asString(Properties outputProperties) 
	throws TransformerException 
	{
		final StringWriter writer = new StringWriter();
		toWriter(writer, outputProperties);
		return writer.toString();        
	}
}
