package civvi.common.builder;

import javax.xml.stream.XMLStreamConstants;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;

//import org.codehaus.stax2.XMLStreamReader2;
//import org.codehaus.stax2.XMLStreamWriter2;

/**
 * Reads all events from a given StAX {@link XMLStreamReader} parser and pipes
 * them into a given StAX {@link XMLStreamWriter} serializer.
 * 
 * @author whoschek.AT.lbl.DOT.gov
 * @author $Author: hoschek $
 * @version $Revision: 1.3 $, $Date: 2006-12-16 01:58:19 $
 */
final class XmlStreamCopier {

	private final XMLStreamReader reader;
	private final XMLStreamWriter writer;
	
	public XmlStreamCopier(XMLStreamReader reader, XMLStreamWriter writer) {
		if (reader == null)
			throw new IllegalArgumentException("reader must not be null");
		if (writer == null)
			throw new IllegalArgumentException("writer must not be null");
		
		this.reader = reader;
		this.writer = writer;
	}
	
	/**
	 * Reads all events from the reader and pipes them into the writer.
	 * 
	 * @param isFragmentMode
	 *            if true, copies all events of the XML fragment starting at the
	 *            current cursor position, if false copies all events of the
	 *            document starting at the current cursor positions.
	 * @throws XMLStreamException
	 *             If an error occurs while copying the stream
	 */
	public void copy(boolean isFragmentMode) throws XMLStreamException {
		int ev = isFragmentMode ? 
			XMLStreamConstants.START_ELEMENT : XMLStreamConstants.START_DOCUMENT;
		reader.require(ev, null, null);

		int depth = 0;
		ev = reader.getEventType();
		while (true) {
			switch (ev) {
				case XMLStreamConstants.START_ELEMENT: {
					final String prefix = reader.getPrefix();
					final String localName = reader.getLocalName();
					final String namespaceURI = reader.getNamespaceURI();
					if (namespaceURI == null) {
						writer.writeStartElement(localName);
					} else {
						writer.writeStartElement(
								prefix, 
								localName, 
								namespaceURI);
					}
					copyAttributes();
					copyNamespaces();
					depth++;
					break;
				}
				case XMLStreamConstants.END_ELEMENT: {
					writer.writeEndElement();
					depth--;
					if (isFragmentMode && depth == 0) {
						writer.flush();
						return; // we're done
					}
					break;
				}
				case XMLStreamConstants.ATTRIBUTE: {
					// can happen as part of an XPath result sequence, or similar
					copyAttribute(0);
					break;
				}
				case XMLStreamConstants.START_DOCUMENT:	{
					copyStartDocument();
					break;
				}
				case XMLStreamConstants.END_DOCUMENT: {
					writer.writeEndDocument();
					writer.flush();
					return; // we're done
				}
				case XMLStreamConstants.PROCESSING_INSTRUCTION: {
					writer.writeProcessingInstruction(
							reader.getPITarget(), reader.getPIData());
					break;
				}
				case XMLStreamConstants.COMMENT: {
					writer.writeComment(reader.getText());
					break;
				}
				case XMLStreamConstants.CDATA: {
					writer.writeCData(reader.getText());
					break;
				}
				case XMLStreamConstants.SPACE:
				case XMLStreamConstants.CHARACTERS: {
					copyText();
					break;
				}
				case XMLStreamConstants.ENTITY_REFERENCE: {
//					writer.writeEntityRef(reader.getLocalName()); // don't expand the ref
					copyText(); // expand the ref (safer)
					break;
				}
				case XMLStreamConstants.DTD: {
					copyDTD();
					break;
				}
				case XMLStreamConstants.ENTITY_DECLARATION: 
					break; // ignore (handled by XMLStreamConstants.DTD)
				case XMLStreamConstants.NOTATION_DECLARATION: 
					break; // ignore (handled by XMLStreamConstants.DTD)
				case XMLStreamConstants.NAMESPACE: {
					// can happen as part of an XPath result sequence, or similar
					writer.writeNamespace(reader.getPrefix(), reader.getNamespaceURI());
					break;
				}
				default: {
					throw new XmlBuilderException("Unrecognized event type: " 
							+ reader.getEventType());
				}
			}
			
			ev = reader.next();
		}
	}
	
	private void copyAttributes() throws XMLStreamException {
		int count = reader.getAttributeCount();
		for (int i = 0; i < count; i++) {
			copyAttribute(i);
		}
	}
	
	private void copyAttribute(int i) throws XMLStreamException {
		final String attributeLocalName = reader.getAttributeLocalName(i);
		final String attributeValue = reader.getAttributeValue(i);
		final String attributeNamespace = reader.getAttributeNamespace(i);
		
		if (attributeNamespace == null) {
			writer.writeAttribute(attributeLocalName, attributeValue);
		} else {
			writer.writeAttribute(
					reader.getAttributePrefix(i),
					attributeNamespace,
					attributeLocalName,
					attributeValue);
		}
	}
	
	private void copyNamespaces() throws XMLStreamException {
		int count = reader.getNamespaceCount();
		for (int i = 0; i < count; i++) {
			String prefix = reader.getNamespacePrefix(i);
			String namespaceURI = reader.getNamespaceURI(i);			
			writer.writeNamespace(prefix, namespaceURI);
		}
	}

//	private char[] buf = new char[2048];
	private void copyText() throws XMLStreamException {
		writer.writeCharacters(reader.getText());
		
//		int len = buf.length;
//		for (int start = 0; ; start += len) {
//			int n = reader.getTextCharacters(start, buf, 0, len);
//			writer.writeCharacters(buf, 0, n);
//			if (n < len) break;
//		}
	}
	
	private void copyStartDocument() throws XMLStreamException {
		String encoding = "UTF-8"; // FIXME can't claim it's UTF-8
		//String encoding = ((XMLStreamWriter2)writer).getEncoding();
		String version = reader.getVersion();
		if (version == null) version = "1.0";
		writer.writeStartDocument(encoding, version); 
	}	

	private void copyDTD() throws XMLStreamException {
		// writer.writeDTD(reader.getText());
//		Nodes nodes = DTDParser.readDocType(reader, new NodeFactory());
//		if (nodes.size() > 0) {
//			String str = nodes.get(0).toXML();
//			writer.writeDTD(str);
//		}		
	}

}
