package civvi.osgi.desktop;

import java.awt.Image;
import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ImageIcon;

import civvi.osgi.desktop.common.ActionComparator;

/**
 * An {@link Action} for representing a bean via it's {@link BeanInfo}.
 * 
 * @author <a href="mailto=dansiviter@gmail.com">Daniel Siviter</a>
 * @since 24th July 2008
 */
public abstract class AbstractBeanInfoAction extends AbstractAction
implements Comparable<AbstractBeanInfoAction>
{
	private static final long serialVersionUID = 7952615618763659079L;
	
	protected final Logger log = Logger.getLogger(getClass().getName());
	protected final Class<?> type;
	protected final BeanInfo beanInfo;
	private final ActionComparator comparator = new ActionComparator();

	/**
	 * Constructs a new Action for the given type.
	 * 
	 * @param type the bean type.
	 */
	public AbstractBeanInfoAction(Class<?> type) {
		super();
		this.type = type;
		
		final BeanInfo beanInfo;
		try {
			beanInfo = Introspector.getBeanInfo(type);
		} catch (IntrospectionException ie) {
			this.beanInfo = null;
			final String message = String.format(
					"Unable to locate Bean Info! [type=%s]", type);
			if (log.isLoggable(Level.WARNING)) {
				log.log(Level.WARNING, message, ie);
			}
			putValue(Action.NAME, message);
			return;
		}
		
		this.beanInfo = beanInfo;
		putValue(Action.NAME, this.beanInfo.getBeanDescriptor().getDisplayName());
		putValue(Action.SMALL_ICON, getImage(BeanInfo.ICON_COLOR_16x16));
		putValue(Action.LARGE_ICON_KEY, getImage(BeanInfo.ICON_COLOR_32x32));
		putValue(Action.SHORT_DESCRIPTION, this.beanInfo.getBeanDescriptor().getShortDescription());
	}

	/**
	 * Returns the image for the given {@code iconKind}.
	 *
	 * @param iconKind
	 * @return an image of {@link ImageIcon} or {@code null} if none is found.
	 * @see BeanInfo#ICON_COLOR_16x16
	 * @see BeanInfo#ICON_COLOR_32x32
	 */
	private ImageIcon getImage(int iconKind) {
		final Image largeIcon = this.beanInfo.getIcon(iconKind);
		return largeIcon != null ? new ImageIcon(largeIcon) : null;
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public int compareTo(AbstractBeanInfoAction o) {
		return this.comparator.compare(this, o);
	}
}
