package civvi.osgi.desktop;

import java.util.prefs.Preferences;

import javax.swing.Action;

import org.osgi.framework.BundleContext;

import civvi.osgi.desktop.preferences.AbstractPreferencePane;
import civvi.osgi.desktop.preferences.DefaultPreferencesPane;
import civvi.osgi.desktop.swingx.AppFrameworkSupport;
import civvi.osgi.desktop.view.AbstractView;

/**
 * An abstract implementation of {@link DesktopService}. Unless you have a
 * really strange requirement, then I'd always use this as your base class
 * for a {@link DesktopService}.
 * 
 * @author <a href="mailto=dansiviter@gmail.com">Daniel Siviter</a>
 * @since v1.0.0 [1st September 2008]
 */
public abstract class AbstractDesktopService<C extends DesktopContext>
implements DesktopService<C>
{
	protected final AppFrameworkSupport<OSGiDesktop> support;
	protected final BundleContext bundleContext;

	private C context;
	private volatile boolean initialised;

	/**
	 * Construct a service taking an instance of the {@link BundleContext}.
	 * 
	 * @param bundleContext
	 */
	public AbstractDesktopService(BundleContext bundleContext) {
		this.support = new AppFrameworkSupport<OSGiDesktop>(this);
		this.bundleContext = bundleContext;
	}

	/**
	 * @return a new instance of the context for this service.
	 */
	protected C createContext() {
		return null;
	}

	@Override
	public synchronized C getContext() {
		if (!isInitialised()) {
			throw new DesktopRuntimeException("Desktop Service not initialised!");
		}
		return this.context;
	}

	@Override
	public final void init() {
		if (isInitialised()) {
			throw new DesktopRuntimeException("Cannot be initialised more than once! " +
					"Invoke #destroy() before attempting to initalise.");
		}
		synchronized (this) {
			if (isInitialised()) {
				return;
			}
			
			this.context = createContext();
			if (this.context != null) this.context.init();
			doInit();
			this.initialised = true;
		}
	}

	/**
	 * Performs the internal process of initialising. Override to add
	 * functionality.
	 */
	protected void doInit() { }

	@Override
	public boolean isInitialised() {
		return initialised;
	}

	@Override
	public final void destroy() {
		if (!isInitialised()) {
			throw new DesktopRuntimeException(
					"Cannot destroy a un-initialised service!");
		}
		synchronized (this) {
			if (!isInitialised()) {
				return;
			}
			if (this.context != null) this.context.destroy();
			doDestroy();
			this.initialised = false;
		}
	}

	/**
	 * Performs the internal process of destroying what was initialised.
	 * Override to add functionality.
	 */
	protected void doDestroy() { }

	@Override
	public Preferences getPreferences() {
		return Preferences.userNodeForPackage(getClass());
	}

	@Override
	@SuppressWarnings("unchecked")
	public Class<? extends AbstractView>[] getViews() {
		return new Class[0];
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Class<? extends AbstractPreferencePane> getPreferencesEditor() {
		return DefaultPreferencesPane.class;
	}

	@Override
	public Action getAction(String actionKey, boolean allowCreateEmpty) {
		return this.support.getAction(actionKey, allowCreateEmpty);
	}
}
