package civvi.osgi.desktop;

import java.beans.IntrospectionException;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JToolBar;

import org.osgi.framework.ServiceReference;

import civvi.osgi.desktop.annotation.Menu;
import civvi.osgi.desktop.annotation.MenuAction;
import civvi.osgi.desktop.annotation.MenuActions;
import civvi.osgi.desktop.annotation.MenuBar;
import civvi.osgi.desktop.annotation.ToolbarAction;
import civvi.osgi.desktop.annotation.ToolbarActions;
import civvi.osgi.desktop.swingx.ActionProvider;
import civvi.osgi.desktop.swingx.JXMenuBar;
import civvi.osgi.desktop.swingx.SwingUtil;
import civvi.osgi.desktop.swingx.docking.perspective.Perspective;
import civvi.osgi.desktop.swingx.menu.MenuModel;
import civvi.osgi.desktop.view.AbstractEditorView;
import civvi.osgi.desktop.view.AbstractView;

/**
 * Manages the actions declared in the application. This utilises the
 * {@link Menu} and {@link ToolbarAction} annotations to configure placement of the
 * items within the components. If declared on a view then they will only be
 * shown if the view is selected. If declared on a plugin, they will be shown
 * at all times.
 * 
 * TODO once rebuild works, attempt insert/remove of actions specific to the
 * registration/unregistration of a source object.
 * 
 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
 * @since v1.0.0 [14 Jun 2010]
 */
public class ActionManager {
//	private final Logger log = Logger.getLogger(getClass().getName());
	private final OSGiDesktop desktop;
	private final Map<String, ActionProvider> registered = new HashMap<String, ActionProvider>();

	/**
	 * Creates an {@link Action} manager for the desktop.
	 * @param desktop
	 */
	public ActionManager(OSGiDesktop desktop) {
		this.desktop = desktop;
		this.desktop.getDockingPane().addPropertyChangeListener(
				"selectedDockable",
				new SelectedHandler());
		this.desktop.getActivator().getServiceManager().addListener(new DesktopServiceHandler());

		register(desktop);
		for (DesktopService<?> service : 
			this.desktop.getActivator().getServiceManager().getAll().values())
		{
			register(service);
		}
	}

	/**
	 * @param source the source object to register.
	 */
	public void register(ActionProvider source) {
		if (this.registered.containsKey(source.getClass().getName())) {
			return;
		}

		this.registered.put(source.getClass().getName(), source);
		this.desktop.getMenuBar().register(source);

		final JXMenuBar menuBar = this.desktop.getMenuBar();
		final MenuModel model = menuBar.getModel();
		final MenuActions menuActions =
			source.getClass().getAnnotation(MenuActions.class);
		if (menuActions != null) {
			for (MenuAction action : menuActions.value()) {
				if (!action.path().startsWith("desktop.main"))
					throw new UnsupportedOperationException("Only 'desktop.main' menu supported at this time! [" + action.path() + "]");
				if (action.path().startsWith("desktop.main/")) {
					model.add(action.actionId(), action.path().replace("desktop.main/", ""));
				} else {
					model.add(action.actionId(), null);
				}
			}
		}

		final JToolBar toolBar = this.desktop.getToolBar();
		final ToolbarActions toolbarActions =
			source.getClass().getAnnotation(ToolbarActions.class);
		if (toolbarActions != null) {
			for (ToolbarAction action : toolbarActions.value()) {
				final AbstractButton button = SwingUtil.createToolbarItem(
						source.getAction(getActionKey(action.actionId()), false));
				toolBar.add(button);
			}
		}
	}

	/**
	 * 
	 * TODO
	 * @param actionId
	 * @param allowCreateEmpty
	 * @return
	 */
	protected Action getAction(String actionId, boolean allowCreateEmpty) {
		return registered.get(getProvider(actionId)).getAction(
				getActionKey(actionId), allowCreateEmpty);
	}

	/**
	 * @param source the source object to un-register.
	 */
	public void unregister(ActionProvider source) {
//		throw new UnsupportedOperationException();
//		if (this.registered.containsKey(source.getClass().getName())) {
//			this.registered.remove(source.getClass().getName());
//			rebuildAll();
//		}
	}

	/**
	 * Rebuilds the {@link JMenuBar}.
	 */
	public void rebuildMenuBar() {
		final JXMenuBar menuBar = this.desktop.getMenuBar();
		final MenuModel model = menuBar.getModel();
		model.clear();

		for (ActionProvider provider : this.registered.values()) {
			menuBar.register(provider);

			final MenuBar menuBarAnnotation = provider.getClass().getAnnotation(MenuBar.class);
			if (menuBarAnnotation != null) {
				if (!menuBarAnnotation.id().equals("desktop.main"))
					throw new UnsupportedOperationException("Only 'desktop.main' menu type supported at this time!");
				throw new UnsupportedOperationException();
				//				for (Menu menu : menuBarAnnotation.value()) {
				//					model.add(menu.id(), provider.getClass().getName() + "." + menu.id());
				//				}
			}

			final MenuActions actions = provider.getClass().getAnnotation(MenuActions.class);
			if (actions != null) {
				for (MenuAction action : actions.value()) {
					if (!action.path().startsWith("desktop.main"))
						throw new UnsupportedOperationException("Only 'desktop.main' menu supported at this time! [" + action.path() + "]");
					if (action.path().startsWith("desktop.main/")) {
						model.add(action.actionId(), action.path().replace("desktop.main/", ""));
					} else {
						model.add(action.actionId(), "");
					}
				}
			}
		}

		//		rebuildPerspectiveMenu(menuBar);
		//		rebuildViewMenu(menuBar);
	}

	/**
	 * Rebuilds the {@link JToolBar}.
	 */
	public void rebuildToolBar() {
		final JToolBar toolBar = this.desktop.getToolBar();
		toolBar.removeAll();
		SwingUtil.populate(toolBar, null,
				this.desktop.getAction("newWizard", false));
	}

	/**
	 * Rebuilds the view menu.
	 */
	@SuppressWarnings("unused")
	private void rebuildViewMenu(JXMenuBar menuBar) {
		final javax.swing.Action action = this.desktop.getAction("showView", true);
		final JMenu menu = SwingUtil.findMenu(menuBar, action);

		if (menu == null)
			return;

		menu.removeAll();

		final List<ViewAction> viewActions = new ArrayList<ViewAction>();

		for (Entry<ServiceReference, DesktopService<?>> entry : 
			this.desktop.getActivator().getServiceManager().getAll().entrySet())
		{
			for (Class<? extends AbstractView> viewType : entry.getValue().getViews()) {
				try {
					if (!viewType.isAssignableFrom(AbstractEditorView.class)) {
						viewActions.add(new ViewAction(viewType, this.desktop));
					}
				} catch (IntrospectionException ie) {
					throw new DesktopRuntimeException(String.format(
							"Unable to create ViewAction for type! [%s]",
							viewType));
				}
			}
		}

		Collections.sort(viewActions);

		for (ViewAction viewAction : viewActions) {
			menu.add(viewAction);
		}
	}

	/**
	 * Builds the perspectives menu from available perspectives.
	 */
	@SuppressWarnings("unused")
	private void rebuildPerspectiveMenu(JMenuBar menuBar) {
		final javax.swing.Action action =
			this.desktop.getAction("openPerspective", false);
		final JMenu menu = SwingUtil.findMenu(menuBar, action);

		if (menu == null) {
			return;
		}

		// XXX Yuk!
		//		while (menu.getItemCount() > 2) {
		//			menu.remove(0);
		//		}

		final Map<String, Perspective> perspectives =
			this.desktop.getViewManager().getAllPerspectives();

		final List<PerspectiveAction> perspectiveActions = new ArrayList<PerspectiveAction>();

		for (Entry<String, Perspective> entry : perspectives.entrySet()) {
			perspectiveActions.add(new PerspectiveAction(entry.getValue(), this.desktop.getViewManager()));
		}

		Collections.sort(perspectiveActions);
		Collections.reverse(perspectiveActions);

		for (PerspectiveAction perspectiveAction : perspectiveActions) {
			menu.insert(perspectiveAction, 0);
		}
	}


	// --- Static Methods ---

	/**
	 * Returns the action provider from the {@code actionId}. This is the
	 * string prior to the last full stop.
	 *
	 * @param actionId
	 * @return
	 */
	private static String getProvider(String actionId) {
		final int index = actionId.lastIndexOf('.');
		return actionId.substring(0, index);
	}

	/**
	 * Returns the action key from the {@code actionId}. This is the token
	 * following the last full stop.
	 *  
	 * @param actionId
	 * @return
	 */
	private static String getActionKey(String actionId) {
		final int index = actionId.lastIndexOf('.');
		return actionId.substring(index + 1, actionId.length());
	}


	// --- Inner Classes ---

	/**
	 * Handles changes to the selected view.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [14 Jun 2010]
	 */
	private class SelectedHandler implements PropertyChangeListener {
		@Override
		public void propertyChange(PropertyChangeEvent evt) {
			
			if (evt.getOldValue() != null) {
				unregister((AbstractView) evt.getOldValue());
			}

			if (evt.getNewValue() != null) {
				register((AbstractView) evt.getNewValue());
			}
		}
	}


	/**
	 * Handles changes to the available {@link DesktopService}s.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [14 Jun 2010]
	 */
	private class DesktopServiceHandler implements DesktopServiceListener {
		@Override
		public void serviceChanged(DesktopServiceEvent event) {
			switch (event.getType()) {
			case ADDED:
				register(event.getService());
				break;
			case REMOVED:
				unregister(event.getService());
				break;
			}
		}
	}
}
