package civvi.osgi.desktop;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.event.EventListenerList;

import org.osgi.framework.BundleContext;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.util.tracker.ServiceTracker;
import org.osgi.util.tracker.ServiceTrackerCustomizer;

import civvi.osgi.desktop.DesktopServiceEvent.Type;
import civvi.osgi.desktop.internal.Activator;
import civvi.osgi.desktop.swingx.docking.AbstractDockable;
import civvi.osgi.desktop.view.AbstractView;
import civvi.osgi.desktop.view.AbstractViewBeanInfoSupport;

/**
 * Manages instances of {@link DesktopService}.
 * 
 * @author <a href="mailto=dansiviter@gmail.com">Daniel Siviter</a>
 * @since 3rd Dec 2007
 */
public class DesktopServiceManager implements ServiceTrackerCustomizer {
	private final Logger log = Logger.getLogger(DesktopServiceManager.class.getName());
	private final Map<ServiceReference, DesktopService<?>> serviceRefMap =
		new HashMap<ServiceReference, DesktopService<?>>();
	private final EventListenerList listeners = new EventListenerList();
	private final Activator activator;
	private final ServiceTracker serviceTracker;

	/**
	 * Constructs a new manager for the given {@code activator}.
	 * 
	 * @param activator
	 * @throws InvalidSyntaxException 
	 */
	public DesktopServiceManager(Activator activator)
	throws InvalidSyntaxException
	{
		this.activator = activator;
		
		final ServiceReference[] refs;
		synchronized (this) {
			this.serviceTracker = new ServiceTracker(
					this.activator.getContext(),
					DesktopService.class.getName(),
					this);

			this.serviceTracker.open();
			refs = this.serviceTracker.getServiceReferences();	
		}

		for (ServiceReference ref : refs) {
			if (this.log.isLoggable(Level.INFO))
				this.log.info(String.format(
						"Adding new service reference. [%s]",
						ref));
			final DesktopService<?> service = get(ref);
			fire(service, ref, Type.ADDED);
		}
	}

	/**
	 * Returns a instance of {@link DesktopService} for the given
	 * {@code ServiceReference}. If the service isn't currently cached then
	 * an instance is got from the {@link BundleContext}. 
	 * 
	 * @param ref the unique desktop key.
	 * @return the instance of the service.
	 */
	public DesktopService<?> get(ServiceReference ref) {
		DesktopService<?> service = this.serviceRefMap.get(ref);
		if (service == null) {
			service = (DesktopService<?>) this.activator.getContext().getService(ref);
			this.serviceRefMap.put(ref, service);
		}
		return service;
	}

	/**
	 * 
	 * TODO
	 *
	 * @param key
	 * @return
	 */
	public Class<? extends AbstractView> getViewType(Object key) {
		if (key == null) {
			throw new NullPointerException("Key is null! Have you defined a view key in the BeanInfo?");
		}

		final List<Class<? extends AbstractView>> beanTypes =
			new ArrayList<Class<? extends AbstractView>>();
		for (DesktopService<?> service : getAll().values()) {
			for (Class<? extends AbstractView> beanType : service.getViews()) {
				try {
					final BeanInfo beanInfo = Introspector.getBeanInfo(beanType);
					if (beanInfo == null || beanInfo.getBeanDescriptor() == null) {
						throw new NullPointerException("Unable to locate BeanInfo for type! [" + beanType + "]");
					}
					if (key.equals(beanInfo.getBeanDescriptor().getValue(AbstractViewBeanInfoSupport.VIEW_KEY))) {
						beanTypes.add(beanType);
					}
				} catch (IntrospectionException ie) {
					log.warning(String.format(
							"Unable to locate BeanInfo! [beanType=%s]", ie, beanType));
				}
			}
		}

		if (beanTypes.size() > 1) {
			throw new IllegalStateException(String.format(
					"More than one view type exists for key! [key=%s,types=%s]",
					key,
					beanTypes));
		}
		
		return beanTypes.isEmpty() ? null : beanTypes.get(0);
	}

	/**
	 * @return all known {@link DesktopService}s.
	 */
	public Map<ServiceReference, DesktopService<?>> getAll() {
		return Collections.unmodifiableMap(this.serviceRefMap);
	}

	@Override
	public synchronized Object addingService(ServiceReference ref) {
		final DesktopService<?> service = get(ref);

		if (this.log.isLoggable(Level.INFO))
			this.log.info(String.format(
					"Adding new desktop service. [%s]",
					service.getClass()));
		fire(service, ref, Type.ADDED);
		return service;
	}

	@Override
	public void modifiedService(ServiceReference reference, Object service) {
		System.out.println("Modifing service: " + reference);
	}

	@Override
	public synchronized void removedService(ServiceReference ref, Object service) {
		if (this.log.isLoggable(Level.INFO))
			this.log.info(String.format(
					"Removing desktop service. [%s]",
					service.getClass()));
		serviceRefMap.remove(ref);
		this.activator.getContext().ungetService(ref);
		fire((DesktopService<?>) service, ref, Type.REMOVED);
	}

	/**
	 * @param l the listener to add.
	 */
	public void addListener(DesktopServiceListener l) {
		this.listeners.add(DesktopServiceListener.class, l);
	}

	/**
	 * @param l the listener to remove.
	 */
	public void removeListener(DesktopServiceListener l) {
		this.listeners.remove(DesktopServiceListener.class, l);
	}

	/**
	 * 
	 * TODO
	 *
	 * @param line
	 * @param type
	 */
	protected void fire(
			DesktopService<?> service,
			ServiceReference ref,
			Type type)
	{
		final DesktopServiceEvent event = new DesktopServiceEvent(
				this, service, ref, type);
		final DesktopServiceListener[] listeners =
			this.listeners.getListeners(DesktopServiceListener.class);
		for (int i = listeners.length - 1; i >= 0; i--) {
			listeners[i].serviceChanged(event);
		}
	}

	/**
	 * Returns the service given by the class.
	 * 
	 * @param <T>
	 * @param cls
	 * @return
	 */
	public <T extends DesktopService<?>> T get(Class<T> cls) {
		for (DesktopService<?> service : getAll().values()) {
			if (service.getClass() == cls) {
				return cls.cast(service);
			}
		}
		return null;
	}

	/**
	 * 
	 * TODO
	 * @param cls
	 * @return
	 */
	public DesktopService<?> getServiceForView(
			Class<? extends AbstractDockable> cls)
	{
		for (DesktopService<?> service : getAll().values()) {
			for (Class<? extends AbstractDockable> dockable : service.getViews()) {
				if (dockable == cls) {
					return service;
				}
			}
		}
		return null;
	}
}
