package civvi.osgi.desktop;

import java.awt.Toolkit;
import java.util.EventObject;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.prefs.Preferences;

import javax.swing.JOptionPane;

import org.jdesktop.application.Application;
import org.jdesktop.application.ExitListener;
import org.jdesktop.application.TaskService;

import civvi.osgi.desktop.preferences.DefaultPreference;
import civvi.osgi.desktop.swingx.AppFrameworkSupport;

/**
 * Handles exit events from the application.
 * 
 * @author <a href="mailto=dansiviter@gmail.com">Daniel Siviter</a>
 * @since 24th July 2008
 */
public class ExitHandler implements ExitListener {
	private final Logger log = Logger.getLogger(getClass().getName());
	private final AppFrameworkSupport<OSGiDesktop> support;
	private final OSGiDesktop desktop;

	/**
	 * Creates a new exit handler.
	 * 
	 * @param desktop
	 */
	public ExitHandler(OSGiDesktop desktop) {
		this.support = new AppFrameworkSupport<OSGiDesktop>(this);
		this.desktop = desktop;
	}

	@Override
	public boolean canExit(EventObject e) {
		final DesktopServiceManager manager = this.desktop.getActivator().getServiceManager(); 
		final Preferences prefs = manager.get(DefaultService.class).getPreferences();

		if (!prefs.getBoolean(DefaultPreference.CONFIRM_ON_EXIT.name(), false)) {
			return true;
		}

		Toolkit.getDefaultToolkit().beep();
		final int option = JOptionPane.showConfirmDialog(
				this.desktop.getMainTopLevel(),
				this.support.getString("message"),
				this.support.getString("title"),
				JOptionPane.OK_CANCEL_OPTION);
		return option == JOptionPane.YES_OPTION;
	}

	@Override
	public void willExit(EventObject e) {
		AppFrameworkSupport.fireAction(this.desktop, "savePerspective");
		final TaskService taskService = Application.getContext().getTaskService();
		try {
			// XXX this currently blocks the EDT, so we need a modal dialog to
			// show if the #awaitTermination takes too long
			taskService.shutdown();
			taskService.awaitTermination(1, TimeUnit.HOURS);
		} catch (InterruptedException ie) {
			// not really much we can do!
			if (this.log.isLoggable(Level.FINE)) {
				this.log.log(Level.FINE, "Not really much we can do!", ie);
			}
		}
	}
}
