package civvi.osgi.desktop;

import java.awt.Component;
import java.awt.Dimension;
import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.JOptionPane;
import javax.swing.JToolBar;

import org.jdesktop.application.Action;
import org.jdesktop.application.LocalStorage;
import org.jdesktop.application.SingleFrameApplication;
import org.jdesktop.application.Task;
import org.jdesktop.observablecollections.ObservableCollections;
import org.jdesktop.observablecollections.ObservableList;
import org.jdesktop.swingx.error.ErrorInfo;

import civvi.osgi.desktop.annotation.MenuAction;
import civvi.osgi.desktop.annotation.MenuActions;
import civvi.osgi.desktop.annotation.ToolbarAction;
import civvi.osgi.desktop.annotation.ToolbarActions;
import civvi.osgi.desktop.component.about.AboutPanel;
import civvi.osgi.desktop.component.view.OtherViewPane;
import civvi.osgi.desktop.help.HelpProviderManager;
import civvi.osgi.desktop.internal.Activator;
import civvi.osgi.desktop.perspective.DesktopPerspectiveFactory;
import civvi.osgi.desktop.preferences.PreferencesEditor;
import civvi.osgi.desktop.swingx.AbstractTask;
import civvi.osgi.desktop.swingx.ActionProvider;
import civvi.osgi.desktop.swingx.AppFrameworkSupport;
import civvi.osgi.desktop.swingx.JXDockingPane;
import civvi.osgi.desktop.swingx.JXMenuBar;
import civvi.osgi.desktop.swingx.JXWizard;
import civvi.osgi.desktop.swingx.SwingExceptionHandler;
import civvi.osgi.desktop.swingx.SwingUtil;
import civvi.osgi.desktop.view.AbstractViewBeanInfoSupport;
import civvi.osgi.desktop.view.DesktopDockFactory;
import civvi.osgi.desktop.view.DesktopViewFactory;
import civvi.osgi.desktop.wizard.neweditor.NewEditableModel;
import civvi.osgi.desktop.wizard.neweditor.NewEditablePanelController;

/**
 * Root class that launches into the OSGi Swing Desktop application.
 *
 * @author <a href="mailto=dansiviter@gmail.com">Daniel Siviter</a>
 * @since v1.0.0 [24th July 2008]
 */
@MenuActions({
	@MenuAction(
			path = "desktop.main",
			actionId = "civvi.osgi.desktop.OSGiDesktop.file"),
	@MenuAction(
			path = "desktop.main/file?section=new",
			actionId = "civvi.osgi.desktop.OSGiDesktop.newWizard"),
	@MenuAction(
			path = "desktop.main/file?section=preferences",
			actionId = "civvi.osgi.desktop.OSGiDesktop.showPreferences"),
	@MenuAction(
			path = "desktop.main/file?section=quit",
			actionId = "civvi.osgi.desktop.OSGiDesktop.quit"),
	@MenuAction(
			path = "desktop.main",
			actionId = "civvi.osgi.desktop.OSGiDesktop.window"),
	@MenuAction(
			path = "desktop.main/window?section=showView",
			actionId = "civvi.osgi.desktop.OSGiDesktop.showView"),
	@MenuAction(
			path = "desktop.main/window/showView?section=otherView",
			actionId = "civvi.osgi.desktop.OSGiDesktop.otherView"),
	@MenuAction(
			path = "desktop.main/window?section=perspective",
			actionId = "civvi.osgi.desktop.OSGiDesktop.openPerspective"),
	@MenuAction(
			path = "desktop.main/window/openPerspective?section=otherPerspective",
			actionId = "civvi.osgi.desktop.OSGiDesktop.otherPerspective"),
	@MenuAction(
			path = "desktop.main/window?section=perspective",
			actionId = "civvi.osgi.desktop.OSGiDesktop.savePerspective"),
	@MenuAction(
			path = "desktop.main/window?section=statusBar",
			actionId = "civvi.osgi.desktop.OSGiDesktop.toggleStatusBar"),
	@MenuAction(
			path = "desktop.main",
			actionId = "civvi.osgi.desktop.OSGiDesktop.help"),
	@MenuAction(
			path = "desktop.main/help?section=help",
			actionId = "civvi.osgi.desktop.OSGiDesktop.showHelpIndex"),
	@MenuAction(
			path = "desktop.main/help?section=about",
			actionId = "civvi.osgi.desktop.OSGiDesktop.showAbout")
})
@ToolbarActions({
	@ToolbarAction(actionId = "civvi.osgi.desktop.DefaultService.newWizard")
})
public class OSGiDesktop extends SingleFrameApplication
implements ActionProvider
{
	private static final String PERSPECTIVES_XML = "perspectives.xml";

	private final Logger log = Logger.getLogger(getClass().getName());
	private final AppFrameworkSupport<OSGiDesktop> support;
	private final Activator activator;

	private JXMenuBar menuBar;
	private JToolBar toolBar;
	private JXDockingPane dockingPane;
	private StatusBar statusBar;

	private ViewManager viewManager;
	private DesktopViewFactory viewFactory;
	private boolean statusBarVisible = true;

	private final ObservableList<ErrorInfo> errorList =
		ObservableCollections.observableList(new ArrayList<ErrorInfo>());

	/**
	 * 
	 * TODO
	 * @param activator
	 */
	public OSGiDesktop(Activator activator) {
		this.activator = activator;
		this.support = new AppFrameworkSupport<OSGiDesktop>(this);
	}

	/**
	 * @return the root dock.
	 */
	JXDockingPane getDockingPane() {
		return this.dockingPane;
	}

	/**
	 * @return the menu bar.
	 */
	JXMenuBar getMenuBar() {
		return menuBar;
	}

	/**
	 * @return the tool bar.
	 */
	JToolBar getToolBar() {
		return toolBar;
	}

	@Override
	public javax.swing.Action getAction(
			String actionKey,
			boolean allowCreateEmpty)
	{
		return this.support.getAction(actionKey, allowCreateEmpty);
	}

	/**
	 * @return the showStatusBar
	 */
	public boolean isStatusBarVisible() {
		return this.statusBarVisible;
	}

	/**
	 * @param showStatusBar the showStatusBar to set
	 */
	public void setStatusBarVisible(boolean statusBarVisible) {
		final boolean oldValue = isStatusBarVisible();
		this.statusBarVisible = statusBarVisible;
		firePropertyChange("statusBarVisible", oldValue, isStatusBarVisible());
	}
	
	
	/**
	 * @return the activator.
	 */
	public Activator getActivator() {
		return activator;
	}

	/**
	 * @return the view manager.
	 */
	public ViewManager getViewManager() {
		return viewManager;
	}

	/**
	 * 
	 * TODO
	 * @return
	 */
	public ObservableList<ErrorInfo> getErrorList() {
		return errorList;
	}


	// --- Actions ---

	/**
	 * Displays the new wizard.
	 */
	@Action
	public void newWizard() {
		final NewEditableModel model = JXWizard.show(
				getMainTopLevel(),
				new NewEditableModel(this.activator.getContext()),
				new NewEditablePanelController(),
				this.support.getString("newWizard.title"));

		if (model != null) {
			// XXX do something?
		}
	}

	/**
	 * Displays the preferences editor.
	 */
	@Action
	public void showPreferences() {
		SwingUtil.showEditorDialog(
				getMainTopLevel(),
				new PreferencesEditor(this),
				this.support.getString("preferences.title"));
	}

	/**
	 * Saves the layout.
	 */
	@Action
	public Task<Object, Object> savePerspective() {
		return new SavePerspectiveTask();
	}

	/**
	 * Displays the other views dialog.
	 * 
	 * @throws IntrospectionException 
	 */
	@Action
	public void otherView() throws IntrospectionException {
		final OtherViewPane otherViewPane = new OtherViewPane(this);
		final int result = JOptionPane.showConfirmDialog(
				getMainTopLevel(),
				otherViewPane,
				"Select View",
				JOptionPane.OK_CANCEL_OPTION,
				JOptionPane.PLAIN_MESSAGE);
		
		if (result == JOptionPane.OK_OPTION) {
			final BeanInfo beanInfo = Introspector.getBeanInfo(otherViewPane.getSelected());
			final Object viewKey = beanInfo.getBeanDescriptor().getValue(
					AbstractViewBeanInfoSupport.VIEW_KEY);
			getViewManager().show(viewKey, true);
		}
	}

	/**
	 * Displays the other perspectives dialog.
	 */
	@Action
	public void otherPerspective() {
		throw new UnsupportedOperationException();
	}

	/**
	 * Show help index action.
	 */
	@Action
	public void showHelpIndex() {
		HelpProviderManager.show("help:index");
	}

	/**
	 * About action.
	 */
	@Action
	public void showAbout() {
		SwingUtil.showEditorDialog(
				getMainTopLevel(),
				new AboutPanel(this),
				this.support.getString("about.title"));
	}

	/**
	 * Toggle the visibility of the status bar.
	 */
	@Action(selectedProperty = "statusBarVisible")
	public void toggleStatusBar() {
		this.statusBar.setVisible(isStatusBarVisible());
	}


	// --- Overridden/Implemented Methods ---

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void initialize(String[] args) {
		// setup default and GUI exception handling, errors should never get
		// this far, but if they should, we'll be here to intercept them!
		Thread.setDefaultUncaughtExceptionHandler(new SwingExceptionHandler());
		super.initialize(args);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void startup() {
		addExitListener(new ExitHandler(this));
		this.viewManager.setPerspectiveKey(ViewManager.DEFAULT_PERSPECTIVE);
		setStatusBarVisible(true);
		new ActionManager(this);
	}

	@Override
	protected JXMenuBar createJMenuBar() {
		return this.menuBar = new JXMenuBar();
	}

	@Override
	protected JToolBar createJToolBar() {
		return this.toolBar = new JToolBar();
	}

	@Override
	protected Component createStatusBar() {
		this.statusBar = new StatusBar(getContext().getTaskMonitor());
//		setStatusBarVisible(true);
		return this.statusBar;
	}

	@Override
	protected Component createMainComponent() {
		this.dockingPane = new JXDockingPane();
		this.dockingPane.setDockFactory(new DesktopDockFactory());
		this.dockingPane.setPreferredSize(new Dimension(640, 480));

		this.viewManager = new ViewManager(this);
		this.viewFactory = new DesktopViewFactory(this);
		this.viewManager.setDockableFactory(this.viewFactory);
		this.viewManager.setPerspectiveFactory(new DesktopPerspectiveFactory());

		try {
			final LocalStorage localStorage = getContext().getLocalStorage();
			final InputStream is = localStorage.openInputFile(PERSPECTIVES_XML);
			this.viewManager.loadPerspectives(is);
			is.close();
		} catch (IOException ioe) {
			if (this.log.isLoggable(Level.FINE)) {
				this.log.log(Level.FINE, "Unable to load perspectives!", ioe);
			}
			// revert to default
		}

		return this.dockingPane;
	}

	@Override
	protected void ready() {
		throw new UnsupportedOperationException("Why is this not called?!");
	}


	// --- Inner Classes ---

	/**
	 * Persists the perspective to local storage.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [24th July 2008]
	 */
	private static class SavePerspectiveTask extends AbstractTask<Object, Object> {
		private final Logger log = Logger.getLogger(getClass().getName());

		@Override
		protected Object doInBackground() throws Exception {
			final ViewManager viewManager = ((OSGiDesktop) getApplication()).getViewManager();
			try {
				// persist perspective
				viewManager.updatePerspectives();
				final LocalStorage localStorage = getContext().getLocalStorage();
				final OutputStream os = localStorage.openOutputFile(PERSPECTIVES_XML);
				viewManager.savePerspectives(os);
				os.close();
			} catch (IOException ioe) {
				this.log.log(Level.WARNING, "Unable to save perspectives!", ioe);
			} catch (RuntimeException re) {
				this.log.log(Level.WARNING, "Unable to save perspectives!", re);
			}
			return null;
		}
	}
}
