package civvi.osgi.desktop;

import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;

/**
 * This class is used as a proxy to defer object creation from shape provider
 * bundles and also as a place-holder shape when previously used shapes are no
 * longer available. These two purposes are actually orthogonal, but were
 * combined into a single class to reduce the number of classes in the
 * application. The proxy-related functionality is introduced as a way to
 * lazily create shape objects in an effort to improve performance; this level
 * of indirection could be removed if eager creation of objects is not a
 * concern.
 * <p/>
 * <strong>Note:</strong> This is only useful if performed if the proxy object
 * isn't created yet.
 * 
 * @author <a href="mailto=dansiviter@gmail.com">Daniel Siviter</a>
 * @since 1st September 2008
 */
public abstract class ProxyService<T>  {
	protected final BundleContext context;
	protected final ServiceReference ref;
	protected final Class<T> proxyType;
	private T proxy;

	/**
	 * This constructs a proxy {@link AbstractDesktopService} that lazily gets
	 * the internal service.
	 * 
	 * @param context the bundle context to use for retrieving the service.
	 * @param ref the service reference of the service.
	 */
	public ProxyService(BundleContext context, ServiceReference ref, Class<T> proxyType) {
		this.context = context;
		this.ref = ref;
		this.proxyType = proxyType;
	}

	/**
	 * This method tells the proxy to dispose of its service object; this
	 * is called when the underlying service goes away.
	 */
	public void dispose() {
		if (this.proxy != null) {
			this.context.ungetService(this.ref);
			this.proxy = null;
		}
	}

	/**
	 * Returns the underlying service.
	 * 
	 * @return
	 */
	protected T getProxy() {
		// double-checked locking
		if (this.proxy == null) {
			synchronized (this.context) {
				if (this.proxy == null) {
					this.proxy = this.proxyType.cast(
							this.context.getService(this.ref));
				}
			}
		}

		return this.proxy;
	}
}
