package civvi.osgi.desktop;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.Icon;
import javax.swing.JLabel;
import javax.swing.JProgressBar;
import javax.swing.Timer;

import org.jdesktop.application.Application;
import org.jdesktop.application.ApplicationContext;
import org.jdesktop.application.ResourceMap;
import org.jdesktop.application.TaskMonitor;
import org.jdesktop.swingx.JXStatusBar;

import civvi.osgi.desktop.swingx.JXHeapView;

/**
 * A StatusBar panel that tracks a {@link TaskMonitor}.
 * <br>
 * This class loads resources from the ResourceBundle called
 * {@code resources.StatusBar}.
 * 
 * @author <a href="mailto=dansiviter@gmail.com">Daniel Siviter</a>
 * @since v1.0.0 [22th November 2007]
 */
public class StatusBar extends JXStatusBar {
	private static final long serialVersionUID = -1121468158458685087L;

	private final JLabel messageLabel;
	private final JProgressBar progressBar;
	private final JLabel iconLabel;
	private final Timer messageTimer;
	private final Timer iconTimer;
	private final Icon idleIcon;
	private final Icon[] busyIcons;
	private final int busyAnimationRate;
	private int iconIndex = 0;


	/**
	 * Constructs a panel that displays messages/progress/state 
	 * properties of the {@code taskMonitor's} foreground task.
	 * 
	 * @param taskMonitor the {@code TaskMonitor} whose 
	 *     {@code PropertyChangeEvents} {@code this StatusBar} will track.
	 */
	public StatusBar(TaskMonitor taskMonitor) {
		super();
		this.messageLabel = new JLabel();
		this.progressBar = new JProgressBar(0, 100);
		this.iconLabel = new JLabel();

		final ApplicationContext appContext = Application.getContext();
		final ResourceMap resourceMap =
			appContext.getResourceMap(StatusBar.class);
		this.messageTimer = new Timer(
				resourceMap.getInteger("messageTimeout"),
				new ClearMessageHandler()); 
		this.messageTimer.setRepeats(false);
		this.busyAnimationRate = resourceMap.getInteger("busyAnimationRate");
		this.idleIcon = resourceMap.getIcon("idleIcon");

		this.busyIcons = new Icon[15];
		for (int i = 0; i < this.busyIcons.length; i++) {
			this.busyIcons[i] = resourceMap.getIcon("busyIcons[" + i + "]");
		}
		this.iconTimer = new Timer(this.busyAnimationRate, new UpdateBusyIconHandler()); 
		this.progressBar.setEnabled(false);
		this.iconLabel.setIcon(this.idleIcon);

		add(this.messageLabel, new Constraint(Constraint.ResizeBehavior.FILL));
		add(new JXHeapView(), new Constraint(100));
		add(this.progressBar, new Constraint(100));
		add(this.iconLabel, new Constraint(Constraint.ResizeBehavior.FIXED));

		taskMonitor.addPropertyChangeListener(new PropertyChangeHandler());
	}

	/**
	 * @param msg the message to display.
	 */
	public void setMessage(String msg) {
		this.messageLabel.setText(msg == null ? "" : msg);
		this.messageTimer.restart();
	}

	/**
	 * Starts the icon animation.
	 */
	private void startIconAnimation() {
		if (!this.iconTimer.isRunning()) {
			this.iconLabel.setIcon(this.busyIcons[0]);
			this.iconIndex = 0;
			this.iconTimer.start();
		}
	}

	/**
	 * Stops the icon animation.
	 */
	private void stopIconAnimation() {
		this.iconTimer.stop();
		this.iconLabel.setIcon(this.idleIcon);
	}


	// --- Inner Classes ---
	
	/**
	 * Handles clearance of the task.
	 * <p/>
	 * The TaskMonitor (constructor arg) tracks a "foreground" task;
	 * this method is called each time a foreground task property
	 * changes.
	 *
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since 22 Nov 2007
	 */
	private class PropertyChangeHandler implements PropertyChangeListener { 
		/**
		 * {@inheritDoc}
		 */
		@Override
		public void propertyChange(PropertyChangeEvent e) {
			String propertyName = e.getPropertyName();
			if ("started".equals(propertyName)) {
				startIconAnimation();
				progressBar.setEnabled(true);
				progressBar.setIndeterminate(true);
			}
			else if ("done".equals(propertyName)) {
				stopIconAnimation();
				progressBar.setIndeterminate(false);
				progressBar.setEnabled(false);
				progressBar.setValue(0);
			}
			else if ("message".equals(propertyName)) {
				String text = (String)(e.getNewValue());
				setMessage(text);
			}
			else if ("progress".equals(propertyName)) {
				int value = (Integer)(e.getNewValue());
				progressBar.setEnabled(true);
				progressBar.setIndeterminate(false);
				progressBar.setValue(value);
			}
		}
	}

	/**
	 * Handles clearance of the message label.
	 *
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since 22 Nov 2007
	 */
	private class ClearMessageHandler implements ActionListener {
		/**
		 * {@inheritDoc}
		 */
		public void actionPerformed(ActionEvent e) {
			StatusBar.this.messageLabel.setText("");
		}
	}

	/**
	 * Handles updates to the status icon.
	 *
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since 22 Nov 2007
	 */
	private class UpdateBusyIconHandler implements ActionListener {
		/**
		 * {@inheritDoc}
		 */
		public void actionPerformed(ActionEvent e) {
			StatusBar.this.iconIndex =
				(StatusBar.this.iconIndex + 1) % StatusBar.this.busyIcons.length;
			StatusBar.this.iconLabel.setIcon(
					StatusBar.this.busyIcons[StatusBar.this.iconIndex]);
		}
	}
}
