package civvi.osgi.desktop;

import java.awt.event.ActionEvent;
import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.util.logging.Level;

import org.jdesktop.application.Action;
import org.jdesktop.application.Task;
import org.jdesktop.application.Task.BlockingScope;

import civvi.osgi.desktop.swingx.AbstractTask;
import civvi.osgi.desktop.swingx.AppFrameworkSupport;
import civvi.osgi.desktop.view.AbstractView;
import civvi.osgi.desktop.view.AbstractViewBeanInfoSupport;

/**
 * An {@link javax.swing.Action} for displaying a view. This will block and load
 * the parent {@link DesktopService} for the view prior to displaying the view
 * in a EDT friendly way.
 * 
 * FIXME we need the threading to be every time we attempt to show a view, not
 * just invocation from this action. So probably in the {@link ViewManager}
 * class.
 * 
 * @author <a href="mailto=dansiviter@gmail.com">Daniel Siviter</a>
 * @since v1.0.0 [24th July 2008]
 */
public class ViewAction extends AbstractBeanInfoAction {
	private static final long serialVersionUID = 7952615618763659079L;

	private final AppFrameworkSupport<OSGiDesktop> support;
	private final Class<? extends AbstractView> type;
	private final OSGiDesktop desktop;
	private final BeanInfo beanInfo;


	/**
	 * 
	 * TODO
	 * @param type
	 * @param desktop
	 * @throws IntrospectionException 
	 */
	public ViewAction(
			Class<? extends AbstractView> type,
			OSGiDesktop desktop)
	throws IntrospectionException
	{
		super(type);
		this.support = new AppFrameworkSupport<OSGiDesktop>(this);
		this.type = type;
		this.desktop = desktop;
		this.beanInfo = Introspector.getBeanInfo(type);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void actionPerformed(ActionEvent e) {
		final Object viewKey = this.beanInfo.getBeanDescriptor().getValue(
				AbstractViewBeanInfoSupport.VIEW_KEY);
		this.desktop.getViewManager().show(viewKey, true);
		this.support.fireAction("show", e);
	}

	/**
	 * Performs the task of displaying the view, but only if it's parent
	 * service is initialised.
	 * 
	 * @param type
	 */
	@Action(block = BlockingScope.WINDOW)
	public Task<BeanInfo, Void> show() {
		return new ShowViewTask();
	}


	// --- Inner Classes ---

	/**
	 * A {@link Task} for loading the parent {@link DesktopService} for the
	 * view prior to displaying the view.
	 * 
	 *  FIXME we need the threading to be every time we attempt to show a view, not
	 * just invocation from this action. So probably in the {@link ViewManager}
	 * class.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [8 Jun 2010]
	 */
	private class ShowViewTask extends AbstractTask<BeanInfo, Void> {
		@Override
		protected BeanInfo doInBackground() throws Exception {
			final DesktopService<?> service = 
				desktop.getActivator().getServiceManager().getServiceForView(type);
			if (!service.isInitialised()) {
				if (this.log.isLoggable(Level.INFO)) {
					this.log.info(String.format(
							"Initialising service for view. [service=%s,type=%s]",
							service,
							type));
				}
				service.init();
			}
			return Introspector.getBeanInfo(type);
		}

		@Override
		protected void succeeded(BeanInfo result) {
			final Object viewKey = result.getBeanDescriptor().getValue(
					AbstractViewBeanInfoSupport.VIEW_KEY);
			desktop.getViewManager().show(viewKey, true);
			ViewAction.this.setEnabled(true);
		}

		@Override
		protected void failed(Throwable cause) {
			super.failed(cause);
			ViewAction.this.setEnabled(true);
		}
	}
}
