package civvi.osgi.desktop;

import org.jdesktop.application.Application;
import org.jdesktop.swingx.MultiSplitLayout;
import org.jdesktop.swingx.MultiSplitLayout.Node;

import civvi.osgi.desktop.editor.Editable;
import civvi.osgi.desktop.swingx.AbstractTask;
import civvi.osgi.desktop.swingx.docking.AbstractDockable;
import civvi.osgi.desktop.swingx.docking.DockManager;
import civvi.osgi.desktop.swingx.docking.perspective.Perspective;
import civvi.osgi.desktop.swingx.docking.perspective.PerspectiveLeaf;
import civvi.osgi.desktop.swingx.docking.perspective.PerspectiveView;
import civvi.osgi.desktop.view.AbstractEditorView;
import civvi.osgi.desktop.view.AbstractView;

/**
 * A OSGi Desktop implementation of {@link DockManager}.
 * 
 * @author <a href="mailto=dansiviter@gmail.com">Daniel Siviter</a>
 * @since v1.0.0 [22th November 2007]
 */
public class ViewManager extends DockManager {
	private final OSGiDesktop desktop;

	/**
	 * Constructs a {@link DockManager} for the given {@link OSGiDesktop}.
	 * 
	 * @param desktop
	 */
	public ViewManager(OSGiDesktop desktop) {
		super(desktop.getDockingPane());
		this.desktop = desktop;
	}

	/**
	 * Returns an {@link AbstractEditorView} for the given key.
	 *
	 * @param key
	 * @return
	 * @throws IllegalArgumentException if the key isn't an {@link Editable}
	 * instance.
	 */
	public AbstractEditorView getEditorView(Object key) {
		if (!(key instanceof Editable)) {
			throw new IllegalArgumentException(String.format(
					"Key is not an editable! [key=%s]",
					key));
		}
		return (AbstractEditorView) getDockable(key, false);
	}

	@Override
	public AbstractView getDockable(Object key, boolean create) {
		return (AbstractView) super.getDockable(key, create);
	}

	/**
	 * Overriden to ensure that parent {@link DesktopService} is initialised
	 * prior to showing the view. 
	 * 
	 * TODO We need to move the initialisation step from {@link ViewAction} in
	 * to here.
	 */
	@Override
	public void dock(AbstractDockable dockable, String position) {
		final AbstractView view = (AbstractView) dockable;

		final DesktopService<?> service = 
			this.desktop.getActivator().getServiceManager().getServiceForView(view.getClass());
		if (!service.isInitialised()) {
			Application.getContext().getTaskService().execute(new ShowViewTask(view, position));
		} else {
			super.dock(dockable, position);
		}
	}

	@Override
	protected void doChangePerspective(String key) {
		final Perspective perspective = getPerspective(key);
		
		// update model
		final Node model = MultiSplitLayout.parseModel(
				perspective.getLayout());
		model.setParent(null); // annoying bug where is always contains a parent
		this.dockingPane.setModel(model);

		for (PerspectiveLeaf pl : perspective.getLeafs()) {
			for (PerspectiveView pv : pl.getViews()) {
				final AbstractDockable view = getDockable(pv.getId(), true);
				if (pv != null) {
					dock(view, pl.getId());
				}
			}
		}
	}


	// --- Inner classes ---

	/**
	 * Performs initialisation of the associated {@link DesktopService} prior
	 * to showing the view.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [28 Jun 2010]
	 */
	private class ShowViewTask extends AbstractTask<DesktopService<?>, Void> {
		private final AbstractView view;
		private final String position;

		public ShowViewTask(AbstractView view, String position) {
			this.view = view;
			this.position = position;
		}
		
		@Override
		protected DesktopService<?> doInBackground() throws Exception {
			final DesktopServiceManager manager = 
				desktop.getActivator().getServiceManager();
			final DesktopService<?> service =
				manager.getServiceForView(this.view.getClass());
			service.init();
			return service;
		}
		
		@Override
		protected void succeeded(DesktopService<?> result) {
			// DesktopService should be initialised now, so recall #dock
			dock(this.view, this.position);
		}
	}
}
