package civvi.osgi.desktop.common;

import java.awt.Image;
import java.beans.BeanDescriptor;
import java.beans.BeanInfo;
import java.beans.EventSetDescriptor;
import java.beans.Introspector;
import java.beans.MethodDescriptor;
import java.beans.PropertyDescriptor;
import java.beans.SimpleBeanInfo;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.ImageIcon;

import civvi.osgi.desktop.OSGiDesktop;
import civvi.osgi.desktop.swingx.AppFrameworkSupport;

/**
 * 
 * TODO
 * 
 * @author <a href="mailto=dansiviter@gmail.com">Daniel Siviter</a>
 * @since 14 Apr 2009
 */
public abstract class BeanInfoSupport extends SimpleBeanInfo {
	private static Map<Class<?>, Boolean> introspectingState = new HashMap<Class<?>, Boolean>();

	private final Logger LOG = Logger.getLogger(getClass().getName());
	private final Map<Integer, Image> images = new HashMap<Integer, Image>();
	private final Map<String, PropertyDescriptor> properties =
		new TreeMap<String, PropertyDescriptor>();
	private final Map<String, EventSetDescriptor> events =
		new TreeMap<String, EventSetDescriptor>();
	private final Map<String, MethodDescriptor> methods =
		new TreeMap<String, MethodDescriptor>();
	private final Class<?> beanClass;
	protected final AppFrameworkSupport<OSGiDesktop> support;

	private int defaultPropertyIndex = -1;
	private int defaultEventIndex = -1;
	private BeanDescriptor beanDescriptor;


	/**
	 * Creates a new instance of BeanInfoSupport.
	 * 
	 * @param beanClass class of the bean.
	 */
	public BeanInfoSupport(Class<?> beanClass) {
		this.beanClass = beanClass;
		this.support = new AppFrameworkSupport<OSGiDesktop>(beanClass);
		if (!isIntrospecting()) {
			introspectingState.put(beanClass, Boolean.TRUE);
			try {
				Class<?> superClass = beanClass.getSuperclass();
				while (superClass != null) {
					Introspector.flushFromCaches(superClass);
					superClass = superClass.getSuperclass();
				}
				BeanInfo info = Introspector.getBeanInfo(beanClass);
				beanDescriptor = info.getBeanDescriptor();
				if (beanDescriptor != null) {
					Class<?> customizerClass = getCustomizerClass();
					beanDescriptor = new BeanDescriptor(beanDescriptor.getBeanClass(),
							customizerClass == null ? beanDescriptor.getCustomizerClass()
									: customizerClass);
				} else {
					beanDescriptor = new BeanDescriptor(beanClass, getCustomizerClass());
				}

				beanDescriptor.setDisplayName(this.support.getString("name"));
				beanDescriptor.setShortDescription(this.support.getString("description"));

				for (PropertyDescriptor pd : info.getPropertyDescriptors()) {
					properties.put(pd.getName(), pd);
				}
				for (EventSetDescriptor esd : info.getEventSetDescriptors()) {
					events.put(esd.getName(), esd);
				}
				for (MethodDescriptor md : info.getMethodDescriptors()) {
					methods.put(md.getName(), md);
				}

				defaultPropertyIndex = info.getDefaultPropertyIndex();
				defaultEventIndex = info.getDefaultEventIndex();
			} catch (Exception e) {
				e.printStackTrace();
			}
			introspectingState.put(beanClass, Boolean.FALSE);
			initialize();
		}
	}

	/**
	 * 
	 * TODO
	 *
	 * @return
	 */
	private boolean isIntrospecting() {
		Boolean b = introspectingState.get(beanClass);
		return b == null ? false : b.booleanValue();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Image getIcon(int size) {
		Image image = this.images.get(size);
		if (image == null) {
			String key;
			switch (size) {
			case BeanInfo.ICON_COLOR_16x16:
				key = "icon.small";
				break;
			case BeanInfo.ICON_COLOR_32x32:
				key = "icon.large";
				break;
			case BeanInfo.ICON_MONO_16x16:
				key = "icon.small.mono";
				break;
			case BeanInfo.ICON_MONO_32x32:
				key = "icon.large.mono";
				break;
			default:
				throw new IllegalArgumentException(String.format(
						"Unknown size! [size=%1$s]",
						size));
			}

			image = loadImage(key);

			if (image != null) {
				this.images.put(size, image);
			}
		}

		return image;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Image loadImage(final String resourceName) {
		ImageIcon imageIcon = this.support.getImageIcon(resourceName);

		if (imageIcon == null) {
			URL url = getClass().getResource(resourceName);
			imageIcon = url == null ? null : new ImageIcon(url);	
		}

		return imageIcon == null ? null : imageIcon.getImage();
	}

	/**
	 * Called by the constructor during the proper time so that subclasses
	 * can override the settings/values for the various beaninfo properties.
	 * For example, you could call setDisplayName("Foo Name", "foo") to change
	 * the foo properties display name
	 */
	protected void initialize() {

	}

	/**
	 * Override this method if you want to return a custom customizer class
	 * for the bean
	 * 
	 * @return <code>null</code>.
	 */
	protected Class<?> getCustomizerClass() {
		return null;
	}

	/**
	 * Changes the display name of the given named property. Property names
	 * are always listed last to allow for varargs
	 * 
	 * @param displayName display name of the property.
	 * @param propertyName name of the property.
	 */
	protected void setDisplayName(String displayName, String propertyName) {
		PropertyDescriptor pd = properties.get(propertyName);
		if (pd != null) {
			pd.setDisplayName(displayName);
		} else {
			LOG.log(Level.WARNING, "Failed to set display name for property '" +
					propertyName + "'. No such property was found");
		}
	}

	/**
	 * Sets the given named properties to be "hidden".
	 * 
	 * @param hidden determines whether the properties should be marked as hidden or not.
	 * @param propertyNames name of properties.
	 * @see PropertyDescriptor
	 */
	protected void setHidden(boolean hidden, String... propertyNames) {
		for (String propertyName : propertyNames) {
			PropertyDescriptor pd = properties.get(propertyName);
			if (pd != null) {
				pd.setHidden(hidden);
			} else {
				LOG.log(Level.WARNING, "Failed to set hidden attribute for property '" +
						propertyName + "'. No such property was found");
			}
		}
	}

	/**
	 * 
	 * TODO
	 *
	 * @param expert
	 * @param propertyNames
	 */
	protected void setExpert(boolean expert, String... propertyNames) {
		for (String propertyName : propertyNames) {
			PropertyDescriptor pd = properties.get(propertyName);
			if (pd != null) {
				pd.setExpert(expert);
			} else {
				LOG.log(Level.WARNING, "Failed to set expert attribute for property '" +
						propertyName + "'. No such property was found");
			}
		}
	}

	/**
	 * 
	 * TODO
	 *
	 * @param preferred
	 * @param propertyNames
	 */
	protected void setPreferred(boolean preferred, String... propertyNames) {
		for (String propertyName : propertyNames) {
			PropertyDescriptor pd = properties.get(propertyName);
			if (pd != null) {
				pd.setPreferred(preferred);
			} else {
				LOG.log(Level.WARNING, "Failed to set preferred attribute for property '" +
						propertyName + "'. No such property was found");
			}
		}
	}

	/**
	 * 
	 * TODO
	 *
	 * @param bound
	 * @param propertyNames
	 */
	protected void setBound(boolean bound, String... propertyNames) {
		for (String propertyName : propertyNames) {
			PropertyDescriptor pd = properties.get(propertyName);
			if (pd != null) {
				pd.setBound(bound);
			} else {
				LOG.log(Level.WARNING, "Failed to set bound attribute for property '" +
						propertyName + "'. No such property was found");
			}
		}
	}

	/**
	 * 
	 * TODO
	 *
	 * @param constrained
	 * @param propertyNames
	 */
	protected void setConstrained(boolean constrained, String... propertyNames) {
		for (String propertyName : propertyNames) {
			PropertyDescriptor pd = properties.get(propertyName);
			if (pd != null) {
				pd.setConstrained(constrained);
			} else {
				LOG.log(Level.WARNING, "Failed to set constrained attribute for property '" +
						propertyName + "'. No such property was found");
			}
		}
	}

	/**
	 * 
	 * TODO
	 *
	 * @param categoryName
	 * @param propertyNames
	 */
	protected void setCategory(String categoryName, String... propertyNames) {
		for (String propertyName : propertyNames) {
			PropertyDescriptor pd = properties.get(propertyName);
			if (pd != null) {
				pd.setValue("category", categoryName);
			} else {
				LOG.log(Level.WARNING, "Failed to set category for property '" +
						propertyName + "'. No such property was found");
			}
		}
	}

	/**
	 * 
	 * TODO
	 *
	 * @param editorClass
	 * @param propertyNames
	 */
	protected void setPropertyEditor(Class<?> editorClass, String... propertyNames) {
		for (String propertyName : propertyNames) {
			PropertyDescriptor pd = properties.get(propertyName);
			if (pd != null) {
				pd.setPropertyEditorClass(editorClass);
			} else {
				LOG.log(Level.WARNING, "Failed to set property editor for property '" +
						propertyName + "'. No such property was found");
			}
		}
	}

	/* *
	 * 
	 * TODO
	 *
	 * @param values
	 * @param propertyNames
	 * /
    protected void setEnumerationValues(EnumerationValue[] values, String... propertyNames) {
        if (values == null) {
            return;
        }

        Object[] enumValues = new Object[values.length * 3];
        int index = 0;
        for (EnumerationValue ev : values) {
            enumValues[index++] = ev.getName();
            enumValues[index++] = ev.getValue();
            enumValues[index++] = ev.getJavaInitializationString();
        }

        for (String propertyName : propertyNames) {
            PropertyDescriptor pd = properties.get(propertyName);
            if (pd != null) {
                pd.setValue("enumerationValues", enumValues);
            } else {
                LOG.log(Level.WARNING, "Failed to set enumeration values for property '" +
                        propertyName + "'. No such property was found");
            }
        }
    }
	 */
	/**
	 * {@inheritDoc}
	 */
	@Override
	public BeanDescriptor getBeanDescriptor() {
		return isIntrospecting() ? null : beanDescriptor;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public PropertyDescriptor[] getPropertyDescriptors() {
		return isIntrospecting() 
		? null
				: properties.values().toArray(new PropertyDescriptor[0]);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public EventSetDescriptor[] getEventSetDescriptors() {
		return isIntrospecting()
		? null
				: events.values().toArray(new EventSetDescriptor[0]);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public MethodDescriptor[] getMethodDescriptors() {
		return isIntrospecting()
		? null
				: methods.values().toArray(new MethodDescriptor[0]);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getDefaultPropertyIndex() {
		return isIntrospecting() ? -1 : defaultPropertyIndex;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getDefaultEventIndex() {
		return isIntrospecting() ? -1 : defaultEventIndex;
	}
}