package civvi.osgi.desktop.internal;

import java.lang.reflect.InvocationTargetException;
import java.util.Properties;
import java.util.logging.Logger;

import javax.swing.SwingUtilities;

import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.service.log.LogReaderService;
import org.osgi.service.log.LogService;

import civvi.osgi.desktop.DefaultService;
import civvi.osgi.desktop.DesktopService;
import civvi.osgi.desktop.DesktopServiceManager;
import civvi.osgi.desktop.OSGiDesktop;
import civvi.osgi.desktop.log.DesktopLogReaderService;
import civvi.osgi.desktop.log.DesktopLogService;


/**
 * Activator for the OSGiDesktop application.
 * 
 * @author <a href="mailto=dansiviter@gmail.com">Daniel Siviter</a>
 * @since 24th July 2008
 */
public class Activator implements BundleActivator {
	private final Logger log = Logger.getLogger(getClass().getName());

	private DesktopServiceManager serviceManager;
	private BundleContext context;
	private OSGiDesktop desktop;

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void start(final BundleContext context)
	throws InterruptedException, InvalidSyntaxException,
	InvocationTargetException
	{
		
		this.log.info("Starting the OSGiDesktop.");

		synchronized (this) {
			this.context = context;
			registerDefaultServices(context);
			// start monitoring DesktopServices
			this.serviceManager = new DesktopServiceManager(this);

			this.log.info("Starting OSGiDesktop GUI.");

			// launch the GUI
			SwingUtilities.invokeAndWait(new Runnable() {
				@Override
				public void run() {
					Activator.this.desktop = new OSGiDesktop(Activator.this);
					Activator.this.desktop.launch();
				}
			});
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void stop(BundleContext context)
	throws InvocationTargetException, InterruptedException
	{
		this.log.info("Stopping the OSGiDesktop.");
		synchronized (this) {
			this.context = null;
			SwingUtilities.invokeAndWait(new ExitRunnable(this.desktop));
		}
	}

	/**
	 * @return the context.
	 */
	public BundleContext getContext() {
		return this.context;
	}

	/**
	 * @return the service mananger.
	 */
	public DesktopServiceManager getServiceManager() {
		return this.serviceManager;
	}

	/**
	 * @return an instance of the desktop.
	 */
	public OSGiDesktop getDesktop() {
		return this.desktop;
	}


	// --- Static Methods ---

	/**
	 * @param context registers the services with this context.
	 */
	private static void registerDefaultServices(BundleContext context) {
		context.registerService(
				DesktopService.class.getName(),
				new DefaultService(context),
				new Properties());
		
		final DesktopLogReaderService readerService = new DesktopLogReaderService();
		context.registerService(
				LogReaderService.class.getName(),
				readerService,
				new Properties());
		context.registerService(
				DesktopLogReaderService.class.getName(),
				readerService,
				new Properties());
		
		final DesktopLogService logService = new DesktopLogService(context.getBundle(), readerService);
		context.registerService(
				LogService.class.getName(),
				logService,
				new Properties());
		context.registerService(
				DesktopLogService.class.getName(),
				logService,
				new Properties());
	}


	// --- Inner Classes ---

	/**
	 * A runnable to perform exit of the application.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since 24th July 2008
	 */
	private static class ExitRunnable implements Runnable {
		private final OSGiDesktop app;

		public ExitRunnable(OSGiDesktop app) {
			this.app = app;
		}

		@Override
		public void run() {
			// FIXME need to allow force exit this can currently be escaped
			// causing the OSGi stuff to get in a inconsistent state
			this.app.exit();
		}
	}
}
