package civvi.osgi.desktop.log;

import static javax.swing.SwingUtilities.invokeLater;
import static javax.swing.SwingUtilities.isEventDispatchThread;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;

import javax.swing.event.EventListenerList;

import org.jdesktop.observablecollections.ObservableCollections;
import org.jdesktop.observablecollections.ObservableList;
import org.osgi.service.log.LogEntry;
import org.osgi.service.log.LogListener;
import org.osgi.service.log.LogReaderService;

/**
 * A OSGi Desktop {@link LogReaderService} to track logging messages. This is
 * EDT friendly so any new messages are added on the EDT.
 * 
 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
 * @since v1.0.0 [24 Sep 2010]
 */
public class DesktopLogReaderService implements LogReaderService {
	private final ObservableList<LogEntry> entries;
	private EventListenerList listenerList;

	/**
	 * 
	 */
	public DesktopLogReaderService() {
		this.entries = ObservableCollections.observableList(
				new ArrayList<LogEntry>());
		this.listenerList = new EventListenerList();
	}
	
	@Override
	public void addLogListener(LogListener listener) {
		this.listenerList.add(LogListener.class, listener);
	}

	@Override
	public void removeLogListener(LogListener listener) {
		this.listenerList.remove(LogListener.class, listener);
	}

	@Override
	public Enumeration<LogEntry> getLog() {
		return new IteratorEnumeration<LogEntry>(getEntries().iterator());
	}

	/**
	 * @return the list of the known entries.
	 */
	public synchronized List<LogEntry> getEntries() {
		return this.entries;
	}

	/**
	 * 
	 * @param entry
	 */
	synchronized void add(final LogEntry entry) {
		if (isEventDispatchThread()) {
			this.entries.add(entry);
			fire(entry);
			return;
		}

		invokeLater(new Runnable() {
			@Override
			public void run() {
				add(entry);
			}
		});
	}

	/**
	 * 
	 * @param entry
	 */
	public void remove(LogEntry entry) {
		this.entries.remove(entry);
	}

	/**
	 * 
	 */
	public void removeAll() {
		this.entries.clear();
	}

	/**
	 * 
	 * @param entry
	 */
	private void fire(LogEntry entry) {
		final LogListener[] listeners = listenerList.getListeners(LogListener.class);
		for (int i = listeners.length - 1; i >= 0; i--)
			listeners[i].logged(entry);
	}


	// --- Inner Classes ---

	/**
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [24 Sep 2010]
	 * @param <E>
	 */
	private class IteratorEnumeration<E> implements Enumeration<E> {
        private Iterator<E> iterator;

        public IteratorEnumeration(Iterator<E> iterator) {
            this.iterator = iterator;
        }

        @Override
        public boolean hasMoreElements() {
            return iterator.hasNext();
        }

        @Override
        public E nextElement() {
            return iterator.next();
        }
    }
}
