package civvi.osgi.desktop.preferences;

import java.awt.BorderLayout;
import java.awt.CardLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;

import javax.swing.BorderFactory;
import javax.swing.ButtonGroup;
import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.JToggleButton;
import javax.swing.SwingConstants;

import org.jdesktop.application.Action;
import org.jdesktop.application.Task.BlockingScope;
import org.jdesktop.swingx.JXPanel;
import org.osgi.framework.ServiceReference;

import civvi.osgi.desktop.AbstractBeanInfoAction;
import civvi.osgi.desktop.DesktopRuntimeException;
import civvi.osgi.desktop.DesktopService;
import civvi.osgi.desktop.OSGiDesktop;
import civvi.osgi.desktop.swingx.EditorComponent;
import civvi.osgi.desktop.swingx.JXButtonBar;
import civvi.osgi.desktop.swingx.SwingFactory;
import civvi.osgi.desktop.swingx.plaf.blue.BlueishButtonBarUI;

/**
 * The parent editor for displaying and editing preferences.
 * 
 * @author <a href="mailto=dansiviter@gmail.com">Daniel Siviter</a>
 * @since v1.0.0 [24th July 2008]
 */
public class PreferencesEditor extends EditorComponent<Object> {
	private static final long serialVersionUID = -8784941505423415969L;

	private final Logger log = Logger.getLogger(getClass().getName());
	private final OSGiDesktop desktop;
	private final Map<DesktopService<?>, AbstractPreferencePane> panels;
	private final JXButtonBar buttonBar;
	private final JXPanel preferencesPanel;
	private final ButtonGroup group;
	private final CardLayout cardLayout;

	/**
	 * 
	 * @param resource
	 */
	public PreferencesEditor(OSGiDesktop desktop) {
		super(new BorderLayout());
		this.desktop = desktop;
		this.panels = new HashMap<DesktopService<?>, AbstractPreferencePane>();

		setPreferredSize(new Dimension(480, 320));

		final JXPanel headerPane = SwingFactory.createHeaderPane(
				this.support.getString("title"),
				this.support.getString("description"),
				this.support.getIcon("icon"));
		add(headerPane, BorderLayout.NORTH);

		final JXPanel panel = new JXPanel(new BorderLayout(5, 5));
		panel.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
		this.group = new ButtonGroup();
		this.buttonBar = new JXButtonBar(SwingConstants.VERTICAL);
		this.buttonBar.setUI(new BlueishButtonBarUI());
		final JScrollPane buttonScrollPane = new JScrollPane(this.buttonBar);
		buttonScrollPane.setBorder(BorderFactory.createEmptyBorder());
		buttonScrollPane.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
		panel.add(buttonScrollPane, BorderLayout.WEST);
		panel.setOpaque(false);
		this.preferencesPanel = new JXPanel(this.cardLayout = new CardLayout());
		this.preferencesPanel.setOpaque(false);
		final JScrollPane scrollPane = new JScrollPane(this.preferencesPanel);
		scrollPane.setBorder(BorderFactory.createEmptyBorder());
		panel.add(scrollPane, BorderLayout.CENTER);
		add(panel, BorderLayout.CENTER);
		addPanels();
		doLoad();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void updateFields(Object editableBean) {
		throw new UnsupportedOperationException();
	}

	/**
	 * 
	 */
	private void addPanels() {
		final Map<ServiceReference, DesktopService<?>> services =
			this.desktop.getActivator().getServiceManager().getAll();

		for (Entry<ServiceReference, DesktopService<?>> entry : services.entrySet()) {
			final Class<? extends AbstractPreferencePane> editorCls = 
				entry.getValue().getPreferencesEditor();

			if (editorCls == null) {
				continue;
			}

			final AbstractPreferencePane editor;
			try {
				 editor = editorCls.newInstance();
			} catch (InstantiationException ie) {
				throw new DesktopRuntimeException(String.format(
						"Unable to create editor! [%s]", editorCls), ie);
			} catch (IllegalAccessException iae) {
				throw new DesktopRuntimeException(String.format(
						"Unable to create editor! [%s]", editorCls), iae);
			}
			
			final JToggleButton button = new JToggleButton(new ToggleAction(editor));

			this.buttonBar.add(button);
			this.group.add(button);
			this.preferencesPanel.add(editor, editor.getClass().getName());
			this.panels.put(entry.getValue(), editor);
		}
	}

	/**
	 * 
	 * TODO
	 *
	 * @return
	 */
	@Action(block = BlockingScope.WINDOW)
	public void save() {
		final int result = JOptionPane.showConfirmDialog(
				this,
				this.support.getString("confirmSave.message"),
				this.support.getString("confirmSave.title"),
				JOptionPane.OK_CANCEL_OPTION,
				JOptionPane.QUESTION_MESSAGE);

		if (result == JOptionPane.OK_OPTION) {
			doSave();
			setState(State.SAVED);
		}
	}

	/**
	 * Loads the preferences.
	 */
	private void doLoad() {
		for (Entry<DesktopService<?>, AbstractPreferencePane> entry : this.panels.entrySet()) {
			final AbstractPreferencePane editor = entry.getValue();
			if (editor != null) {
				editor.load(entry.getKey().getPreferences());
			}
		}
	}

	/**
	 * Saves the preferences. 
	 */
	private void doSave() {
		for (Entry<DesktopService<?>, AbstractPreferencePane> entry : this.panels.entrySet()) {
			try {
				final AbstractPreferencePane editor = entry.getValue();
				final Preferences preferences = entry.getKey().getPreferences();
				if (editor != null) {
					editor.save(preferences);
				}
				preferences.flush();
			} catch (BackingStoreException bse) {
				log.log(Level.WARNING, bse.getMessage(), bse);
			}
		}
	}


	// --- Inner classes ---

	/**
	 * An {@link javax.swing.Action} for toggling which panel is selected.
	 * 
	 * <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since 1.0.0 [23 Oct 2009]
	 */
	private class ToggleAction extends AbstractBeanInfoAction {
		private static final long serialVersionUID = 1L;

		private final AbstractPreferencePane editor;

		public ToggleAction(AbstractPreferencePane editor) {
			super(editor.getClass());
			this.editor = editor;
		}

		@Override
		public void actionPerformed(ActionEvent e) {
			cardLayout.show(preferencesPanel, editor.getClass().getName());
		}
	}
}
