package civvi.osgi.desktop.swingx;

import static civvi.common.GenericsUtil.getParamType;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableModel;

import org.jdesktop.observablecollections.ObservableCollections;
import org.jdesktop.observablecollections.ObservableList;
import org.jdesktop.observablecollections.ObservableListListener;

/**
 * Abstract implementation of a {@link TableModel} which has a few additions to
 * ease development including being compatible with BeansBinding (JSR295),
 * implements the {@link List} interface and has an alternative column and
 * row methods.
 * </p>
 * <strong>Note:</strong> the {@link Column} implementation is expected to be
 * an {@link Enum}. If not {@link #getColumns()} would
 * require being overriden.
 * 
 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
 * @since v1.0.0 [22nd October 2007]
 * @param <E> Element type
 */
public abstract class AbstractListTableModel<E, C extends Column>
extends AbstractTableModel
implements List<E>
{
	private static final long serialVersionUID = 5427447843092519142L;

	private final ObservableListHandler observableListHandler;
	protected final Class<C> columnType;

	private ObservableList<E> delegate;
	private C[] columns;

	/**
	 * Default constructor.
	 */
	@SuppressWarnings("unchecked")
	public AbstractListTableModel() {
		this.observableListHandler = new ObservableListHandler(this);
		// do a cheeky bit of reflection so we don't have to pass in the 
		// column class avoiding unnecessary boiler-plate code
		this.columnType = (Class<C>) getParamType(this.getClass(), 1);
		set(new ArrayList<E>()); // set empty list
	}

	/**
	 * <strong>WARNING:</strong> This will return the actual internal list.
	 * Modifications to this list will cause changes to the model.
	 * 
	 * @return the internal elements list.
	 */
	public ObservableList<E> getDelegate() {
		return this.delegate;
	}

	/**
	 * @param elements the elements to set.
	 */
	public void set(Collection<E> elements) {
		set(new ArrayList<E>(elements));
	}

	/**
	 * @param elements the elements to set.
	 */
	public void set(List<E> elements) {
		final ObservableList<E> oldValue = this.delegate;

		if (oldValue != null)
			oldValue.removeObservableListListener(this.observableListHandler);

		if (elements instanceof ObservableList<?>) {
			this.delegate = (ObservableList<E>) elements;
		} else {
			this.delegate = ObservableCollections.observableList(elements);
		}

		this.delegate.addObservableListListener(this.observableListHandler);
		fireTableDataChanged();
	}

	/**
	 * @return creates and returns column instances. Override if custom column
	 * types are used.
	 * @throws IllegalStateException if the column type isn't an enum.
	 */
	protected C[] createColumns() {
		if (!this.columnType.isEnum())
			throw new IllegalStateException("Column type isn't an Enum! [" + columnType + "]");
		return this.columnType.getEnumConstants();
	}
	
	/**
	 * @return the columns. 
	 */
	public C[] getColumns() {
		if (this.columns == null) {
			this.columns = createColumns();
		}
		return this.columns;
	}

	/**
	 * Returns the column instance for the given index.
	 * 
	 * @param i the column index.
	 * @return the column instance.
	 */
	public C getColumn(int i) {
		return getColumns()[i];
	}

	/**
	 * Returns the value for the cell at {@code column} and {@code row}.
	 * 
	 * @param row the row.
	 * @param column the column.
	 * @return the value at the row and column position.
	 */
	protected abstract Object getValueAt(E row, C column);

	/**
	 * This empty implementation is provided so users don't have to implement
     * this method if their data model is not editable.
	 * 
	 * @param value the value to set.
	 * @param row the row to update. 
	 * @param column the column to update.
	 */
	protected void setValueAt(Object value, E row, C column) { }

	
	// --- Overridden/Implemented Methods ---
	
	@Override
	public E get(int index) {
		return this.delegate.get(index);
	}

	@Override
	public void add(int index, E element) {
		this.delegate.add(index, element);
	}

	@Override
	public boolean add(E element) {
		return this.delegate.add(element);
	}

	@Override
	public boolean addAll(Collection<? extends E> elements) {
		return this.delegate.addAll(elements);
	}

	@Override
	public boolean addAll(int index, Collection<? extends E> elements) {
		return this.delegate.addAll(index, elements);
	}

	@Override
	public E set(int index, E element) {
		return this.delegate.set(index, element);
	}

	@Override
	public E remove(int index) {
		return this.delegate.remove(index);
	}

	@Override
	public boolean remove(Object element) {
		return this.delegate.remove(element);
	}

	@Override
	public boolean removeAll(Collection<?> elements) {
		return this.delegate.removeAll(elements);
	}

	@Override
	public boolean retainAll(Collection<?> elements) {
		return this.delegate.retainAll(elements);
	}

	@Override
	public int indexOf(Object element) {
		return this.delegate.indexOf(element);
	}

	@Override
	public int lastIndexOf(Object element) {
		return this.delegate.lastIndexOf(element);
	}

	@Override
	public boolean contains(Object element) {
		return this.delegate.contains(element);
	}

	@Override
	public boolean containsAll(Collection<?> elements) {
		return this.delegate.containsAll(elements);
	}

	@Override
	public void clear() {
		this.delegate.clear();
	}

	@Override
	public boolean isEmpty() {
		return this.delegate.isEmpty();
	}

	@Override
	public Iterator<E> iterator() {
		throw new UnsupportedOperationException("Computer says no!");
	}

	@Override
	public ListIterator<E> listIterator() {
		throw new UnsupportedOperationException("Computer says no!");
	}

	@Override
	public ListIterator<E> listIterator(int index) {
		throw new UnsupportedOperationException("Computer says no!");
	}

	@Override
	public List<E> subList(int fromIndex, int toIndex) {
		throw new UnsupportedOperationException("Computer says no!");
	}

	@Override
	public Object[] toArray() {
		return this.delegate.toArray();
	}

	@Override
	public <T> T[] toArray(T[] a) {
		return this.delegate.toArray(a);
	}

	@Override
	public int size() {
		return this.delegate.size();
	}

	@Override
	public int getRowCount() {
		return this.delegate.size();
	}

	@Override
	public String getColumnName(int i) {
		return getColumn(i).getDisplayName();
	}

	@Override
	public int getColumnCount() {
		return getColumns().length;
	}

	@Override
	public Class<?> getColumnClass(int i) {
		return getColumn(i).getType();
	}

	@Override
	public Object getValueAt(int rowIndex, int columnIndex) {
		return getValueAt(
				get(rowIndex),
				getColumn(columnIndex));
	}

	@Override
	public void setValueAt(Object value, int rowIndex, int columnIndex) {
		setValueAt(
				value,
				get(rowIndex),
				getColumn(columnIndex));
	}


	// --- Inner Classes ---

	/**
	 * A listener for passing events to the table model.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [22nd October 2007]
	 */
	@SuppressWarnings("rawtypes")
	private static class ObservableListHandler
	implements ObservableListListener
	{
		private final AbstractListTableModel<?,?> tableModel;

		private ObservableListHandler(AbstractListTableModel<?,?> tableModel) {
			this.tableModel = tableModel;
		}

		@Override
		public void listElementsAdded(
				ObservableList list,
				int i,
				int length)
		{
			this.tableModel.fireTableRowsInserted(i, i + length - 1);
		}

		@Override
		public void listElementsRemoved(
				ObservableList list,
				int i,
				List oldElements)
		{
			this.tableModel.fireTableRowsDeleted(
					i, i + oldElements.size());
		}

		@Override
		public void listElementReplaced(
				ObservableList list,
				int i,
				Object oldElement)
		{
			this.tableModel.fireTableRowsUpdated(i, i);
		}

		@Override
		public void listElementPropertyChanged(
				ObservableList list,
				int i)
		{
			this.tableModel.fireTableRowsUpdated(i, i);
		}
	}
}
