package civvi.osgi.desktop.swingx;

import java.awt.event.ActionEvent;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.KeyStroke;

import org.jdesktop.application.Application;
import org.jdesktop.application.ApplicationContext;
import org.jdesktop.application.MnemonicText;
import org.jdesktop.application.ResourceMap;
import org.jdesktop.application.Task;
import org.jdesktop.application.TaskService;
import org.jdesktop.application.ResourceMap.LookupException;

/**
 * Simple class to take the pain out of leveraging the AppFramework.
 * 
 * @author <a href="mailto:dansiviter@gmail.com">Dan Siviter</a>
 * @since v1.0.0 [21 May 2009]
 */
public class AppFrameworkSupport<A extends Application<?>> {
	private final Object source;
	private final Class<?> sourceCls;

	private ResourceMap resourceMap;
	private ActionMap actionMap;

	/**
	 * Creates a new AppFramework support class for the given resource source
	 * instance.
	 * 
	 * @param source the source to create for.
	 */
	public AppFrameworkSupport(Object source) {
		if (source == null)
			throw new NullPointerException("Souce cannot be null!");
		this.source = source;
		this.sourceCls = this.source.getClass();
	}

	/**
	 * Creates a new AppFramework support class for the given resource source
	 * class.
	 * 
	 * @param cls the class to create for.
	 */
	public AppFrameworkSupport(Class<?> cls) {
		if (cls == null)
			throw new NullPointerException("Souce class cannot be null!");
		this.source = null;
		this.sourceCls = cls;
	}

	/**
	 * @return an instance of the Application.
	 */
	@SuppressWarnings("unchecked")
	public final A getApplication() {
		return (A) getContext().getApplication();
	}

	/**
	 * @return the application context.
	 */
	public final ApplicationContext getContext() {
		return Application.getContext();
	}

	/**
	 * @return the resource map.
	 */
	public final ResourceMap getResourceMap() {
		if (this.resourceMap == null) {
			this.resourceMap = getContext().getResourceMap(this.sourceCls);
		}
		return this.resourceMap;
	}

	/**
	 * @return the action map.
	 */
	public ActionMap getActionMap() {
		if (this.actionMap == null) {
			if (this.source == null) {
				this.actionMap = getContext().getActionMap();
			} else {
				this.actionMap = getContext().getActionMap(
						this.sourceCls.getSuperclass(),
						this.source);
			}
		}
		return this.actionMap;
	}

	/**
	 * @return the task service.
	 */
	public TaskService getTaskService() {
		return getContext().getTaskService();
	}


	// --- ResourceMap ---

	/**
	 * If no arguments are specified, return the String value
	 * of the resource named <tt>key</tt>.  This is
	 * equivalent to calling <tt>getObject(key, String.class)</tt>
	 * If arguments are provided, then the type of the resource
	 * named <tt>key</tt> is assumed to be
	 * {@link String#format(String, Object...) format} string,
	 * which is applied to the arguments if it's non null.
	 * For example, given the following resources
	 * <pre>
	 * hello = Hello %s
	 * </pre>
	 * then the value of <tt>getString("hello", "World")</tt> would
	 * be <tt>"Hello World"</tt>.
	 *
	 * @param key the name of the resource.
	 * @param args the arguments used in
	 * {@link String#format(String, Object...)}.
	 * @return the String value of the resource named <tt>key</tt>
	 * @throws LookupException if an error occurs during lookup or String
	 * conversion.
	 * @throws IllegalArgumentException if key is {@code null}.
	 * @see ResourceMap#getObject(String, Class)
	 * @see String#format(String, Object...)
	 * @see ResourceMap#getString(String, Object...)
	 */
	public String getString(String key, Object... args) {
		return getResourceMap().getString(key, args);
	}

	/**
	 * A convenience method that calls
	 * {@link ResourceMap#getIcon(String)}.
	 *
	 * @param key the name of the resource.
	 * @return the Icon value of the resource.
	 * @throws LookupException if an error occurs during lookup or string
	 * conversion.
	 * @throws IllegalArgumentException if key is {@code null}.
	 * @see ResourceMap#getObject(String, Class)
	 * @see ResourceMap#getIcon(String)
	 */
	public Icon getIcon(String key) {
		return getResourceMap().getIcon(key);
	}

	/**
	 * A convenience method that calls
	 * {@link ResourceMap#getImageIcon(String)}.
	 * 
	 * @param key the name of the resource
	 * @return the ImageIcon value of the resource.
	 * @see ResourceMap#getIcon(String)
	 */
	public ImageIcon getImageIcon(String key) {
		return getResourceMap().getImageIcon(key);
	}

	/**
	 * A convenience method that calls {@link ResourceMap#getInteger(String)}. 
	 * 
	 * @param key the name of the resource.
	 * @return the integer value of the resource.
	 * @see ResourceMap#getInteger(String)
	 */
	public Integer getInteger(String key) {
		return getResourceMap().getInteger(key);
	}


	// --- ActionMap ---

	/**
	 * Returns an {@link Action} for the given key.
	 *
	 * @param key the action key.
	 * @param allowCreateEmpty if {@code true} if no existing action is found 
	 * a new empty action will be created. This is useful when constructing
	 * menu bars with sub-menus.
	 * @return the action instance.
	 */
	public Action getAction(Object key, boolean allowCreateEmpty) {
		final Action action = getActionMap().get(key);
		if (action == null && allowCreateEmpty) {
			return new EmptyAction(getResourceMap(), (String) key);
		}
		return action;
	}

	/**
	 * Returns the action for the given key. If no action for the corresponding
	 * key isn't found it will return {@code null}.
	 *
	 * @param key the action key.
	 * @return the action instance or {@code null} if none is found.
	 */
	public Action getAction(Object key) {
		return getAction(key, false);
	}

	/**
	 * Fires a {@link Action} using the given parameters.
	 *
	 * @param source
	 * @param actionsObject
	 * @param context
	 * @param actionKey
	 */
	public void fireAction(String actionKey) {
		SwingUtil.fireAction(this.source, getActionMap().get(actionKey), actionKey);
	}

	/**
	 * Fires a {@link Action} using the given parameters.
	 *
	 * @param source
	 * @param actionsObject
	 * @param context
	 * @param actionKey
	 */
	public void fireAction(String actionKey, ActionEvent evt) {
		SwingUtil.fireAction(getActionMap().get(actionKey), evt);
	}

	/**
	 * @param task the task to fire.
	 */
	public void fire(Task<?,?> task) {
		getContext().getTaskService().execute(task);
	}


	// --- Static Methods ---

	/**
	 * Fires an {@link Action} for the given source and {@code actionKey}.
	 * 
	 * @param source the source object.
	 * @param actionKey the key of the action to fire.
	 */
	public static void fireAction(Object source, String actionKey) {
		SwingUtil.fireAction(
				source,
				Application.getContext().getActionMap(source).get(actionKey),
				actionKey);
	}


	// --- Inner Classes ---

	/**
	 * A basic action that will create it self from the {@link ResourceMap}.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [13 Jun 2010]
	 */
	private class EmptyAction extends AbstractAction {
		private static final long serialVersionUID = 1L;

		public EmptyAction(ResourceMap resourceMap, String actionName){
			boolean iconOrNameSpecified = false;  // true if Action's icon/name properties set

			// Action.text => Action.NAME,MNEMONIC_KEY,DISPLAYED_MNEMONIC_INDEX_KEY
			String text = resourceMap.getString(actionName + ".Action.text");
			if (text != null) {
				MnemonicText.configure(this, text);
				iconOrNameSpecified = true;
			}
			// Action.mnemonic => Action.MNEMONIC_KEY
			Integer mnemonic = resourceMap.getKeyCode(actionName + ".Action.mnemonic");
			if (mnemonic != null) {
				putValue(MNEMONIC_KEY, mnemonic);
			}
			// Action.mnemonic => Action.DISPLAYED_MNEMONIC_INDEX_KEY
			Integer index = resourceMap.getInteger(actionName + ".Action.displayedMnemonicIndex");
			if (index != null) {
				putValue(DISPLAYED_MNEMONIC_INDEX_KEY, index);
			}
			// Action.accelerator => Action.ACCELERATOR_KEY
			KeyStroke key = resourceMap.getKeyStroke(actionName + ".Action.accelerator");
			if (key != null) {
				putValue(ACCELERATOR_KEY, key);
			}
			// Action.icon => Action.SMALL_ICON,LARGE_ICON_KEY
			Icon icon = resourceMap.getIcon(actionName + ".Action.icon");
			if (icon != null) {
				putValue(SMALL_ICON, icon);
				putValue(Action.LARGE_ICON_KEY, icon);
				iconOrNameSpecified = true;
			}
			// Action.smallIcon => Action.SMALL_ICON
			Icon smallIcon = resourceMap.getIcon(actionName + ".Action.smallIcon");
			if (smallIcon != null) {
				putValue(SMALL_ICON, smallIcon);
				iconOrNameSpecified = true;
			}
			// Action.largeIcon => Action.LARGE_ICON
			Icon largeIcon = resourceMap.getIcon(actionName + ".Action.largeIcon");
			if (largeIcon != null) {
				putValue(LARGE_ICON_KEY, largeIcon);
				iconOrNameSpecified = true;
			}
			// Action.shortDescription => Action.SHORT_DESCRIPTION
			putValue(SHORT_DESCRIPTION,
					resourceMap.getString(actionName + ".Action.shortDescription"));
			// Action.longDescription => Action.LONG_DESCRIPTION
			putValue(LONG_DESCRIPTION,
					resourceMap.getString(actionName + ".Action.longDescription"));
			// Action.command => Action.ACTION_COMMAND_KEY
			putValue(ACTION_COMMAND_KEY,
					resourceMap.getString(actionName + ".Action.command"));
			// If no visual was defined for this Action, i.e. no text
			// and no icon, then we default Action.NAME
			if (!iconOrNameSpecified) {
				putValue(NAME, actionName);
			}
		}

		@Override
		public void actionPerformed(ActionEvent e) {
			// do nuffin'!
		}
	}
}
