package civvi.osgi.desktop.swingx;

import java.awt.Component;
import java.awt.Point;
import java.awt.Rectangle;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.SwingUtilities;

import org.jdesktop.swingx.JXMultiSplitPane;
import org.jdesktop.swingx.MultiSplitLayout;
import org.jdesktop.swingx.MultiSplitLayout.Divider;
import org.jdesktop.swingx.MultiSplitLayout.Leaf;
import org.jdesktop.swingx.MultiSplitLayout.Node;
import org.jdesktop.swingx.MultiSplitLayout.Split;
import org.jdesktop.swingx.plaf.LookAndFeelAddons;

import civvi.osgi.desktop.swingx.docking.AbstractDock;
import civvi.osgi.desktop.swingx.docking.AbstractDockable;
import civvi.osgi.desktop.swingx.docking.DockFactory;
import civvi.osgi.desktop.swingx.docking.AbstractDock.DockPosition;
import civvi.osgi.desktop.swingx.plaf.DockingPaneUI;
import civvi.osgi.desktop.swingx.plaf.JXDockingPaneAddon;

/**
 * The {@link JXDockingPane} is a very simple and lightweight docking
 * component. I was getting annoyed with how complex they all seemed to be and
 * none of them worked out of the box and with the correct native look and
 * feel.
 * 
 * TODO Drag'n'drop functionality
 * TODO Double-click full size functionality
 * FIXME Don't like the selected functionality... think of something else!
 * 
 * @author <a href="mailto:dansiviter@gmail.com">Dan Siviter</a>
 * @since 20th Nov 2007
 */
public class JXDockingPane extends JXMultiSplitPane {
	private static final long serialVersionUID = 7436133280739850908L;
	public static final String uiClassID = "DockingPaneUI";

	public static final String DEFAULT_DOCK = "default";

	static {
		LookAndFeelAddons.contribute(new JXDockingPaneAddon());
	}

	private final Logger log = Logger.getLogger(getClass().getName());
	private AbstractDockable selectedDockable;
	private DockFactory dockFactory;


	/**
	 * Default constructor.
	 */
	public JXDockingPane() {
		super();
		setDockFactory(new DefaultDockFactory());
		getMultiSplitLayout().setLayoutByWeight(true);
	}

	/**
	 * @return the dockFactory.
	 */
	public DockFactory getDockFactory() {
		return this.dockFactory;
	}

	/**
	 * @param dockFactory the dockFactory to set.
	 */
	public void setDockFactory(DockFactory dockFactory) {
		DockFactory oldValue = getDockFactory();
		this.dockFactory = dockFactory;
		firePropertyChange("dockFactory", oldValue, getDockFactory());
	}

	/**
	 * Convenience method to perform parsing of model and adding of tabbed
	 * panes.
	 * 
	 * @param modelStr the string to parse.
	 * @throws IllegalArgumentException if model does not have a 'default'
	 * leaf.
	 */
	public void setModel(String modelStr) {
		if (modelStr == null || modelStr.trim().length() == 0) {
			throw new IllegalStateException("Model cannot be null!");
		}

		final Node model = MultiSplitLayout.parseModel(modelStr);

		if (findLeaf(model, DEFAULT_DOCK) == null) {
			throw new IllegalArgumentException(String.format(
					"Model must contain a dock at position! [%s]",
					DEFAULT_DOCK));
		}
		setModel(model);
	}

	/**
	 * Returns the {@link Leaf} for the given position.
	 * 
	 * @param position the position to look for.
	 * @return the leaf.
	 * @throws IllegalArgumentException if unable to locate position.
	 */
	public Leaf getLeaf(String position) {
		final Leaf leaf = findLeaf(getMultiSplitLayout().getModel(), position);

		if (leaf == null)
			throw new IllegalArgumentException(String.format(
					"Unable to locate leaf for position! [%s]",
					position));
		return leaf;
	}

	/**
	 * Returns a dock for the given position. If a leaf exists but no dock
	 * currently exists, then one will be created.
	 * <p/>
	 * <strong>NOTE</strong>: The position has to exist in the model otherwise
	 * an {@link IllegalArgumentException} will be thrown
	 *
	 * @param position the position of the required dock.
	 * @return returns a dock for the position.
	 * @throws IllegalArgumentException if a leaf for the given position does
	 * not exist.
	 */
	public AbstractDock getDock(String position) {
		final Leaf leaf = getLeaf(position);

		if (leaf == null) {
			throw new IllegalArgumentException(String.format(
					"Unknown leaf for position! [%s]",
					position));
		}

		return (AbstractDock) getMultiSplitLayout().getComponentForNode(leaf);
	}

	/**
	 * Attempts to dock with the default position. If that is unavailable, it
	 * searches for one that it is permitted to dock with. 
	 *
	 * @param dockable the dockable to dock.
	 * @return {@code true} if the dock was successful.
	 */
	public boolean dock(AbstractDockable dockable) {
		if (getMultiSplitLayout().getModel() == null) {
			throw new IllegalStateException(
			"Model cannot be null! Set model first.");
		}

		// default should always be available, but may not accept the dockable
		if (dock(dockable, DEFAULT_DOCK))
			return true;

		final Set<Leaf> leafs = getAllLeaf(getMultiSplitLayout().getModel());
		
		for (Leaf leaf : leafs) {
			if (dock(dockable, leaf.getName())) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Docks the given dockable at the position. If the dockable is already
	 * docked to another, then it will be removed from the old dock.
	 *
	 * @param dockable the component to dock.
	 * @param position the position to dock to.
	 * @return {@code true} if the dock completed successfully.
	 */
	public boolean dock(AbstractDockable dockable, String position) {
		return dock(dockable, position, -1);
		
/*		if (getMultiSplitLayout().getModel() == null) {
			throw new IllegalStateException(
			"Model cannot be null! Set model first.");
		}

		if (this.logger.isLoggable(Level.FINE)) {
			this.logger.log(Level.FINE,
					"Docking component. [dockable={0},position={1}]",
					new Object[] { dockable, position });
		}

		AbstractDock dock = getDock(position);
		if (dock == null) { // create if none exists
			dock = getDockFactory().create(position);
			if (this.logger.isLoggable(Level.FINE)) {
				this.logger.log(Level.FINE,
						"New dock created. [dock={0}]",
						new Object[] { dock });
			}
			add(dock, position);
		}

		if (dockable.isDocked())
			dockable.getDock().remove(dockable);

		return dock.add(dockable);*/
	}


	/**
	 * Docks the given dockable to the new position and if already docked it
	 * will remove from it. If the old dock permits close on empty then it will
	 * be cleaned up and removed.
	 * 
	 * @param dockable the dockable to dock.
	 * @param position the position of the dock.
	 * @param tabIndex the tab index to place the dockable. If -1 then it
	 * will add to the end of the tabs.
	 * @return {@code true} if the dock completed successfully.
	 * @throws NullPointerException if position dock does not exist.
	 */
	public boolean dock(
			AbstractDockable dockable,
			String position,
			int tabIndex)
	{
		if (this.log.isLoggable(Level.FINE)) {
			this.log.log(Level.FINE,
					"Docking component. [dockable={0},position={1},tabIndex={2}]",
					new Object[] { dockable, position, tabIndex });
		}
		
		if (getMultiSplitLayout().getModel() == null) {
			throw new IllegalStateException(
			"Model cannot be null! Set model first.");
		}
		
//		final AbstractDock sourceDock = dockable.getDock();
		AbstractDock targetDock = getDock(position);

		if (targetDock == null) { // create if none exists
			targetDock = getDockFactory().create(position);
			if (this.log.isLoggable(Level.FINE)) {
				this.log.log(Level.FINE,
						"New dock created. [dock={0}]",
						new Object[] { targetDock });
			}
			// check before we add it as there may be no reason
			// not the best approach I know, but prevents adding erroneous
			// components
			if (!targetDock.isDockable(dockable)) {
				return false;
			}
			add(targetDock, position);
		}

		// check if we're allowed to add this component before we remove it
		if (!targetDock.isDockable(dockable)) {
			return false;
		}

//		// source dock exists
//		if (sourceDock != null) {
//			int sourceTabIndex = sourceDock.indexOfComponent(dockable);
//			if (sourceDock == targetDock) {
//				// no change, do nothing!
//				if (sourceTabIndex == tabIndex) {
//					return false;
//				}
//				sourceDock.remove(sourceTabIndex);
//			} else {
//				sourceDock.remove(dockable);
//			}
//		}
//		
		if (tabIndex <= -1 || tabIndex > targetDock.getTabCount()) {
			tabIndex = targetDock.getTabCount();
		}
		
		return targetDock.dock(tabIndex, dockable);
	}

	/**
	 * Docks the dockable next to the position in the given direction
	 * effectively splitting the dock. 
	 *
	 * @param dockable the dockable to dock after split.
	 * @param position the position to split.
	 * @param splitDir the direction to split.
	 * @return the name of the new leaf.
	 * @see DockPlacement
	 */
	public String dock(
			AbstractDockable dockable,
			String position,
			DockPosition splitDir)
	{
		// lots of nasty code to get round deficiency in the MultiSplitPane
		final Leaf leaf = getLeaf(position);
		final Split split = leaf.getParent();
		final boolean rowLayout =
			splitDir == DockPosition.EAST || splitDir == DockPosition.WEST;
		final Leaf newLeaf = new Leaf("LEAF" + UUID.randomUUID().toString());

		if (split == null) {
			final Split newSplit = createSplit(newLeaf, leaf, splitDir);
			getMultiSplitLayout().setModel(newSplit);
		} else {
			final List<Node> children = split.getChildren();
			// add to current split
			if ((split.isRowLayout() && rowLayout) ||
					(!split.isRowLayout() && !rowLayout))
			{
				leaf.setWeight(leaf.getWeight() / 2);
				newLeaf.setWeight(leaf.getWeight() / 2);

				final int index = children.indexOf(leaf);
				if (splitDir == DockPosition.NORTH 
						|| splitDir == DockPosition.WEST)
				{
					// append before
					children.add(index, new Divider());
					children.add(index, newLeaf);
				} else {
					// append after
					children.add(index + 1, newLeaf);
					children.add(index + 1, new Divider());
				}

				split.setChildren(children);
			} else {
				// create new split in different direction
				final Split newSplit = createSplit(newLeaf, leaf, splitDir);
				split.replace(leaf, newSplit);
				// re-associate parent as setChildren will dis-associate
				// old leaf
				leaf.setParent(newSplit);
			}
		}

		System.out.println(printModel(getMultiSplitLayout().getModel()));
		
		final AbstractDock targetDock = getDockFactory().create(newLeaf.getName());
		if (this.log.isLoggable(Level.FINE)) {
			this.log.log(Level.FINE,
					"New dock created. [dock={0}]",
					new Object[] { targetDock });
		}
		add(targetDock, newLeaf.getName());
		dock(dockable, newLeaf.getName());
		return newLeaf.getName();
	}

	/**
	 * Convenience method to remove the leaf. This should cleanup parentage
	 * that {@link Split} doesn't do correctly!
	 *
	 * @param leaf the leaf to remove.
	 */
	public void remove(Leaf leaf) {
		if (leaf.getParent() != null) {
			final Split split = leaf.getParent();
			split.remove(leaf);
			leaf.setParent(null);

			if (split.getChildren().size() == 1) {
				final Node remaining = split.getChildren().get(0);
				if (split.getParent() != null) {
					split.getParent().replace(split, remaining);
					split.getParent().remove(split);
					split.setParent(null);
				} else {
					split.remove(remaining);
					remaining.setParent(null);
					getMultiSplitLayout().setModel(remaining);
				}
			}
		}
	}

	/**
	 * @return the selected dockable.
	 */
	public AbstractDockable getSelectedDockable() {
		return this.selectedDockable;
	}

	/**
	 * @param dockable sets the selected dockable.
	 * 
	 * TODO this should bring it to the front and sort focus
	 */
	public void setSelectedDockable(AbstractDockable dockable) {
		final AbstractDockable oldValue = getSelectedDockable();
		this.selectedDockable = dockable;
		firePropertyChange(
				"selectedDockable",
				oldValue,
				getSelectedDockable());
	}

	/**
	 * Returns the name of the L&F class that renders this component.
	 * 
	 * @return the string {@link #uiClassID}
	 * @see javax.swing.JComponent#getUIClassID
	 * @see javax.swing.UIDefaults#getUI
	 */
	@Override
	public String getUIClassID() {
		return uiClassID;
	}

	@Override
	public DockingPaneUI getUI() {
		return (DockingPaneUI) super.getUI();
	}

	/**
	 * Adds the dock at the given position.
	 * 
	 * @param dock the dock to add.
	 * @param position the position to place the dock.
	 */
	public void add(AbstractDock dock, String position) {
		getUI().installUI(dock);
		super.add(dock, position);
	}

	/**
	 * {@inheritDoc}
	 * <p/>
	 * Not supported in {@link JXDockingPane}
	 */
	@Override
	public void add(Component comp, Object constraints) {
		throw new UnsupportedOperationException("Use #dock(...) methods!");
	}

	/**
	 * {@inheritDoc}
	 * <p/>
	 * Not supported in {@link JXDockingPane}
	 */
	@Override
	public Component add(Component comp) {
		throw new UnsupportedOperationException("Use #dock(...) methods!");
	}

	/**
	 * {@inheritDoc}
	 * <p/>
	 * Not supported in {@link JXDockingPane}
	 */
	@Override
	public Component add(Component comp, int index) {
		throw new UnsupportedOperationException("Use #dock(...) methods!");
	}

	/**
	 * {@inheritDoc}
	 * <p/>
	 * Not supported in {@link JXDockingPane}
	 */
	@Override
	public void add(Component comp, Object constraints, int index) {
		throw new UnsupportedOperationException("Use #dock(...) methods!");
	}

	/**
	 * {@inheritDoc}
	 * <p/>
	 * Not supported in {@link JXDockingPane}
	 */
	@Override
	public Component add(String name, Component comp) {
		throw new UnsupportedOperationException("Use #dock(...) methods!");
	}

	/**
	 * Returns the tab index for the dock at the given point.
	 *
	 * @param pt
	 * @return
	 */
	@SuppressWarnings("unused")
	private int getTargetTabIndex(AbstractDock dock, Point pt) {
		Point p = (Point) pt.clone();
		for (int i = 0; i < dock.getTabCount(); i++) {
			Rectangle rect = dock.getBoundsAt(i);
			// rect.setRect(rect.x - rect.width / 2, rect.y, rect.width, rect.height);
			if (rect.contains(p)) {
				return i;
			}
		}
		Rectangle rect = dock.getBoundsAt(dock.getTabCount() - 1);
		rect.x = rect.x + rect.width;
		// rect.setRect(rect.x + rect.width / 2, rect.y, rect.width + 100, rect.height);
		if (rect.contains(p)) {
			return dock.getTabCount();
		}

		return -1;
	}

	/**
	 * Moves the tab from the previous index to the next within the dock.
	 * 
	 * @param dock the dock to perform the tab manipulation.
	 * @param prev the old index to move from.
	 * @param next the new index to move to.
	 */
	@SuppressWarnings("unused")
	private void convertTab(AbstractDock dock, int prev, int next) {
		if (next < 0 || prev == next) {
			return;
		}

		final Component cmp = dock.getComponentAt(prev);
		String str = dock.getTitleAt(prev);
		if (next == dock.getTabCount()) {
			dock.remove(prev);
			dock.addTab(str, cmp);
			dock.setSelectedIndex(dock.getTabCount() - 1);
		} else if (prev > next) {
			remove(prev);
			dock.insertTab(str, null, cmp, null, next);
			dock.setSelectedIndex(next);
		} else {
			remove(prev);
			dock.insertTab(str, null, cmp, null, next);
			dock.setSelectedIndex(next);
		}
	}
	
	/**
	 * Attempts to find the dock at the given screen point.
	 *
	 * @param p the point of the dock.
	 * @return the found dock, or {@code null}.
	 */
	public AbstractDock getDockFromScreen(Point p) {
		p = (Point) p.clone();
		SwingUtilities.convertPointFromScreen(p, this);
		return (AbstractDock) getComponentAt(p);
	}


	// --- Static Methods ---

	/**
	 * Attempts to find the {@link Leaf} by name from within the given node.
	 * This will iterate through the child nodes.
	 * 
	 * @param node the node to search from.
	 * @param name the name of the leaf to find.
	 * @return the found leaf, or {@code null} if not found.
	 */
	public static Leaf findLeaf(Node node, String name) {
		if (node instanceof Leaf && name.equals(((Leaf) node).getName())) {
			return (Leaf) node;
		}

		if (node instanceof Split) {
			for (Node child : ((Split) node).getChildren()) {
				final Leaf leaf = findLeaf(child, name);
				if (leaf != null)
					return leaf;
			}
		}
		return null;
	}

	public static Set<Leaf> getAllLeaf(Node node) {
		final Set<Leaf> leaf = new HashSet<Leaf>();
		getAllLeaf(leaf, node);
		return leaf;
	}
	
	/**
	 * Returns all {@link Leaf} from within the given node and append them to
	 * the list.
	 * 
	 * @param leaf the list of existing leafs.
	 * @param node the node to search from.
	 * @return a list of leafs.
	 */
	public static Set<Leaf> getAllLeaf(Set<Leaf> leaf, Node node) {
		if (node instanceof Split) {
			for (Node child : ((Split) node).getChildren())
				getAllLeaf(leaf, child);
		} else if (node instanceof Leaf) {
			leaf.add((Leaf) node);
		}
		return leaf;
	}

	/**
	 * Returns the bounds of the whole tab area for the given dock.
	 *
	 * @param dock the to test.
	 * @return the bounds of the whole tab area.
	 */
	@SuppressWarnings("unused")
	private static Rectangle getTabAreaBound(AbstractDock dock) {
		final Rectangle lastTab =
			dock.getUI().getTabBounds(dock, dock.getTabCount() - 1);
		return new Rectangle(0, 0,
				dock.getWidth(), lastTab.y + lastTab.height);
	}
	
	/**
	 * Creates a new {@link Split} in the {@link Leaf}.
	 *
	 * @param newLeaf the new leaf to add.
	 * @param leaf the leaf to split.
	 * @param splitDir the direction in which to split.
	 * @return the created split.
	 */
	private static Split createSplit(Leaf newLeaf, Leaf leaf, DockPosition splitDir) {
		final Split newSplit;
		if (splitDir == DockPosition.NORTH || splitDir == DockPosition.WEST) {
			newSplit = new Split(newLeaf, new Divider(), leaf);
		} else {
			newSplit = new Split(leaf, new Divider(), newLeaf);
		}
		newSplit.setRowLayout(
				splitDir == DockPosition.EAST || splitDir == DockPosition.WEST);
		// split inherits weight to maintain layout
		newSplit.setWeight(leaf.getWeight());

		// distribute evenly
		leaf.setWeight(.5d);
		newLeaf.setWeight(.5d);

		return newSplit;
	}

	/**
	 * Returns a parent {@link AbstractDock} for the component. 
	 *
	 * @param comp
	 * @return
	 */
	public static AbstractDock getDock(Component comp) {
		if (comp instanceof AbstractDock)
			return (AbstractDock) comp;
		return (AbstractDock) SwingUtilities.getAncestorOfClass(
				AbstractDock.class, comp);
	}

	/**
	 * Print the tree with enough detail for simple debugging.
	 * 
	 * @param root the model to print.
	 * @return a {@link String} representation of the model.
	 */
	public static String printModel(Node root) {
		final StringBuilder builder = new StringBuilder();
		printModel(builder, root);
		return builder.toString();
	}

	/**
	 * TODO
	 * 
	 * MultiSplitLayout.Leaf "default" weight=1.0 java.awt.Rectangle[x=2,y=0,width=638,height=480]
	 * (LEAF name=default weight=1.0)
	 * 
	 * @param builder
	 * @param root
	 */
	private static void printModel(StringBuilder builder, Node root) {
		if (root instanceof Divider) {
			return;
		} else if (root instanceof Split) {
			final Split split = (Split) root;
			builder.append(String.format(
					"(%1$s weight=%2$g ",
					split.isRowLayout() ? "ROW" : "COLUMN",
							split.getWeight()));
			for(Node child : split.getChildren()) {
				printModel(builder, child);
			}
			builder.append(")");
		} else if (root instanceof Leaf) {
			final Leaf leaf = (Leaf) root;
			final Split parent = leaf.getParent();
			// use bounds to decide weightings
			final double weight;
			if (parent != null) {
				weight = parent.isRowLayout() ?
							leaf.getBounds().getWidth() / parent.getBounds().getWidth() :
							leaf.getBounds().getHeight() / parent.getBounds().getHeight();
			} else {
				weight = 1d;
			}
			builder.append(String.format(
					"(LEAF name=%1$s weight=%2$g)",
					leaf.getName(),
					weight));
		} else {
			throw new IllegalArgumentException();
		}
	}


	// --- Inner Classes ---

	/**
	 * Default implementation of the dockfactory.
	 *
	 * @author <a href="mailto:dansiviter@gmail.com">Dan Siviter</a>
	 * @since 3 Dec 2007
	 */
	public static class DefaultDockFactory implements DockFactory {
		/**
		 * {@inheritDoc}
		 */
		@Override
		public AbstractDock create(String position) {
			return new AbstractDock() {
				private static final long serialVersionUID = 1L;
			};
		}
	}
}
