package civvi.osgi.desktop.swingx;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.GradientPaint;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JComponent;
import javax.swing.SwingUtilities;
import javax.swing.Timer;
import javax.swing.UIManager;

import org.jdesktop.swingx.JXPanel;
import org.jdesktop.swingx.painter.AbstractPainter;
import org.jdesktop.swingx.painter.CompoundPainter;
import org.jdesktop.swingx.painter.MattePainter;
import org.jdesktop.swingx.painter.Painter;
import org.jdesktop.swingx.painter.TextPainter;
import org.jdesktop.swingx.plaf.LookAndFeelAddons;

import civvi.osgi.desktop.swingx.filter.LinerFadeFilter;
import civvi.osgi.desktop.swingx.painter.GridPainter;
import civvi.osgi.desktop.swingx.plaf.HeapViewUI;
import civvi.osgi.desktop.swingx.plaf.JXHeapViewAddon;

import com.jhlabs.image.ShadowFilter;

/**
 * A component to view the Java heap usage. The concept was originally taken
 * from Scott Violet's blog but I updated it to use the SwingX {@link Painter}s
 * system to drastically cut down the code and reduce complexity (813 lines to
 * 352, damn I'm good!).
 * 
 * NOTE The graphics represented are not strictly accurate, and should
 * only be used as a visual guide to the current memory usage. I need to look
 * into improving this, but at this time I can't be bothered!  
 *
 * @author <a href="mailto:dansiviter@gmail.com">Dan Siviter</a>
 * @since 13-Nov-2005
 */
public class JXHeapView extends JXPanel {
	private static final long serialVersionUID = 1L;
	
	private static final GradientPaint BACKGROUND_GRADIENT =
		new GradientPaint(
				new Point2D.Double(0, 0),
				UIManager.getColor("controlShadow"),
				new Point2D.Double(0, 1),
				UIManager.getColor("control"));
	private static final Paint TICK_PAINT =
		UIManager.getColor("textHighlight");
	private static final Paint POINT_PAINT =
		UIManager.getColor("controlDkShadow");
	
	public final static String uiClassID = "HeapViewUI";

	static {
		LookAndFeelAddons.contribute(new JXHeapViewAddon());
	}
	
	/* Data for the graph as a percentage of the heap used. */
	private List<Long> data;
	private TextPainter textPainter;
	private AbstractPainter<? extends JComponent> ticksPainter;
	private Timer timer;


	/**
	 * Default constructor.
	 */
	public JXHeapView() {
		super();
		this.data = Collections.synchronizedList(new ArrayList<Long>());
		setBackgroundPainter(new CompoundPainter<JXHeapView>(
				new MattePainter(BACKGROUND_GRADIENT, true),
				new GridPainter(new Color(255, 255, 255, 63)),
				this.ticksPainter = new TicksPainter(),
				this.textPainter = new TextPainter()));
		this.ticksPainter.setFilters(new LinerFadeFilter(0x22, 0x88));
//		this.textPainter.setLocation(new Point2D.Float(0.5f, 0.5f));
		this.textPainter.setFilters(new ShadowFilter(3, 1, -1, 1));
		this.textPainter.setFillPaint(UIManager.getColor("controlHighlight"));
		setBorder(BorderFactory.createLineBorder(UIManager.getColor("controlDkShadow"), 1));
		this.timer = new Timer(1000, new ActionHandler());
	}

	/**
	 * Invoked when the update timer fires. Updates the necessary data
	 * structures and triggers repaints.
	 */
	private void updateData() {
		if (!isShowing()) {
			// Either we've become invisible, or one of our ancestors has.
			// Stop the timer and bale. Next paint will trigger timer to
			// restart.
			this.timer.stop();
			return;
		}
		Runtime r = Runtime.getRuntime();
		long byteUsage = r.totalMemory() - r.freeMemory();
		
		// add to the start
		this.data.add(0, byteUsage);
		
		Object[] args = {
				(float) byteUsage / 1024 / 1024,	// used
				(float) r.totalMemory() / 1024 / 1024,					// available
				(float) r.maxMemory() / 1024 / 1024,					// max
				(r.totalMemory() - r.freeMemory()) / (float) r.maxMemory()
		};

		this.textPainter.setText(MessageFormat.format(getUI().getTextFormat(), args));
		this.ticksPainter.clearCache();
		setToolTipText(MessageFormat.format(getUI().getToolTipFormat(), args));
		repaint();
	}

	
	// --- Overridden/Implemented Methods ---
	
	/**
	 * Returns the name of the L&F class that renders this component.
	 * 
	 * @return the string {@link #uiClassID}
	 * @see javax.swing.JComponent#getUIClassID
	 * @see javax.swing.UIDefaults#getUI
	 */
	@Override
	public String getUIClassID() {
		return uiClassID;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public HeapViewUI getUI() {
		return (HeapViewUI) super.getUI();
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void paintComponent(Graphics g) {
		if (!this.timer.isRunning()) {
			this.timer.start();
		}
		
		// fix the length of the data, so it never gets too big!
		if (this.data.size() > getWidth()) {
			for (int i = this.data.size() - 1; i > getWidth(); i--) {
				this.data.remove(i);
			}
		}
		
		super.paintComponent(g);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void processMouseEvent(MouseEvent e) {
		super.processMouseEvent(e);
		if (!e.isConsumed()) {
			if (e.getID() == MouseEvent.MOUSE_CLICKED
					&& SwingUtilities.isLeftMouseButton(e)
					&& e.getClickCount() == 1)
			{
				System.gc();
			}
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Dimension getMinimumSize() {
		if (isMinimumSizeSet()) {
			return super.getMinimumSize();
		}
		return new Dimension(100, 20);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Dimension getPreferredSize() {
		if (isPreferredSizeSet()) {
			return super.getPreferredSize();
		}
		return getMinimumSize();
	}
	
	
	// --- Inner Classes ---
	
	/**
	 * A handler for updating the data.
	 *
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since 8 Dec 2006
	 */
	private class ActionHandler implements ActionListener {
		/**
		 * {@inheritDoc}
		 */
		public void actionPerformed(ActionEvent e) {
			updateData();
		}
	}

	/**
	 * Paints the ticks representing memory usage.
	 *
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since 8 Dec 2006
	 */
	private class TicksPainter extends AbstractPainter<JXHeapView> {
		/**
		 * {@inheritDoc}
		 */
		@Override
		protected void doPaint(
				Graphics2D g,
				JXHeapView comp,
				int width,
				int height)
		{
			long totalMemory = Runtime.getRuntime().totalMemory();
			long maxMemory = Runtime.getRuntime().maxMemory();
			
			double prev_x = 0;
			double prev_y = 0;
			
			for (int i = 0; i < width && i < data.size(); i++) {
				g.setPaint(POINT_PAINT);
				double percentageOfTotal = data.get(i) / (double) totalMemory;
				double x = width - i;
				double y = height - percentageOfTotal * height;
				final Line2D line = new Line2D.Double(
						prev_x > 0 ? prev_x : x,
						prev_y > 0 ? prev_y : y,
						x,
						y);
				g.draw(line);
				
				prev_x = x;
				prev_y = y;
				
				// paint percentage of max graph
				g.setPaint(TICK_PAINT);
				double percentageOfMax = data.get(i) / (double) maxMemory;
				g.fillRect(
						width - i,
						height - (int) Math.round(percentageOfMax * height),
						1,
						height);
			}
		}
	}
}
