package civvi.osgi.desktop.swingx;

import java.awt.Component;
import java.awt.Container;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.swing.Action;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JSeparator;

import civvi.osgi.desktop.swingx.menu.MenuModel;
import civvi.osgi.desktop.swingx.menu.MenuModelEvent;
import civvi.osgi.desktop.swingx.menu.MenuModelListener;
import civvi.osgi.desktop.swingx.menu.MenuModel.Node;

/**
 * A {@link JMenuBar} that has an internal model representation for complex
 * dynamic menus.
 * 
 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
 * @since v1.0.0 [13 Jun 2010]
 */
public class JXMenuBar extends JMenuBar {
	private static final long serialVersionUID = 1L;

	private final MenuModelListener modelListener;
	private final Map<String, ActionProvider> providers = new HashMap<String, ActionProvider>();
	private final List<String> utilisedActions = new ArrayList<String>();
	private MenuModel model;

	/**
	 * Default constructor.
	 */
	public JXMenuBar() {
		addPropertyChangeListener("model", new ModelChangedHandler());
		this.modelListener = new MenuModelHandler();
		setModel(new MenuModel());
	}

	/**
	 * @return the instance of the model associated with this
	 * {@link JXMenuBar}.
	 */
	public MenuModel getModel() {
		return model;
	}

	/**
	 * @param model the model to set.
	 */
	private void setModel(MenuModel model) {
		final MenuModel oldValue = this.model;
		this.model = model;
		firePropertyChange("model", oldValue, this.model);
	}

	public void register(ActionProvider provider) {
		this.providers.put(provider.getClass().getName(), provider);
		// FIXME all associated action should appear from the menu when registered
	}

	public void unregister(ActionProvider provider) {
		this.providers.remove(provider.getClass().getName());
		// FIXME all associated action should disappear from the menu when unregistered
	}

	/**
	 * Locates {@link Action} from the give providers using the
	 * {@code actionId} in the format:
	 * <pre>
	 * 	&lt;providerClassName&gt;.&lt;actionKey&gt;
	 * </pre> 
	 * 
	 * @param actionId
	 * @return
	 * @throws NullPointerException if the action provider cannot be found. 
	 */
	private Action getAction(String actionId) {
		final ActionProvider provider = this.providers.get(getProviderName(actionId));
		if (provider == null) {
			throw new NullPointerException(String.format(
					"Unable to located provider for actionId! [%s]",
					actionId));
		}

		final String actionKey = getActionKey(actionId);
		final Action action = provider.getAction(actionKey, true);
		if (action == null) {
			throw new IllegalStateException("Unable to locate action! [" + actionId + "]");
		}
		return action;
	}

	/**
	 * Adds the node.
	 * 
	 * @param leaf
	 * @return {@code true} if add was successful (see {@link Container#add(Component)}).
	 * @throws IllegalArgumentException if node already exists.
	 */
	private boolean add(Node leaf) {
		if (leaf == getModel().getRoot()) {
			return true;
		}

		if (find(leaf.getPath()) != null) {
			throw new IllegalStateException(
					"JMenuItem already exists for leaf! [" + leaf + "]");
		}

		final Action action = getAction(leaf.getActionId());
		action.putValue("Node", leaf);

		final Node parent = leaf.getParent();

		if (parent == getModel().getRoot()) {
			final JMenu menu = new JMenu(action);
			final int index = leaf.getPrev() != null ? indexOf(this, getAction(leaf.getPrev().getActionId())) : 0;
			return add(menu, index) != null;
		}

		JMenuItem parentItem = find(parent.getPath());
		
		if (parentItem == null) {
			throw new IllegalStateException("Parent doesn't exist! [" + parent + "]");
		}
		
		// convert to a JMenu if not one
		if (!(parentItem instanceof JMenu)) {
			if (parentItem instanceof JCheckBoxMenuItem) {
				throw new IllegalStateException(
						"Cannot convert a selectable JMenuItem to a JMenu! Invalid menu setup.");
			}

			if (parent.getParent() == getModel().getRoot()) {
				final int index = parentItem != null ? SwingUtil.indexOf(this, parentItem) : 0;
				if (parentItem != null ) remove(parentItem);
				parentItem = new JMenu(getAction(parent.getActionId()));
				add(parentItem, index);
			} else {
				final JMenu menu = (JMenu) find(parent.getParent().getPath());
				final int index = SwingUtil.indexOf(menu, parentItem);
				menu.remove(parentItem);
				parentItem = new JMenu(parentItem.getAction());
				menu.add(parentItem, index);
			}
		}

		this.utilisedActions.add(leaf.getActionId());
		final JComponent component = SwingUtil.createMenuItem(action);
//		final Range range = getSection((JMenu) parentItem, leaf.getSection());
		
		// if no range, then the section doesn't exist
		// if there is a range, but 'start' = 0, then there will be a separator after 'end'
		// if there is a range, but 'end' = size, then there will be a separator before 'start'
		// if there is a range, but neither of the other two occur then there will be a separator before 'start' and after 'end'
		
		removeSeparators((JMenu) parentItem);
		
		final String sectionName = leaf.getSection();
		int index = ((JMenu) parentItem).getMenuComponentCount();

		if (sectionName != null) {
			final Map<String, Section> sections = getSections((JMenu) parentItem);
			final Section section = sections.get(sectionName);
			if (section != null) {
				index = section.end + 1;
			}
		}

		try {
			return parentItem.add(component, index) != null;
		} finally {
			addSeparators((JMenu) parentItem);
		}
	}

	private void removeSeparators(JMenu menu) {
		for (int i = menu.getMenuComponentCount() - 1; i >=0; i--) {
			if (menu.getMenuComponent(i) instanceof JSeparator) {
				menu.remove(i);
			}
		}
	}

	private void addSeparators(JMenu menu) {
		String prevSection = null;
		for (int i = menu.getMenuComponentCount() - 1; i >=0; i--) {
			final JMenuItem item = (JMenuItem) menu.getMenuComponent(i);
			final Node node = (Node) item.getAction().getValue("Node");
			
			if (prevSection != null) {
				if (!prevSection.equals(node.getSection())) {
					menu.add(new JPopupMenu.Separator(), i + 1);
				}
			}
			prevSection = node.getSection();
		}
	}
	
	/**
	 * 
	 * TODO
	 * @param menu
	 * @param section
	 * @return
	 */
//	private Section getSection(JMenu menu, String section) {
//		int start = -1, end = -1;
//		for (int i = 0; i < menu.getMenuComponentCount(); i++) {
//			final JMenuItem item = (JMenuItem) menu.getMenuComponent(i);
//			
//			final Node node = (Node) item.getAction().getValue("Node");
//			if (section.equals(node.getSection())) {
//				if (start == -1) {
//					start = i;
//				}
//			} else if (start != -1 && !section.equals(node.getSection())) {
//				end = i - 1;
//				return new Section(section, start, end);
//			}
//		}
//		return null;
//	}
	
	/**
	 * 
	 * @param menu
	 * @return
	 */
	private Map<String, Section> getSections(JMenu menu) {
		final Map<String, Section> sections = new HashMap<String, Section>();
		
		for (int i = 0; i < menu.getMenuComponentCount();) {
			final Component item = menu.getMenuComponent(i);

			if (item instanceof JSeparator) {
				
				continue;
			}

			final Node node = (Node) ((JMenuItem) item).getAction().getValue("Node");
			final String sectionName = node.getSection();
			int sectionEnd = getSectionEnd(i, sectionName, menu); 
			sections.put(sectionName, new Section(sectionName, i, sectionEnd));
			i = sectionEnd + 1;
		}
		System.out.println(sections);
		return sections;
	}

	/**
	 * 
	 * TODO
	 * @param start
	 * @param name
	 * @param menu
	 * @return
	 */
	private int getSectionEnd(int start, String name, JMenu menu) {
		for (int i = start; i < menu.getMenuComponentCount(); i++) {
			final Component comp = menu.getMenuComponent(i);
			
			if (comp instanceof JSeparator)
				return i - 1;
			
			final JMenuItem item = (JMenuItem) menu.getMenuComponent(i);
			final Node node = (Node) item.getAction().getValue("Node");
			if (!name.equals(node.getSection())) {
				return i - 1;
			}
		}
		return menu.getMenuComponentCount() - 1;
	}
	
	/**
	 * Attempts to locate a {@link JMenuItem} that contains the given
	 * {@link Action}.
	 * 
	 * @param item
	 * @param action
	 * @return
	 */
	private int indexOf(JComponent item, Action action) {
		for (int i = 0; i < item.getComponentCount(); i++) {
			final Component comp = item.getComponent(i);
			if (comp instanceof JMenuItem && ((JMenuItem) comp).getAction().equals(action)) {
				return i;
			}
		}
		return -1;

	}

	/**
	 * Remove the node from the menu.
	 * 
	 * @param node the node to remove.
	 * @param force if {@code true} then it will remove {@link JMenu}s, which
	 * their existence means they still have children or are top-level
	 * components within {@link JXMenuBar}.
	 * @throws IllegalArgumentException if unable to locate the node in the
	 * menu bar.
	 */
	public boolean remove(Node node, boolean force, String... path) {
		final JMenuItem item = find(path);

		if (item == null)
			throw new IllegalArgumentException(
					"Unable to locate node! [" + node + "]");
//		if (!force && item instanceof JMenu)
//			throw new IllegalArgumentException(
//					"Unable remove a node that is a JMenu! Remove all child nodes first. [" + node + "]");

		if (path[0].equals(this.model.getRoot().getId())) {
			remove(item);
		} else {
			final JMenuItem parentItem = (JMenuItem) find(path).getParent();
			parentItem.remove(item);
		}
		this.utilisedActions.remove(node.getActionId());
		return true;
	}

	/**
	 * Rebuild the model from scratch.
	 */
	private void rebuild() {
		throw new UnsupportedOperationException();
	}

	/**
	 * Finds the {@link JMenuItem} at the given path starting with
	 * {@code root}.
	 * 
	 * @param path the path to the node starting with {@code root}.
	 * @return the found menu item or {@code null} if it cannot be found.
	 */
	private JMenuItem find(String... path) {
		if (!path[0].equals(getModel().getRoot().getId())) {
			throw new IllegalStateException("Erroneous state! Something went a tad wrong here!");
		}
		
		JComponent menuItem = this;
		for (int i = 1; i < path.length; i++) {
			menuItem = find(menuItem, path[i]);

			if (menuItem == null)
				return null;
			if (menuItem == this)
				throw new IllegalStateException("Erroneous state! Something went a tad wrong here!");
		}
		return (JMenuItem) menuItem;
	}

	/**
	 * Finds the {@link JMenuItem} with a {@link Node#getId()} that matches
	 * the given {@code nodeId}.
	 * 
	 * @param parent the parent component to search through.
	 * @param nodeId the node id to find.
	 * @return the found item or {@code null} if none is found.
	 */
	private JMenuItem find(JComponent parent, Object nodeId) {
		final int compCount = parent instanceof JMenu ? ((JMenu) parent).getMenuComponentCount() : parent.getComponentCount();
		for (int i = 0; i < compCount; i++) {
			final Component current;
		
			if (parent instanceof JMenu) {
				current = ((JMenu) parent).getMenuComponent(i);
			} else {
				current = parent.getComponent(i);
			}
			if (!(current instanceof JMenuItem)) continue;
				
			final Node node = (Node) ((JMenuItem) current).getAction().getValue("Node");
			if (node.getId().equals(nodeId))
				return (JMenuItem) current;
		}
		return null;
	}

//	private void populatePath(Node leaf, List<Node> path) {
//		path.add(path.size(), leaf);
//		if (leaf.getParent() != null)
//			populatePath(leaf.getParent(), path);
//	}


	private static String getProviderName(String actionId) {
		return actionId.substring(0, actionId.lastIndexOf('.'));
	}
	
	private static String getActionKey(String actionId) {
		return actionId.substring(actionId.lastIndexOf('.') + 1, actionId.length());
	}
	
	// --- Inner Classes ---

	/**
	 * Handles changes to the model.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [13 Jun 2010]
	 */
	private class MenuModelHandler implements MenuModelListener {
		@Override
		public void changed(MenuModelEvent evt) {
			switch (evt.getType()) {
			case ADDED:
				add(evt.getNode());
				break;
			case REMOVED:
				remove(evt.getNode(), false, evt.getPath());
				break;
			case REBUILD:
				rebuild();
			}
		}
	}

	/**
	 * Handles changes to the model.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [13 Jun 2010]
	 */
	private class ModelChangedHandler implements PropertyChangeListener {
		@Override
		public void propertyChange(PropertyChangeEvent evt) {
			if (evt.getOldValue() != null) {
				((MenuModel) evt.getOldValue()).removeListener(JXMenuBar.this.modelListener);
			}
			if (evt.getNewValue() != null) {
				((MenuModel) evt.getNewValue()).addListener(JXMenuBar.this.modelListener);
			}
		}
	}

	/**
	 * TODO
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [13 Jun 2010]
	 */
	private static class Section {
		private final String name;
		private final int start, end;
		
		Section(String name, int start, int end) {
			this.name = name;
			this.start = start;
			this.end = end;
		}
		
		@Override
		public String toString() {
			return "Section[" + name + "," + start + "," + end + "]";
		}
	}
}
