package civvi.osgi.desktop.swingx;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.Frame;
import java.awt.Graphics2D;
import java.awt.HeadlessException;
import java.awt.Image;
import java.awt.Paint;
import java.awt.RadialGradientPaint;
import java.awt.Window;
import java.awt.MultipleGradientPaint.CycleMethod;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.geom.Point2D;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.PixelGrabber;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.logging.Level;

import javax.swing.AbstractAction;
import javax.swing.AbstractButton;
import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRootPane;
import javax.swing.JSeparator;
import javax.swing.JToggleButton;
import javax.swing.JToolBar;
import javax.swing.RowSorter;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.border.EmptyBorder;

import org.jdesktop.application.ApplicationAction;
import org.jdesktop.swingx.JXErrorPane;
import org.jdesktop.swingx.JXHeader;
import org.jdesktop.swingx.JXPanel;
import org.jdesktop.swingx.JXTable;
import org.jdesktop.swingx.JXTreeTable;
import org.jdesktop.swingx.error.ErrorInfo;
import org.jdesktop.swingx.util.PaintUtils;

import civvi.osgi.desktop.swingx.EditorComponent.State;

import com.jgoodies.forms.layout.CellConstraints;

/**
 * Utility methods for Swing development.
 * 
 * @author <a href="mailto:dansiviter@gmail.com">Dan Siviter</a>
 * @since 24th July 2008
 */
public class SwingUtil {
	private static final String DEFUALT_TRUNCATE_SEPARATOR = "...";
	public final static Color ALTERNATE_LINE_COLOUR = new Color(0xF5, 0xF5, 0xF0);

	/**
	 * Private constructor.
	 */
	private SwingUtil() {
		// do nothing!
	}

	/**
	 * Creates a {@link JMenuBar} from the given list and {@link ActionMap}.
	 * 
	 * @param actionMap
	 * @param list
	 * @return
	 */
	public static JMenuBar createMenuBar(ActionMap actionMap, List<?> list) {
		final JMenuBar menubar = new JMenuBar();
		populate(menubar, actionMap, list);
		return menubar;
	}

	/**
	 * Populates a {@link JMenuBar} from the given list and {@link ActionMap}.
	 * 
	 * @param menubar
	 * @param actionMap
	 * @param list
	 * @return
	 */
	public static void populate(JMenuBar menubar, ActionMap actionMap, List<?> list) {
		JMenu menu = null;

		for (Object element : list) {
			if (element == null) {
				if (menu != null) {
					menu.addSeparator();
				}
			} else if (element instanceof List<?>) {
				menu = createMenu(actionMap, (List<?>)element);
				if (menu != null) {
					menubar.add(menu);
				}
			} else if (element.getClass().isArray()) {
				menu = createMenu(actionMap, (Object[]) element);
				if (menu != null) {
					menubar.add(menu);
				}
			} else  {
				if (menu != null) {
					menu.add(actionMap.get(element));
				}
			}
		}
	}

	/**
	 * Creates a {@link JMenuBar} from the given array and action map. 
	 * 
	 * @param actionMap
	 * @param array
	 * @return
	 * @see #createMenu(ActionMap, List)
	 */
	public static JMenuBar createMenuBar(
			ActionMap actionMap,
			Object... array)
	{
		return createMenuBar(actionMap, Arrays.asList(array));
	}

	/**
	 * Creates a {@link JMenu} from the given list and action map.
	 * 
	 * @param actionMap
	 * @param list
	 * @return
	 */
	public static JMenu createMenu(ActionMap actionMap, List<?> list) {
		// The first item will be the action for the JMenu
		final Action titleAction = actionMap.get(list.get(0));
		if (titleAction == null) {
			return null;
		}
		final JMenu menu = new JMenu();
		menu.setAction(titleAction);

		// The rest of the items represent the menu items.
		for (Object element : list.subList(1, list.size())) {
			if (element == null) {
				menu.addSeparator();
			} else if (element instanceof List<?>) {
				JMenu newMenu = createMenu(actionMap, (List<?>) element);
				if (newMenu != null) {
					menu.add(newMenu);
				}
			} else if (element.getClass().isArray()) {
				JMenu newMenu = createMenu(actionMap, (Object[]) element);
				if (newMenu != null) {
					menu.add(newMenu);
				}
			} else {
				final Action action = actionMap.get(element);
				if (action == null) {
					continue;
				} else {
					menu.add(createMenuItem(action));
				}
			}
		}
		return menu;
	}

	/**
	 * Creates a suitable menu item.
	 * 
	 * @param action
	 * @return
	 */
	public static JMenuItem createMenuItem(Action action) {
		final JMenuItem mi;
		final Boolean selected = (Boolean) action.getValue(Action.SELECTED_KEY);
		if (selected != null) {
			mi = new JCheckBoxMenuItem(action);
		} else {
			mi = new JMenuItem(action);
		}
		mi.setHorizontalTextPosition(JButton.TRAILING);
		mi.setVerticalTextPosition(JButton.CENTER);
		return mi;
	}
	
	/**
	 * Creates a {@link JMenu} from the given array and action map.
	 * 
	 * @param actionMap
	 * @param keys
	 * @return
	 * @see #createMenu(ActionMap, List)
	 */
	public static JMenu createMenu(ActionMap actionMap, Object... array) {
		return createMenu(actionMap, Arrays.asList(array));
	}

	/**
	 * Finds a {@link JMenu} for the given {@link Action} within the given
	 * {@link JMenuBar}.
	 *
	 * @param menuBar
	 * @param action
	 * @return
	 */
	public static JMenu findMenu(
			JMenuBar menuBar,
			Action action)
	{
		for (int i = 0; i < menuBar.getMenuCount(); i++) {
			final JMenu found = findMenu(
					menuBar.getMenu(i),
					action);
			if (found != null)
				return found;
		}
		return null;
	}

	/**
	 * Finds a {@link JMenu} for the given {@link Action} within the given
	 * menu.
	 *
	 * @param menu
	 * @param action
	 * @return
	 */
	public static JMenu findMenu(JMenu menu, Action action) {
		if (action.equals(menu.getAction())) {
			return menu;
		}
		for (Component comp : menu.getMenuComponents()) {
			if (comp instanceof JMenu) {
				if (((JMenu) comp).getAction().equals(action)) {
					return (JMenu) comp;
				}
				return findMenu((JMenu) comp, action);
			}
		}
		return null;
	}

	/**
	 * Creates a {@link JToolBar} for tab panels.
	 *
	 * @param actionMap
	 * @param items
	 * @return
	 */
	public static JToolBar createTabToolbar(
			ActionMap actionMap,
			Object... items)
	{
		return createTabToolbar(actionMap, Arrays.asList(items));
	}

	/**
	 * Creates a {@link JToolBar} for tab panels.
	 *
	 * @param actionMap
	 * @param items
	 * @return
	 */
	public static JToolBar createTabToolbar(
			ActionMap actionMap,
			List<?> items)
	{
		final JToolBar toolBar = createToolbar(actionMap, items);
		toolBar.setFloatable(false);
		toolBar.add(Box.createHorizontalGlue(), 0);
		return toolBar;
	}

	/**
	 * Creates a {@link JToolBar}.
	 *
	 * @param actionMap
	 * @param items
	 * @return
	 */
	public static JToolBar createToolbar(
			ActionMap actionMap,
			Object... items)
	{
		return createToolbar(actionMap, Arrays.asList(items));
	}
	
	/**
	 * Creates a {@link JToolBar} with the list items and {@link ActionMap}.
	 *
	 * @param actionMap
	 * @param items
	 * @return
	 */
	public static JToolBar createToolbar(
			ActionMap actionMap,
			List<?> items)
	{
		final JToolBar toolBar = new JToolBar();
		populate(toolBar, actionMap, items);
		return toolBar;
	}

	/**
	 * Populates a {@link JToolBar} with the items and {@link ActionMap}.
	 * 
	 * @param toolBar
	 * @param actionMap
	 * @param items
	 */
	public static void populate(JToolBar toolBar, ActionMap actionMap, Object... items) {
		populate(toolBar, actionMap, Arrays.asList(items));
	}

	/**
	 * Populates a {@link JToolBar} with the list items and {@link ActionMap}.
	 * 
	 * @param toolBar
	 * @param actionMap
	 * @param items
	 */
	public static void populate(JToolBar toolBar, ActionMap actionMap, List<?> items) {
		toolBar.setOpaque(false);
		for (Object item : items) {
			if (item instanceof Component) {
				toolBar.add((Component) item);
			} else if (item == null) {
				toolBar.addSeparator();
			} else {
				final Action action = item instanceof Action ?
						(Action) item : actionMap.get(item);
				final AbstractButton button = createToolbarItem(action);
				toolBar.add(button);
			}
		}
	}

	/**
	 * 
	 * TODO
	 * @param action
	 * @return
	 */
	public static AbstractButton createToolbarItem(Action action) {
		final AbstractButton button;
		if (action == null) {
			throw new NullPointerException("Action cannot be null!");
		} else if (action.getValue(Action.SELECTED_KEY) != null) {
			button = new JToggleButton(action);
		} else {
			button = new JButton(action);
		}

		button.setOpaque(false);
		// hide text if icon is available
		if (action != null && (action.getValue(Action.SMALL_ICON) != null ||
				action.getValue(Action.LARGE_ICON_KEY) != null)) {
			button.setHideActionText(true);
		}
		button.setHorizontalTextPosition(JButton.CENTER);
		button.setVerticalTextPosition(JButton.BOTTOM);
		return button;
	}

	/**
	 * Method to throw exception if on the EventDispatch Thread (EDT). It has
	 * limited use but good to use while developing to force a piece of code to
	 * use a none EDT thread. Another option is to use:
	 * <pre>
	 * assert EventQueue.isDispatchThread() : "Don't use on EDT!";
	 * </pre>
	 *
	 * @throws IllegalThreadStateException if on EDT.
	 */
	public static void forceNoneEDT()
	throws IllegalThreadStateException
	{
		if (EventQueue.isDispatchThread()) {
			throw new IllegalThreadStateException(
			"Operation not permitted on EventDispatch Thread!");
		}
	}

	/**
	 * Splits a recursive string into its parts. Useful for defining a menu
	 * structure via resources. 
	 * <p/>
	 * E.g.
	 * <pre>
	 * file[new,subMenu[bob,dave],|,exit],view[showMenu],help[help,|,about]
	 * </pre>
	 * 
	 * @param str the string to split.
	 * @return the created list.
	 */
	public static List<Object> split(String str) {
		final List<Object> list = new ArrayList<Object>();
		split(list, str);
		return list;
	}

	/**
	 * A convenience method to split a recursive string into its parts. Useful
	 * for defining a menu structure via resources. 
	 * <p/>
	 * E.g.
	 * <pre>
	 * file[new,subMenu[bob,dave],|,exit],view[showMenu],help[help,|,about]
	 * </pre>
	 * 
	 * @param list the list to insert into.
	 * @param str the string to split.
	 */
	private static void split(List<Object> list, String str) {
		int depth = 0;
		int start = 0;
		for (int i = 0; i < str.length(); i++) {
			final char c = str.charAt(i);

			if (c == '[') {
				depth++;
			} else if (c == ']') {
				depth--;
			}

			if (depth == 0 &&
					(i + 1 == str.length() || (str.charAt(i + 1) == ',')))
			{
				final String token = str.substring(start, i + 1);

				if (token.contains("[")) {
					final List<Object> subList = new ArrayList<Object>();
					subList.add(token.substring(0, token.indexOf('[')));
					split(subList, token.substring(token.indexOf('[') + 1, token.length() - 1));
					list.add(subList);
				} else if (token == null || "|".equals(token) || "".equals(token)) {
					list.add(null);
				} else {
					list.add(token);
				}
				start = i + 2;
			}
		}
	}

	/**
	 * Convenience method to get an alpha altered version of the input colour.
	 *
	 * @param colour the input colour.
	 * @param t the new alpha value (between 0.0 and 1.0 inclusive).
	 * @return the new colour instance.
	 */
	public static Color alpha(Color colour, float t) {
		return alpha(colour, (int) (t * 255f));
	}

	/**
	 * Convenience method to get an alpha altered version of the input colour.
	 *
	 * @param colour
	 * @param t the new alpha value (between 0 and 255 inclusive).
	 * @return the new colour instance.
	 */
	public static Color alpha(Color colour, int t) {
		return new Color(
				colour.getRed(),
				colour.getGreen(),
				colour.getBlue(),
				t);
	}

	/**
	 * Truncates the file name. This uses a default separator of
	 * '...'.
	 *
	 * @param file the file to truncate.
	 * @param maxChars the maximum number of characters required.
	 * @return the representation of the file.
	 * @see #truncateFilename(File, String, int)
	 */
	public static String truncateFilename(
			final File file,
			final int maxChars)
	{
		return truncateFilename(file, DEFUALT_TRUNCATE_SEPARATOR, maxChars);
	}

	/**
	 * Truncates the file name. This is useful for gui components that only
	 * want to display an indication of what the file is (i.e. toolbars,
	 * tab titles, etc). It has a preference for the file name opposed to
	 * displaying the path. For example:
	 * <pre>
	 * C:\A\Long\Path\My Really Long File name.foo
	 * </pre>
	 *  With a max length of 33 and seperator of '...' would become:
	 * <pre>
	 * C:\A\L...\My Really Long File name.foo
	 * </pre>
	 * With a max length of 20 and seperator of '...' would become:
	 * <pre>
	 * C:\... File name.foo
	 * </pre>
	 * 
	 * @param file the file to truncate.
	 * @param truncateSeparator the separator to indicate that a truncation
	 * has occurred.
	 * @param maxChars the maximum number of characters required.
	 * @return the truncated string representation of the file.
	 */
	public static String truncateFilename(
			final File file,
			final String truncateSeparator,
			final int maxChars)
	{
		final String filePath = file.getAbsolutePath();

		if (filePath.length() <= maxChars) {
			return filePath;
		}

		final StringBuffer buffer = new StringBuffer();

		final String fileName = file.getName();
		if (fileName.length() >= maxChars - 6) {
			buffer.append(filePath.substring(0, 3));
			buffer.append(truncateSeparator);
			buffer.append(fileName.substring(
					fileName.length() - maxChars + 3 + truncateSeparator.length(),
					fileName.length()));
/*		} else if (fileName.length() >= maxChars - 6) {
			buffer.append(filePath.substring(
					0,
					3));
			buffer.append(truncateSeparator);
			buffer.append(fileName.substring(fileName.length() + 6 - maxChars, fileName.length()));
*/		} else {
			buffer.append(filePath.substring(
					0,
					maxChars - fileName.length() - File.separator.length() - truncateSeparator.length()));
			buffer.append(truncateSeparator);
			buffer.append(File.separator);
			buffer.append(fileName);
		}
		return buffer.toString();
	}

	/**
	 * A utility method for displaying a dialog.
	 *
	 * @param parentComponent
	 * @param message
	 * @param title
	 * @return
	 */
	public static int showDialog(
			Component parentComponent,
			Object message,
			String title)
	{
		JOptionPane.showMessageDialog(
				parentComponent,
				message,
				title,
				JOptionPane.PLAIN_MESSAGE);

		return -1;
	}

	/**
	 * Adds fill components to empty cells in the first row and first column of the grid.
	 * This ensures that the grid spacing will be the same as shown in the designer.
	 * 
	 * @param cols an array of column indices in the first row where fill components should be added.
	 * @param rows an array of row indices in the first column where fill components should be added.
	 */
	public static void addFillComponents(
			Container panel,
			int[] cols,
			int[] rows)
	{
		final Dimension filler = new Dimension(10,10);

		boolean filled_cell_11 = false;
		final CellConstraints cc = new CellConstraints();
		if (cols.length > 0 && rows.length > 0) {
			if (cols[0] == 1 && rows[0] == 1) {
				/** add a rigid area  */
				panel.add( Box.createRigidArea(filler), cc.xy(1,1));
				filled_cell_11 = true;
			}
		}

		for (int index = 0; index < cols.length; index++) {
			if (cols[index] == 1 && filled_cell_11) {
				continue;
			}
			panel.add(Box.createRigidArea(filler), cc.xy(cols[index], 1));
		}

		for (int index = 0; index < rows.length; index++) {
			if (rows[index] == 1 && filled_cell_11) {
				continue;
			}
			panel.add(Box.createRigidArea(filler), cc.xy(1, rows[index]));
		}
	}

	/**
	 * Fires a {@link ActionListener} using zero modifiers. This is useful for
	 * firing {@link ApplicationAction}s to show blocking dialog.
	 *
	 * @param source the source of the event.
	 * @param listener the listener to fire.
	 * @param command a string that may specify a command (possibly one 
	 * of several) associated with the event.
	 * @see #fireAction(Object, ActionListener, String, int)
	 */
	public static void fireAction(
			Object source,
			ActionListener listener,
			String command)
	{
		fireAction(source, listener, command, 0);
	}

	/**
	 * Fires a {@link ActionListener}. This is useful for firing 
	 * {@link ApplicationAction}s to show blocking dialog.
	 *
	 * @param source the source of the event.
	 * @param listener the listener to fire.
	 * @param command a string that may specify a command (possibly one 
	 * of several) associated with the event.
	 * @param modifiers the modifier keys held down during this action.
	 * @see ActionEvent#ActionEvent(Object, int, String, long, int)
	 */
	public static void fireAction(
			final Object source,
			final ActionListener listener,
			final String command,
			final int modifiers)
	{
		fireAction(listener, new ActionEvent(
				source,
				ActionEvent.ACTION_PERFORMED,
				command,
				System.currentTimeMillis(),
				modifiers));
	}

	/**
	 * Fires a {@link ActionListener}. This is useful for firing 
	 * {@link ApplicationAction}s to show blocking dialog. This can be called
	 * on and off the EDT.
	 *
	 * @param listener the listener to fire.
	 * @param evt the action event to fire.
	 * @param modifiers the modifier keys held down during this action.
	 * @see ActionEvent#ActionEvent(Object, int, String, long, int)
	 */
	public static void fireAction(
			final ActionListener listener,
			final ActionEvent evt)
	{
		if (!SwingUtilities.isEventDispatchThread()) {
			SwingUtilities.invokeLater(new Runnable() {
				@Override
				public void run() {
					fireAction(listener, evt);
				}
			});
			return;
		}
		listener.actionPerformed(evt);
	}

	/**
	 * Packs all the columns on the {@link JXTreeTable}. This is a workaround
	 * for an annoying issue where the
	 * {@link JXTreeTable.TreeTableModelAdapter#delayedFireTableStructureChanged}
	 * will delay the structure updates meaning you're unable to packAll
	 * directly after setting data onto the the tree-table.
	 *
	 * @param treeTable
	 */
	public static void packAll(final JXTreeTable treeTable) {
		// JXTreeTable will break #packAll due to stupid 
		// TreeTableModelAdaptor#delayedFireTableStructureChanged
		SwingUtilities.invokeLater(new Runnable() {
			@Override
			public void run() {
				treeTable.packAll();	
			}			
		});
	}
	
	/**
	 * Gets an {@code Icon} given the url {@code resource}.
	 * 
	 * @param resource
	 *            the url path to a valid icon
	 * @return An {@code Icon}
	 * @throws RuntimeException if the {@link ClassLoader} could not locate the resource.
	 */
	@Deprecated
	public static Icon getIcon(String resource) {
		ClassLoader cl = Thread.currentThread().getContextClassLoader();
		URL url = cl.getResource(resource);
		if (null == url) {
			throw new RuntimeException("ClassLoader could not locate " + resource);
		}
		return new ImageIcon(url);
	}

	/**
	 * Converts an {@code Icon} to an {@code Image}.
	 * 
	 * @param icon
	 *            {@code Icon} that needs to be converted
	 * @return the converted {@code Image}
	 */
	public static Image iconToImage(Icon icon) {
		if (icon instanceof ImageIcon)
			return ((ImageIcon) icon).getImage();
		throw new IllegalArgumentException(
				"Unable to get Image from Icon type! [" + icon + "]");
	}

	/**
	 * Displays an error dialog.
	 *
	 * @param parentComponent
	 * @param title
	 * @param message
	 * @param category
	 * @param cause
	 * @param level
	 */
	public static void showErrorDialog(
			Component parentComponent,
			String title,
			String message,
			String category,
			Throwable cause,
			Level level)
	{
		final ErrorInfo errorInfo = new ErrorInfo(
				title,
				message,
				null,
				category,
				cause,
				level,
				null);
		
		JXErrorPane errorPane = new JXErrorPane();
		errorPane.setErrorInfo(errorInfo);
	
/*		if (System.getProperties().containsKey(EmailReporter.SMTP_HOST)) {
			final Properties props = new Properties();
			props.setProperty(EmailReporter.SMTP_HOST, System.getProperty(EmailReporter.SMTP_HOST));
			props.setProperty(EmailReporter.TO, System.getProperty(EmailReporter.TO));
			props.setProperty(EmailReporter.FROM, System.getProperty(EmailReporter.FROM));
			errorPane.setErrorReporter(new EmailReporter(props));
		}*/

		if (parentComponent != null) {
			JXErrorPane.showDialog(parentComponent, errorPane);
		} else {
			JXErrorPane.showFrame(null, errorPane);
		}
	}

	/**
	 * Displays an {@link JDialog}.
	 *
	 * @param parentComponent
	 * @param title
	 * @return
	 */
	public static void showDialog(
			Component parentComponent,
			JComponent component,
			String title)
	{
		final JDialog dialog;
		final Window window = getWindowForComponent(parentComponent);
		if (window instanceof Frame) {
			dialog = new JDialog((Frame)window, title, true);	
		} else {
			dialog = new JDialog((Dialog)window, title, true);
		}
		initDialog(dialog, component, parentComponent);
		
		dialog.setLocationRelativeTo(parentComponent);
		dialog.setVisible(true);
	}

	/**
	 * Displays an editor dialog.
	 * 
	 * @param parentComponent the parent component used for centralising the
	 * component on screen and component hierarchy.
	 * @param component the editor component to show.
	 * @param title the title of the dialog.
	 * @return
	 */
	public static State showEditorDialog(
			Component parentComponent,
			EditorComponent<?> component,
			String title)
	{
		final JDialog dialog;
		final Window window = getWindowForComponent(parentComponent);
		if (window instanceof Frame) {
			dialog = new JDialog((Frame)window, title, true);	
		} else {
			dialog = new JDialog((Dialog)window, title, true);
		}
		initEditorDialog(dialog, component, parentComponent);
		
		dialog.setLocationRelativeTo(parentComponent);
		dialog.setVisible(true);
		return component.getState();
	}

	/**
	 * Initialises the {@link JDialog} for the {@link EditorComponent}.
	 *
	 * @param dialog
	 * @param component
	 * @param parentComponent
	 */
	private static void initEditorDialog(
			final JDialog dialog,
			final EditorComponent<?> component,
			final Component parentComponent)
	{
		dialog.setResizable(true);
		dialog.setComponentOrientation(component.getComponentOrientation());
		Container contentPane = dialog.getContentPane();

		contentPane.setLayout(new BorderLayout());
		contentPane.add(component, BorderLayout.CENTER);
		
		final int buttonWidth = 75;
		
		final JPanel buttonPanel = new JPanel();
		buttonPanel.setLayout(new BoxLayout(buttonPanel, BoxLayout.X_AXIS));
		buttonPanel.setBorder(BorderFactory.createEmptyBorder(2, 4, 4, 4));
		
		if (component.getHelpAction() != null) {
			final JButton helpButton = new JButton(component.getHelpAction());
			helpButton.setBorderPainted(false);
			helpButton.setOpaque(false);
			helpButton.setFocusable(false);
			helpButton.setBorder(new EmptyBorder(3, 3, 3, 3));
			helpButton.setHideActionText(true);
			buttonPanel.add(helpButton);
		}

		buttonPanel.add(Box.createHorizontalGlue());

		for (Action action : component.getOtherActions()) {
			final JButton button = new JButton(action);
			buttonPanel.add(Box.createHorizontalStrut(5));
			fixWidth(button, buttonWidth);
			buttonPanel.add(button);
		}
		buttonPanel.add(Box.createHorizontalStrut(10));
		if (component.getSaveAction() != null) {
			buttonPanel.add(Box.createHorizontalStrut(5));
			final JButton saveButton = new JButton(component.getSaveAction());
			fixWidth(saveButton, buttonWidth);
			buttonPanel.add(saveButton);
		}
		final JButton cancelButton = new JButton(component.getCancelAction());
		buttonPanel.add(Box.createHorizontalStrut(5));
		fixWidth(cancelButton, buttonWidth);
		buttonPanel.add(cancelButton);
		contentPane.add(buttonPanel, BorderLayout.SOUTH);
		
		if (JDialog.isDefaultLookAndFeelDecorated()) {
			boolean supportsWindowDecorations =
				UIManager.getLookAndFeel().getSupportsWindowDecorations();
			if (supportsWindowDecorations) {
				dialog.setUndecorated(true);
				component.getRootPane().setWindowDecorationStyle(JRootPane.PLAIN_DIALOG);
			}
		}
		dialog.pack();
		dialog.setLocationRelativeTo(parentComponent);
		WindowAdapter adapter = new WindowAdapter() {
			private boolean gotFocus = false;
			public void windowClosing(WindowEvent we) {
				fireAction(we.getSource(), component.getCancelAction(), "cancel");
			}
			public void windowGainedFocus(WindowEvent we) {
				// Once window gets focus, set initial focus
				if (!gotFocus) {
                    component.setState(State.DEFAULT);
					gotFocus = true;
				}
			}
		};
		dialog.addWindowListener(adapter);
		dialog.addWindowFocusListener(adapter);
		dialog.addComponentListener(new ComponentAdapter() {
			public void componentShown(ComponentEvent ce) {
				// reset value to ensure closing works properly
				component.setState(State.DEFAULT);
			}
		});
		
		component.addPropertyChangeListener("state", new PropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent event) {
				// Let the defaultCloseOperation handle the closing
				// if the user closed the window without selecting a button
				// (newValue = null in that case).  Otherwise, close the dialog.
				if (dialog.isVisible() && event.getSource() == component &&
						event.getPropertyName().equals("state") &&
						event.getNewValue() != null &&
						(event.getNewValue() == State.CANCELLED || event.getNewValue() == State.SAVED)) {
					component.disposeDialog();
				}
			}
		});
	}

	/**
	 * Initialises the {@link JDialog} for the {@link JComponent}.
	 * 
	 * @param dialog
	 * @param component
	 * @param parentComponent
	 */
	private static void initDialog(
			final JDialog dialog,
			final JComponent component,
			final Component parentComponent)
	{
		dialog.setResizable(true);
		dialog.setComponentOrientation(component.getComponentOrientation());
		Container contentPane = dialog.getContentPane();

		contentPane.setLayout(new BorderLayout());
		contentPane.add(component, BorderLayout.CENTER);
		
		final int buttonWidth = 75;
		
		final JPanel buttonPanel = new JPanel();
		buttonPanel.setLayout(new BoxLayout(buttonPanel, BoxLayout.X_AXIS));
		buttonPanel.setBorder(BorderFactory.createEmptyBorder(2, 4, 4, 4));
		
		buttonPanel.add(Box.createHorizontalGlue());
		
		@SuppressWarnings("serial")
		final Action closeAction = new AbstractAction("Close") {
			@Override
			public void actionPerformed(ActionEvent e) {
				dialog.dispose();
			}
		};
		
		final JButton button = new JButton(closeAction);
			fixWidth(button, buttonWidth);
			buttonPanel.add(button);
		
		contentPane.add(buttonPanel, BorderLayout.SOUTH);
		
		if (JDialog.isDefaultLookAndFeelDecorated()) {
			boolean supportsWindowDecorations =
				UIManager.getLookAndFeel().getSupportsWindowDecorations();
			if (supportsWindowDecorations) {
				dialog.setUndecorated(true);
				component.getRootPane().setWindowDecorationStyle(JRootPane.PLAIN_DIALOG);
			}
		}
		dialog.pack();
		dialog.setLocationRelativeTo(parentComponent);
		WindowAdapter adapter = new WindowAdapter() {
//			private boolean gotFocus = false;
			public void windowClosing(WindowEvent we) {
				fireAction(we.getSource(), closeAction, "close");
			}
		};
		dialog.addWindowListener(adapter);
		dialog.addWindowFocusListener(adapter);
	}

	/**
	 * Returns the parent window for the component by navigating up the
	 * component hierarchy. 
	 *
	 * @param comp the component to test.
	 * @return the parent {@link Window} if one is found.
	 * @throws HeadlessException
	 */
	public static Window getWindowForComponent(Component comp) 
	throws HeadlessException
	{
		if (comp instanceof Frame || comp instanceof Dialog) {
			return (Window) comp;
		}
		return getWindowForComponent(comp.getParent());
	}

	/**
	 * Fixes the width of the component but maintaining it old height.
	 *
	 * @param comp
	 * @param width
	 */
	public static void fixWidth(JComponent comp, int width) {
		comp.setPreferredSize(new Dimension(
				width,
				comp.getPreferredSize().height));
		comp.setMaximumSize(comp.getPreferredSize());
	}

	/**
	 * Fixes the height of the component but maintaining it old width.
	 *
	 * @param comp
	 * @param height
	 */
	public static void fixHeight(JComponent comp, int height) {
		comp.setPreferredSize(new Dimension(
				comp.getPreferredSize().width,
				height));
	}

	/**
	 * Resizes the gradient to fit the with and height.
	 *
	 * @param p
	 * @param width
	 * @param height
	 * @return
	 */
	public static Paint resizeGradient(Paint p, int width, int height) {
        if (p == null) {
        	return p;
        }
        if (p instanceof RadialGradientPaint) {
        	final RadialGradientPaint gp = (RadialGradientPaint)p;
        	final Point2D center = gp.getCenterPoint();
        	final Point2D focus = gp.getFocusPoint();
        	final float[] fractions = gp.getFractions();
        	final Color[] colors = gp.getColors();
            final CycleMethod cycleMethod = gp.getCycleMethod();
            // XXX can't do ellipses yet
            final float radius = width > height ? width : height;
            return new RadialGradientPaint(center, radius, focus, fractions, colors, cycleMethod);
        }
        return PaintUtils.resizeGradient(p, width, height);
    }

	/**
	 * Returns a {@link List} of the rows visible to the user.
	 * 
	 * @param <T>
	 * @param table the parent table.
	 * @param tableModel the table model data.
	 * @return the visible rows.
	 */
	public static <T> List<T> getVisibleRows(
			JXTable table,
			AbstractListTableModel<T,?> tableModel)
	{
		final List<T> data = new ArrayList<T>();
		final RowSorter<?> rowSorter = table.getRowSorter();
		for (int i = 0; i < rowSorter.getViewRowCount(); i++) {
			final int index = rowSorter.convertRowIndexToModel(i);
			data.add(tableModel.get(index));
		}
		return Collections.unmodifiableList(data);
	}

	/**
	 * Turns on display of text in all child buttons within the
	 * {@link JToolBar}.
	 * 
	 * @param toolBar
	 * @param show
	 */
	public static void showText(JToolBar toolBar, boolean show) {
		for (Component comp : toolBar.getComponents()) {
			if (comp instanceof AbstractButton) {
				((AbstractButton) comp).putClientProperty("hideActionText", Boolean.valueOf(show));
			}
		}
		toolBar.revalidate();
	}
	
	/**
	 * 
	 * @param image
	 * @return
	 */
	public static BufferedImage toBufferedImage(ImageIcon image) {
		return toBufferedImage(image.getImage());
	}

	/**
	 * Converts a {@link Image} into a {@link BufferedImage} using
	 * {@link BufferedImage#TYPE_INT_ARGB_PRE} type. If the image is already a
	 * {@link BufferedImage} then the same instance is returned.
	 * 
	 * @param image the image to convert to a buffered image.
	 * @return the converted image or the same instance if already a
	 * {@link BufferedImage}.
	 */
	public static BufferedImage toBufferedImage(Image imageIn) {
		return toBufferedImage(imageIn, BufferedImage.TYPE_INT_ARGB_PRE);
	}

	/**
	 * Converts a {@link Image} into a {@link BufferedImage}. If the image 
	 * is already a {@link BufferedImage} then the same instance is returned.
	 *
	 * @param image the image to convert to a buffered image.
	 * @param imageType the image type to use.
	 * @return the converted image or the same instance if already a
	 * {@link BufferedImage}.
	 */
	public static BufferedImage toBufferedImage(
			Image image,
			int imageType)
	{
		if (image instanceof BufferedImage) {
			return (BufferedImage) image;
		}
		final BufferedImage buffImage = new BufferedImage(
				image.getWidth(null),
				image.getHeight(null),
				imageType);
		final Graphics2D gfx = buffImage.createGraphics();
		gfx.drawImage(image, 0, 0, null);
		return buffImage;
	}

	/**
	 * This method returns true if the specified image has transparent pixels
	 *
	 * @param image
	 * @return
	 */
	public static boolean hasAlpha(Image image) {
		// If buffered image, the colour model is readily available
		if (image instanceof BufferedImage) {
			return ((BufferedImage) image).getColorModel().hasAlpha();
		}

		// Use a pixel grabber to retrieve the image's colour model;
		// grabbing a single pixel is usually sufficient
		PixelGrabber pg = new PixelGrabber(image, 0, 0, 1, 1, false);
		try {
			pg.grabPixels();
		} catch (InterruptedException e) {
		}

		// Get the image's colour model
		ColorModel cm = pg.getColorModel();
		return cm.hasAlpha();
	}

	/**
	 * TODO
	 * 
	 * @param comp
	 * @param type
	 * @return
	 */
	public static <T extends JComponent> T find(Container comp,
			Class<T> type)
	{
		if (comp.getClass() == type) {
			return type.cast(comp);
		}
		for (Component child : comp.getComponents()) {
			final T found = find((Container) child, type);
			if (found != null) {
				return found;
			}
		}
		return null;
	}

	/**
	 * Populates the toolbar with actions.
	 * 
	 * @param toolBar
	 * @param actionMap
	 * @param list
	 */
	public static void populateToolbar(JToolBar toolBar, ActionMap actionMap, List<Object> list) {
		for (Object object : list) {
			if (object instanceof Action) {
				toolBar.add((Action) object);
			} else {
				toolBar.add(actionMap.get(object));
			}
		}
	}

	/**
	 * Attempts to locate the {@link Component} within the {@link Container}.
	 * Returns {@code -1} if unable to locate in this {@link Container}. 
	 * 
	 * @param container
	 * @param child
	 * @return
	 */
	public static int indexOf(Container container, Component child) {
		final int count = container instanceof JMenu ?
				((JMenu) container).getMenuComponentCount() :
					container.getComponentCount();
		
		for (int i = 0; i < count; i++) {
			if (container instanceof JMenu) {
				if (((JMenu) container).getMenuComponent(i).equals(child)) {
					return i;
				}
			} else if (container.getComponent(i).equals(child)) {
				return i;
			}
		}
		return -1;
	}

	/**
	 * 
	 * TODO
	 * @param header
	 * @return
	 */
	public static JXPanel wrap(JXHeader header) {
		final JXPanel panel = new JXPanel(new BorderLayout());
		panel.add(header, BorderLayout.CENTER);
		panel.add(new JSeparator(), BorderLayout.SOUTH);
		return panel;
	}
}
