package civvi.osgi.desktop.swingx;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.LayoutManager;
import java.io.Serializable;

import javax.swing.JViewport;

/**
 * Creates a layout that will wrap the contents. This is loosely based on the
 * {@link java.awt.GridLayout}.
 * 
 * TODO dan.siviter - fitWidth flag functionality, this is currently on by
 * default. I don't have time to put the code in as this does what I currently
 * need it to!
 * 
 * @author <a href="mailto:dansiviter@gmail.com">Dan Siviter</a>
 * @since 15th October 2007
 */
public class WrapLayout implements LayoutManager, Serializable {
	private static final long serialVersionUID = -110422915315839594L;
	
	private int hGap;
	private int vGap;
	private boolean fitHeight;

	private int rows;
	private int cols;

	/**
	 * Creates a layout that will wrap the contents based on prefered and
	 * minimum sizes. By default this will have zero horizontal and vertical
	 * gaps and will fit to height.
	 */
	public WrapLayout() {
		this(0, 0, true);
	}

	/**
	 * Creates a layout that will wrap the contents based on prefered and
	 * minimum sizes.
	 * 
	 * @param hGap
	 *            the horizontal gap
	 * @param vGap
	 *            the vertical gap
	 * @param fitHeight
	 *            if {@code true} this fits the components to their row height.
	 */
	public WrapLayout(int hGap, int vGap, boolean fitHeight) {
		setHgap(hGap);
		setVgap(vGap);
		setFitHeight(fitHeight);
	}

	/**
	 * @return the horizontal gap between components
	 */
	public int getHgap() {
		return this.hGap;
	}

	/**
	 * @param hGap
	 *            the horizontal gap between components
	 */
	public void setHgap(int hGap) {
		this.hGap = hGap;
	}

	/**
	 * @return the vertical gap between components
	 */
	public int getVgap() {
		return this.vGap;
	}

	/**
	 * @param vGap
	 *            the vertical gap between components
	 */
	public void setVgap(int vGap) {
		this.vGap = vGap;
	}

	/**
	 * @return if {@code true} this fits the components to their row height.
	 */
	public boolean isFitHeight() {
		return this.fitHeight;
	}

	/**
	 * @param fitHeight
	 *            if {@code true} this fits the components to their row height.
	 */
	public void setFitHeight(boolean fitHeight) {
		this.fitHeight = fitHeight;
	}

	/**
	 * @see java.awt.LayoutManager#addLayoutComponent(java.lang.String,
	 *      java.awt.Component)
	 */
	public void addLayoutComponent(String name, Component comp) {
		// do nothing!
	}

	/**
	 * @see java.awt.LayoutManager#removeLayoutComponent(java.awt.Component)
	 */
	public void removeLayoutComponent(Component comp) {
		// do nothing!
	}

	/**
	 * @see java.awt.LayoutManager#preferredLayoutSize(java.awt.Container)
	 */
	public Dimension preferredLayoutSize(Container parent) {
		synchronized (parent.getTreeLock()) {
			Insets insets = parent.getInsets();
			int ncomponents = parent.getComponentCount();

			this.cols = calcColumns(parent, true);
			this.rows = (ncomponents + this.cols - 1) / this.cols;

			int w = 0;

			for (int i = 0; i < ncomponents; i++) {
				Component comp = parent.getComponent(i);
				Dimension d = comp.getPreferredSize();
				if (w < d.width) {
					w = d.width;
				}
			}

			int[] rowHeights = calcHeights(parent, this.cols, true);
			int h = 0;

			for (int i = 0; i < rowHeights.length; i++) {
				h += rowHeights[i];
			}

			return new Dimension(insets.left + insets.right + this.cols * w
					+ (this.cols - 1) * this.hGap, insets.top + insets.bottom
					+ h + (this.rows - 1) * this.vGap);
		}
	}

	/**
	 * @see java.awt.LayoutManager#minimumLayoutSize(java.awt.Container)
	 */
	public Dimension minimumLayoutSize(Container parent) {
		synchronized (parent.getTreeLock()) {
			Insets insets = parent.getInsets();
			int ncomponents = parent.getComponentCount();

			this.cols = calcColumns(parent, false);
			this.rows = (ncomponents + this.cols - 1) / this.cols;

			int w = 0;
			for (int i = 0; i < ncomponents; i++) {
				Component comp = parent.getComponent(i);
				Dimension d = comp.getMinimumSize();
				if (w < d.width) {
					w = d.width;
				}
			}

			int[] rowHeights = calcHeights(parent, this.cols, true);
			int h = 0;

			for (int i = 0; i < rowHeights.length; i++) {
				h += rowHeights[i];
			}

			return new Dimension(insets.left + insets.right + this.cols * w
					+ (this.cols - 1) * this.hGap, insets.top + insets.bottom
					+ h + (this.rows - 1) * this.vGap);
		}
	}

	/**
	 * @see java.awt.LayoutManager#layoutContainer(java.awt.Container)
	 */
	public void layoutContainer(Container parent) {
		synchronized (parent.getTreeLock()) {
			Insets insets = parent.getInsets();
			int ncomponents = parent.getComponentCount();
			boolean ltr = parent.getComponentOrientation().isLeftToRight();

			int[] rowHeights = calcHeights(parent, this.cols, true);

			if (ncomponents == 0) {
				return;
			}
			int w = parent.getBounds().width - (insets.left + insets.right);
			w = (w - (this.cols - 1) * this.hGap) / this.cols;

			if (ltr) {
				for (int c = 0, x = insets.left; c < this.cols; c++, x += w
						+ this.hGap) {
					for (int r = 0, y = insets.top; r < this.rows; r++, y += rowHeights[r - 1]
							+ this.vGap) {
						int i = r * this.cols + c;
						if (i < ncomponents) {
							final Component comp = parent.getComponent(i);
							comp.setBounds(x, y, w,
									this.fitHeight ? rowHeights[r] : comp
											.getPreferredSize().height);
						}
					}
				}
			} else {
				for (int c = 0, x = parent.getBounds().width - insets.right - w; c < this.cols; c++, x -= w
						+ this.hGap) {
					for (int r = 0, y = insets.top; r < this.rows; r++, y += rowHeights[r - 1]
							+ this.vGap) {
						int i = r * this.cols + c;
						if (i < ncomponents) {
							final Component comp = parent.getComponent(i);
							comp.setBounds(x, y, w,
									this.fitHeight ? rowHeights[r] : comp
											.getPreferredSize().height);
						}
					}
				}
			}
		}
	}

	/**
	 * Calcuates the row heights using the child components for the given parent
	 * container.
	 * 
	 * @param parent
	 *            the parent container to extract child components from.
	 * @param preferredSize
	 *            if {@code true} it will use prefered size of the child
	 *            components, otherwise it will use minimum.
	 * @return an array that contains all the row heights in pixels.
	 */
	private int[] calcHeights(Container parent, int cols, boolean preferredSize) {
		final int[] heights = new int[parent.getComponentCount() / cols
				+ (parent.getComponentCount() % cols > 0 ? 1 : 0)];

		int compIndex = 0;
		for (int i = 0; i < heights.length; i++) {
			for (int j = 0; j < cols; j++) {
				if (compIndex >= parent.getComponentCount())
					break;

				final Dimension size;

				if (preferredSize)
					size = parent.getComponent(compIndex).getPreferredSize();
				else
					size = parent.getComponent(compIndex).getMinimumSize();

				if (size.height > heights[i])
					heights[i] = size.height;

				compIndex++;
			}
		}
		return heights;
	}

	/**
	 * Calculates the requied amound of columns based on the child components of
	 * the given parent container.
	 * 
	 * @param parent
	 *            the parent container to extract child components from.
	 * @param preferedSize
	 *            if {@code true} it will use prefered size of the child
	 *            components, otherwise it will use minimum.
	 * @return the required amount of columns.
	 */
	private int calcColumns(Container parent, boolean preferedSize) {
		if (parent.getComponentCount() == 0)
			return 1;

		final Dimension available;

		if (parent.getParent() instanceof JViewport)
			available = ((JViewport) parent.getParent()).getExtentSize();
		else
			available = parent.getSize();

		int largestWidth = 0;

		for (int i = 0; i < parent.getComponentCount(); i++) {
			final Dimension size;

			if (preferedSize)
				size = parent.getComponent(i).getPreferredSize();
			else
				size = parent.getComponent(i).getMinimumSize();

			if (size.width > largestWidth)
				largestWidth = size.width;
		}

		final int totalWidth = parent.getComponentCount() * largestWidth
				+ (parent.getComponentCount() - 1) * this.hGap;
		final int nCols = available.width
				/ (totalWidth / parent.getComponentCount());
		return nCols > 1 ? nCols : 1;
	}
}
