package civvi.osgi.desktop.swingx.docking;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.Stroke;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.event.MouseEvent;
import java.io.IOException;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.JTabbedPane;
import javax.swing.TransferHandler;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.MouseInputAdapter;

import org.jdesktop.swingx.MultiSplitLayout;
import org.jdesktop.swingx.MultiSplitLayout.Leaf;
import org.jdesktop.swingx.plaf.basic.core.DragRecognitionSupport;
import org.jdesktop.swingx.plaf.basic.core.DragRecognitionSupport.BeforeDrag;

import civvi.osgi.desktop.swingx.JXDockingPane;
import civvi.osgi.desktop.swingx.JXTabbedPane;

/**
 * An enhanced {@link JTabbedPane} that is dockable within the framework.
 * 
 * TODO Additional tab controls (i.e. dropdowns in tab bar).
 * 
 * @author <a href="mailto:dansiviter@gmail.com">Dan Siviter</a>
 * @since v1.0.0 [28th Jan 2008]
 */
public abstract class AbstractDock extends JTabbedPane {
	private static final long serialVersionUID = -7300790952177893424L;
	public static final DataFlavor DOCKABLE_FLAVOR = new DataFlavor(
			DataFlavor.javaJVMLocalObjectMimeType,
			"AbstractDock.tab");
	protected final Logger log = Logger.getLogger(getClass().getName());

	private DragPreview dragPreview = new DragPreview();

	/**
	 * Constructs a dock to hold {@link AbstractDockable}s and has a default set
	 * of dockables.
	 * 
	 * @param dockables
	 *            the default set of dockables to add to the dock.
	 */
	public AbstractDock(AbstractDockable... dockables) {
		this();
		for (AbstractDockable dockable : dockables) {
			dock(dockable);
		}
	}

	/**
	 * Creates an empty {@link JXTabbedPane} with a default tab placement of
	 * {@link JTabbedPane.TOP}.
	 * 
	 * @see #addTab
	 */
	public AbstractDock() {
		super(TOP);

		addChangeListener(new ChangeHandler());
		setTransferHandler(new DockableTransferHandler());
		final MouseInputHandler listener = new MouseInputHandler();
		addMouseListener(listener);
		addMouseMotionListener(listener);
	}

	/**
	 * Returns the index of the tab at the given point.
	 * 
	 * @param p
	 *            the point to test.
	 * @return the index of the tab, or {@code -1} if there is no tab.
	 */
	int indexOfTab(Point p) {
		for (int i = 0; i < getTabCount(); i++) {
			if (getUI().getTabBounds(this, i).contains(p))
				return i;
		}

		// Component 1 should always bet TabContainer
		if (isInTabPane(p)/* || getBounds().contains(p) */)
			return getTabCount();

		return -1;
	}

	/**
	 * Returns if the point is within the pane containing the tabs.
	 * 
	 * @param p
	 * @return
	 */
	boolean isInTabPane(Point p) {
		final JPanel tabPane = getTabPane();
		return tabPane != null ? tabPane.getBounds().contains(p) : false;
	}

	/**
	 * 
	 * TODO
	 * @return
	 */
	JPanel getTabPane() {
		if (getTabCount() > 0) {
			for (int i = 0; i < getComponentCount(); i++) {
				if ("javax.swing.plaf.basic.BasicTabbedPaneUI$TabContainer"
						.equals(getComponent(i).getClass().getName())) {
					return (JPanel) getComponent(i);
				}
			}
		}

		return null;
	}
	
	/**
	 * Returns the split direction for the give point. </p> The idea is to split
	 * the dock into 5 distinct areas: North, South, East, West and Central.
	 * When North, South, East and West are detected then it will attempt to
	 * split in that direction. When Center is detected, then {@code null} is
	 * returned so nothing is done. The layout is show in this rather rough
	 * diagram below.
	 * 
	 * <pre>
	 *  _______
	 * |\  N  /|
	 * | \___/ |
	 * |E| C |W|
	 * | /¯¯¯\ |
	 * |/  S  \|
	 *  ¯¯¯¯¯¯¯
	 * </pre>
	 * 
	 * XXX there must be a simpler calculation than this?!
	 * 
	 * @param dock
	 *            the parent dock to split from.
	 * @param p
	 *            the point to test.
	 * @return the direction of the split.
	 */
	public DockPosition splitDirectionOfPoint(Point p) {
		if (!contains(p)) return null;
		
		if (isInTabPane(p)) return DockPosition.TAB;

		final Rectangle dimensions = getBounds();

		final Polygon north = new Polygon();
		north.addPoint(dimensions.x, dimensions.y);
		north.addPoint(dimensions.x + dimensions.width, dimensions.y);
		north.addPoint((int) dimensions.getCenterX(), (int) dimensions
				.getCenterY());
		if (north.contains(p)) return DockPosition.NORTH;
		
		final Polygon south = new Polygon();
		south.addPoint(dimensions.x, dimensions.y + dimensions.height);
		south.addPoint(dimensions.x + dimensions.width, dimensions.y
				+ dimensions.height);
		south.addPoint((int) dimensions.getCenterX(), (int) dimensions
				.getCenterY());
		if (south.contains(p)) return DockPosition.SOUTH;
		
		final Polygon west = new Polygon();
		west.addPoint(dimensions.x, dimensions.y);
		west.addPoint(dimensions.x, dimensions.y + dimensions.height);
		west.addPoint((int) dimensions.getCenterX(), (int) dimensions
				.getCenterY());
		if (west.contains(p)) return DockPosition.WEST;
		
		final Polygon east = new Polygon();
		east.addPoint(dimensions.x + dimensions.width, dimensions.y);
		east.addPoint(dimensions.x + dimensions.width, dimensions.y
				+ dimensions.height);
		east.addPoint((int) dimensions.getCenterX(), (int) dimensions
				.getCenterY());
		if (east.contains(p)) return DockPosition.EAST;

		return null; // nothing to see here!
	}

	/**
	 * @return the dockable selected within this dock.
	 */
	public AbstractDockable getSelectedDockable() {
		return (AbstractDockable) getSelectedComponent();
	}

	/**
	 * Used to determine if the dockable can be docked to this. By default this
	 * returns {@code true} but override if you require specific functionality.
	 * 
	 * @param dockable
	 *            the dockable to test.
	 * @return if {@code true} if the test dockable can be docked to this.
	 */
	public boolean isDockable(AbstractDockable dockable) {
		return true;
	}

	/**
	 * Docks the given dockable with this dock.
	 * 
	 * @param dockable
	 *            the dockable to add to the end of the dock.
	 * @return {@code true} if the dock completed successfully.
	 * @see #dock(int, AbstractDockable)
	 */
	public final boolean dock(AbstractDockable dockable) {
		return dock(getTabCount(), dockable);
	}

	/**
	 * Docks the given dockable with this dock at the given index. If the
	 * dockable already has a parent dock then it will be removed from it and
	 * docked to this.
	 * 
	 * @param index
	 *            the index in where to dock the dockable.
	 * @param dockable
	 *            the component to dock.
	 * @return {@code true} if the dock completed successfully.
	 */
	public boolean dock(int index, AbstractDockable dockable) {
		if (!isDockable(dockable)) // check we can dock before continuing
			return false;

		if (dockable.isDocked()) {
			if (dockable.getDock() == this) {
				// move tab if index is different
				final int sourceIndex = indexOfComponent(dockable);
				// check if we need to actually dock
				if (sourceIndex == index)
					return true;
				// if source tab appears before the target tab, then we
				// shift the list down by one
				if (index > sourceIndex)
					index--;
			}
			dockable.getDock().remove(dockable);
		}

		getDockingPane().getUI().installUI(dockable);
		insertTab(dockable.getTitle(), dockable.getSmallIcon(), dockable,
				dockable.getTooltip(), index);

		// set new tab if one exists!
		if (dockable.getTabComponent() != null) {
			setTabComponentAt(index, dockable.getTabComponent());
		}
		setSelectedComponent(dockable);
		return true;
	}

	/**
	 * @param dockable
	 *            the dockable to remove.
	 */
	public void remove(AbstractDockable dockable) {
		super.remove(dockable);
		if (getTabCount() == 0 && isCloseableOnEmpty()) {
			final JXDockingPane dockingPane = getDockingPane();
			dockingPane.remove(getLeaf());
			dockingPane.remove(this);
			dockingPane.getUI().uninstallUI(dockable);
		}
	}

	/**
	 * @return the leaf for this dock.
	 */
	public Leaf getLeaf() {
		final JXDockingPane dockingPane = (JXDockingPane) getParent();
		if (dockingPane != null) {
			final MultiSplitLayout layout = dockingPane.getMultiSplitLayout();
			return dockingPane.getLeaf(layout.getNameForComponent(this));
		}
		return null;
	}

	/**
	 * @return {@code true} if this dock can be closed if empty.
	 */
	public boolean isCloseableOnEmpty() {
		return true;
	}

	/**
	 * @return the docking pane.
	 */
	public JXDockingPane getDockingPane() {
		return (JXDockingPane) getParent();
	}

	@Override
	public void paint(Graphics g) {
		
		super.paint(g);
		
		if (this.dragPreview != null) {
			this.dragPreview.drawPreview((Graphics2D) g, this); 
		}
	}
	
	@Override
	protected String paramString() {
		// put at front 'cause it's easier to see! ;o)
		return getLeaf() + "," + super.paramString();
	}

	// --- Inner Classes ---

	/**
	 * Handles drawing of the drag previews. By default this uses an XOR type
	 * pattern.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [28th Jan 2008]
	 */
	public static class DragPreview {
		protected DockPosition position;
		protected int tabIndex = -1;
	
		
		public void setRequired(DockPosition position, int tabIndex) {
			this.position = position;
			this.tabIndex = tabIndex;
		}
		
		public void clear() {
			this.position = null;
			this.tabIndex = -1;
		}
		
		/**
		 * TODO
		 * 
		 * @param gfx
		 * @param dock
		 * @param comp
		 */
		public void drawPreview(
				Graphics2D gfx,
				AbstractDock dock)
		{
			if (position == null || position != DockPosition.TAB)
				return;

			final float[] pattern = { 1.0f, 1.0f };
			final Stroke stroke = new BasicStroke(2, BasicStroke.CAP_BUTT,
					BasicStroke.JOIN_ROUND, 1f, pattern, 0f);
			gfx.setStroke(stroke);
			gfx.setColor(Color.BLACK);
			gfx.setXORMode(Color.WHITE);

			if (tabIndex < dock.getTabCount()) {
				gfx.draw(dock.getUI().getTabBounds(dock, tabIndex));
			} else {
//				final Rectangle tabPaneBounds = dock.getTabPane().getBounds();
				final Rectangle lastTabBounds = dock.getUI().getTabBounds(dock, dock.getTabCount() - 1);
				final Rectangle remainingBounds = new Rectangle(
						(int) lastTabBounds.getMaxX(),
						lastTabBounds.y,
						lastTabBounds.width,
						lastTabBounds.height
				);
				gfx.draw(remainingBounds);
			}
		}
	}

	/**
	 * Handles changes to the selected tab.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [28th Jan 2008]
	 */
	private class ChangeHandler implements ChangeListener {
		@Override
		public void stateChanged(ChangeEvent e) {
			getDockingPane().setSelectedDockable(getSelectedDockable());
		}
	}

	/**
	 * Handles the Drag'n'Drop operations.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [28th Jan 2008]
	 */
	private class DockableTransferHandler extends TransferHandler {
		private static final long serialVersionUID = 1L;

		@Override
		public int getSourceActions(JComponent c) {
			return MOVE;
		}

		@Override
		protected Transferable createTransferable(JComponent c) {
			return ((AbstractDock) c).getSelectedDockable();
		}

		@Override
		protected void exportDone(
				JComponent source,
				Transferable data,
				int action)
		{
			super.exportDone(source, data, action);
			if (dragPreview != null) {
				dragPreview.clear();
				repaint();
			}
		}

		@Override
		public boolean canImport(TransferSupport support) {
			try {
				if (!support.isDataFlavorSupported(DOCKABLE_FLAVOR)) return false;

				final DropLocation dl = support.getDropLocation();
				final DockPosition position = splitDirectionOfPoint(dl
						.getDropPoint());
				if (position != null) {
					final AbstractDockable dockable = 
						(AbstractDockable) support.getTransferable().getTransferData(
								DOCKABLE_FLAVOR);

					if (position == DockPosition.TAB && !isDockable(dockable))
						return false;
					
					if (dragPreview != null) {
						dragPreview.setRequired(position, indexOfTab(dl.getDropPoint()));
						repaint();
					}
					return true;
				}
			} catch (UnsupportedFlavorException e) {
				log.log(Level.WARNING, "Unsupported flavor!", e);
			} catch (IOException e) {
				log.log(Level.WARNING, "Unable to transfer!", e);
			}
			return false;
		}

		@Override
		public boolean importData(TransferSupport support) {
			try {
				final AbstractDockable dockable = 
					(AbstractDockable) support.getTransferable().getTransferData(
							DOCKABLE_FLAVOR);
				final DockPosition dir = splitDirectionOfPoint(support
						.getDropLocation().getDropPoint());

				if (dir == null)
					return false;

				if (dir == DockPosition.TAB) {
					final int targetIndex = indexOfTab(support
							.getDropLocation().getDropPoint());
					getDockingPane().dock(dockable, getLeaf().getName(),
							targetIndex);
					return true;
				}

				getDockingPane().dock(dockable, getLeaf().getName(), dir);
				return true;
			} catch (UnsupportedFlavorException e) {
				log.log(Level.WARNING, "Unsupported flavor!", e);
			} catch (IOException e) {
				log.log(Level.WARNING, "Unable to transfer!", e);
			}
			return false;
		}
	}

	/**
	 * Handles input from the moust.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [28th Jan 2008]
	 */
	private class MouseInputHandler extends MouseInputAdapter
	implements BeforeDrag
	{
		private boolean dragStarted;

		@Override
		public void dragStarting(MouseEvent me) {
			this.dragStarted = true;
		}

		@Override
		public void mouseClicked(MouseEvent e) {
			if (getUI().tabForCoordinate(
					AbstractDock.this,
					e.getX(),
					e.getY()) != -1)
			{
				getDockingPane().setSelectedDockable(getSelectedDockable());

				if (e.getClickCount() > 1) {
					// XXX double click functionality?!
				}
			}
		}
		
		@Override
		public void mousePressed(MouseEvent e) {
			this.dragStarted = false;
			if (DragRecognitionSupport.mousePressed(e)) e.consume();
		}

		@Override
		public void mouseReleased(MouseEvent e) {
			if (this.dragStarted) e.consume();
			DragRecognitionSupport.mouseReleased(e);
		}

		@Override
		public void mouseDragged(MouseEvent e) {
			if (this.dragStarted || DragRecognitionSupport.mouseDragged(e, this))
				e.consume();
		}
	}
	
	/**
	 * Defines the directions in which the dock can be positioned.
	 *
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [3 Dec 2007]
	 */
	public enum DockPosition {
		/** Split Direction: North */
		NORTH,
		/** Split Direction: South */
		SOUTH,
		/** Split Direction: East */
		EAST,
		/** Split Direction: West */
		WEST,
		/** Dock as tab: West */
		TAB
	}
}