package civvi.osgi.desktop.swingx.docking;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.event.ActionEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;

import javax.swing.AbstractAction;
import javax.swing.Icon;
import javax.swing.JComponent;
import javax.swing.JToolBar;
import javax.swing.UIManager;

import org.jdesktop.swingx.JXPanel;

import civvi.osgi.desktop.swingx.JXDockingPane;

/**
 * Represents a component that can be docked within a {@link JXDockingPane}.
 * 
 * @author <a href="mailto:dansiviter@gmail.com">Dan Siviter</a>
 * @since 20 Nov 2007
 */
public abstract class AbstractDockable extends JXPanel implements Transferable {
	private static final long serialVersionUID = -480795346194068471L;
	public static final String CLOSE_ACTION_KEY = "close";

	protected final Logger log = Logger.getLogger(getClass().getName());
	private final Object id;
	private final Map<Object, Object> properties;
	private Component tabComponent;
	private boolean closable = true;


	/**
	 * Constructs a new dockable with the given title.
	 *
	 * @param title the title of the dockable.
	 */
	public AbstractDockable(Object id, String title) {
		super(new BorderLayout());
		this.id = id;
		this.properties = new HashMap<Object, Object>();
		getActionMap().put(CLOSE_ACTION_KEY, new CloseAction(this));
		setOpaque(false);
		setTitle(title);
	}

	@Override
	public DataFlavor[] getTransferDataFlavors() {
		return new DataFlavor[] { AbstractDock.DOCKABLE_FLAVOR };
	}

	@Override
	public Object getTransferData(DataFlavor flavor)
	throws UnsupportedFlavorException
	{
		if (isDataFlavorSupported(flavor)) {
			return this;
		}
		throw new UnsupportedFlavorException(flavor);
	}

	@Override
	public boolean isDataFlavorSupported(DataFlavor flavor) {
		for (DataFlavor dataFlavor : getTransferDataFlavors()) {
			if (dataFlavor.equals(flavor))
				return true;
		}
		return false;
	}

	/**
	 * @return the unique identifier for this dockable.
	 */
	public Object getId() {
		return id;
	}

	/**
	 * @return the title of the dockable.
	 */
	public String getTitle() {
		return (String) this.properties.get("title");
	}

	/**
	 * @param title the title of the dockable to set.
	 */
	public void setTitle(String title) {
		final String oldValue = getTitle();
		this.properties.put("title", title);
		firePropertyChange("title", oldValue, getTitle());
	}

	/**
	 * @param content the content component.
	 */
	protected void setContent(JComponent content) {
		add(content, BorderLayout.CENTER);
	}
	
	/**
	 * @param content the content component.
	 */
	protected void setToobar(JToolBar toolbar) {
		add(toolbar, BorderLayout.NORTH);
	}

	/**
	 * @return the tab tooltip.
	 */
	public String getTooltip() {
		return (String) this.properties.get("tooltip");
	}

	/**
	 * @param tooltip the tool-tip to set.
	 */
	public void setTooltip(String tooltip) {
		final String oldValue = getTooltip();
		this.properties.put("tooltip", tooltip);
		firePropertyChange("tooltip", oldValue, getTooltip());
	}

	/**
	 * @return the tab small icon.
	 */
	public Icon getSmallIcon() {
		return (Icon) this.properties.get("smallIcon");
	}

	/**
	 * @param smallIcon the small icon to set.
	 */
	public void setSmallIcon(Icon smallIcon) {
		final Icon oldValue = getSmallIcon(); 
		this.properties.put("smallIcon", smallIcon);
		firePropertyChange("smallIcon", oldValue, getSmallIcon());
	}

	/**
	 * @return the tab large icon.
	 */
	public Icon getLargeIcon() {
		return (Icon) this.properties.get("largeIcon");
	}

	/**
	 * @param largeIcon the icon to set.
	 */
	public void setLargeIcon(Icon largeIcon) {
		final Icon oldValue = getSmallIcon(); 
		this.properties.put("largeIcon", largeIcon);
		firePropertyChange("largeIcon", oldValue, getSmallIcon());
	}

	/**
	 * @return the actions used in the toolbar. This should never be {@code null}!
	 */
	public Object[] getToolbarActionKeys() {
		return new String[0];
	}

	/**
	 * @return the component placed into the tab.
	 */
	public Component getTabComponent() {
		return this.tabComponent;
	}

	/**
	 * @param component the component to be placed into the tab.
	 */
	public void setTabComponent(Component tabComponent) {
		Component oldValue = getTabComponent();
		this.tabComponent = tabComponent;
		firePropertyChange("tabComponent", oldValue, getTabComponent());
	}

	/**
	 * @return {@code true} if this dockable is associated with a dock.
	 */
	public boolean isDocked() {
		return getParent() != null;
	}

	/**
	 * @return the parent dock.
	 */
	public AbstractDock getDock() {
		return getParent() != null ? (AbstractDock) getParent() : null;
	}

	/**
	 * @return if {@code true} then the user is unable to close the
	 * dockable.
	 */
	public boolean isClosable() {
		return this.closable;
	}

	/**
	 * @param closable if {@code true} then the user is unable to close the
	 * dockable.
	 */
	public void setClosable(boolean closable) {
		final boolean oldValue = isClosable();
		this.closable = closable;
		firePropertyChange("closeable", oldValue, isClosable());
	}

	/**
	 * Closes the dockable. Override this method to perform further cleanup
	 * duties.
	 */
	protected void close() {
		if (isClosable()) {
			getDock().remove(this);
		}
	}

	
	// --- Inner Classes ---
	
	/**
	 * The action used for closing the dockable.
	 *
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since 18 Dec 2007
	 */
	private static class CloseAction extends AbstractAction {
		private static final long serialVersionUID = 4347906530900347102L;

		private final AbstractDockable dockable;
		
		public CloseAction(AbstractDockable dockable) {
			super(UIManager.getString(JXDockingPane.class.getCanonicalName() + ".closeAction.name"));
			this.dockable = dockable;
			putValue(SMALL_ICON, UIManager.getIcon(JXDockingPane.class.getCanonicalName() + ".closeAction.icon"));
			putValue(SMALL_ICON + "Rollover", UIManager.getIcon(JXDockingPane.class.getCanonicalName() + ".closeAction.rolloverIcon"));
			dockable.addPropertyChangeListener("closeable", new PropertyChangeListener() {
				@Override
				public void propertyChange(PropertyChangeEvent evt) {
					setEnabled((Boolean) evt.getNewValue());
				}
			});
			setEnabled(dockable.isClosable());
		}

		/**
		 * {@inheritDoc}
		 */
		public void actionPerformed(ActionEvent e) {
			this.dockable.close();
		}
	}
}
