package civvi.osgi.desktop.swingx.docking;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.xml.bind.JAXB;

import org.jdesktop.beans.AbstractBean;
import org.jdesktop.swingx.MultiSplitLayout;
import org.jdesktop.swingx.MultiSplitLayout.Leaf;
import org.jdesktop.swingx.MultiSplitLayout.Node;

import civvi.osgi.desktop.swingx.JXDockingPane;
import civvi.osgi.desktop.swingx.docking.perspective.Perspective;
import civvi.osgi.desktop.swingx.docking.perspective.PerspectiveFactory;
import civvi.osgi.desktop.swingx.docking.perspective.PerspectiveLeaf;
import civvi.osgi.desktop.swingx.docking.perspective.PerspectiveView;
import civvi.osgi.desktop.swingx.docking.perspective.Perspectives;


/**
 * Handles the state of the docking framework. Basically, anything that doesn't
 * have to be in {@link JXDockingPane}. 
 * 
 * @author <a href="mailto:dansiviter@gmail.com">Dan Siviter</a>
 * @since 22th November 2007
 */
public class DockManager extends AbstractBean {
	/** A key to the default perspective */
	public static final String DEFAULT_PERSPECTIVE = "default";
	
	protected final Logger log = Logger.getLogger(getClass().getName());

	protected final JXDockingPane dockingPane;
	private final Map<Object, AbstractDockable> dockables;
	private final Map<String, Perspective> perspectivesMap;
	
	private DockableFactory dockableFactory;
	private PerspectiveFactory perspectiveFactory;
	private String perspectiveKey;

	/**
	 * Singleton that requires a rootWindow.
	 * 
	 * @param rootWindow
	 */
	public DockManager(JXDockingPane dockingPane) {
		this.dockingPane = dockingPane;
		this.dockables = new HashMap<Object, AbstractDockable>();
		this.perspectivesMap = new HashMap<String, Perspective>();
		setDockableFactory(new DefaultDockableFactory());
		setPerspectiveFactory(new DefaultPerspectiveFactory());
		addPropertyChangeListener(
				"currentPerspectiveKey",
				new PerspectiveChangeHandler());
	}

	/**
	 * @return the dockableFactory.
	 */
	public DockableFactory getDockableFactory() {
		return this.dockableFactory;
	}

	/**
	 * @param dockableFactory the dockableFactory to set.
	 */
	public void setDockableFactory(DockableFactory dockableFactory) {
		final DockableFactory oldValue = getDockableFactory();
		this.dockableFactory = dockableFactory;
		firePropertyChange("dockableFactory", oldValue, getDockableFactory());
	}

	/**
	 * @return the perspectiveFactory.
	 */
	public PerspectiveFactory getPerspectiveFactory() {
		return perspectiveFactory;
	}

	/**
	 * @param perspectiveFactory the perspectiveFactory to set.
	 */
	public void setPerspectiveFactory(PerspectiveFactory perspectiveFactory) {
		final PerspectiveFactory oldValue = getPerspectiveFactory();
		this.perspectiveFactory = perspectiveFactory;
		firePropertyChange("perspectiveFactory", oldValue, getPerspectiveFactory());
	}

	/**
	 * @param key the key of the {@link AbstractDockable} to dock.
	 * @return
	 */
	public void show(Object key) {
		show(key, false);
	}

	/**
	 * @param key the key of the {@link AbstractDockable} to dock.
	 * @param create if {@code true} if a view instance isn't found one will
	 * be created.
	 * @return
	 */
	public void show(Object key, boolean create) {
		dock(getDockable(key, create), null);
	}

	/**
	 * @param dockable the dockable to dock.
	 * @param position
	 */
	public void dock(AbstractDockable dockable, String position) {
		if (!dockable.isDocked()) {
			if (this.log.isLoggable(Level.FINE)) {
				this.log.log(Level.FINE,
						"Showing dockable. [dockable={0}]",
						new Object[] { dockable });
			}
			if (position != null) {
				this.dockingPane.dock(dockable, position);
			} else {
				this.dockingPane.dock(dockable);
			}
		}
	}

	/**
	 * TODO
	 * 
	 * @return
	 */
	public JXDockingPane getDockingPane() {
		return this.dockingPane;
	}
	
	/**
	 * @param perspectiveKey the key of the current perspective.
	 */
	public void setPerspectiveKey(String perspectiveKey) {
		if (this.perspectiveKey == null || 
				!this.perspectiveKey.equals(perspectiveKey))
		{
			if (this.log.isLoggable(Level.FINE)) {
				this.log.log(Level.FINE,
						"Changing perspective. [perspectiveKey={0}]",
						new Object[] { perspectiveKey });
			}
			
			final String oldValue = getPerspectiveKey();
			this.perspectiveKey = perspectiveKey;
			firePropertyChange("currentPerspectiveKey",
					oldValue, getPerspectiveKey());
		}
	}
	
	/**
	 * @return the current perspective.
	 */
	public String getPerspectiveKey() {
		return this.perspectiveKey;
	}

	/**
	 * 
	 * TODO
	 *
	 * @param key
	 * @return
	 */
	public Class<? extends AbstractDockable> getDockableClass(Object key) {
		return getDockableFactory().getClass(key);
	}
	
	/**
	 * 
	 * @param key
	 * @param create
	 * @return
	 */
	public AbstractDockable getDockable(Object key, boolean create) {
		AbstractDockable dockable = this.dockables.get(key);
		if (create && dockable == null) {
			dockable = create(key);
		}
		return dockable;
	}
	
	/**
	 * TODO
	 * 
	 * @param id
	 * @return
	 */
	public AbstractDockable create(Object id) {
		if (this.dockables.containsKey(id)) {
			throw new IllegalArgumentException(String.format(
					"Dockable already exists for key! [key%s]",
					id));
		}
		final AbstractDockable dockable = getDockableFactory().create(id);
		this.dockables.put(dockable.getId(), dockable);
		return dockable;
	}
	
	/**
	 * Gets a perspective for the given key.
	 * 
	 * @param key the key of the perspective.
	 * @return the found perspective.
	 * @throws IllegalArgumentException if unable to find the perspective.
	 */
	public Perspective getPerspective(String key) throws IllegalArgumentException {
		checkPerspectives();
		final Perspective perspective = this.perspectivesMap.get(key);

		if (perspective != null) {
			return perspective;
		}

		throw new IllegalArgumentException(String.format(
				"Unknown perspective! [key=%1s]", key));
	}
	
	/**
	 * TODO
	 * 
	 * @return
	 */
	public Map<String, Perspective> getAllPerspectives() {
		checkPerspectives();
		return Collections.unmodifiableMap(this.perspectivesMap);
	}
	
	/**
	 * Load the perspectives from file.
	 *
	 * @param is the stream to load perspectives from.
	 * @throws IOException thrown if unable to load.
	 */
	public void loadPerspectives(InputStream is) throws IOException {
		if (this.perspectivesMap.size() == 0) {
			final Perspectives perspectives =
				JAXB.unmarshal(is, Perspectives.class);
			set(perspectives);
		}
	}
	
	/**
	 * Updates the current {@link Perspective} with the layout.
	 */
	public void updatePerspectives() {
		final Perspective currentPerspective = 
			getPerspective(getPerspectiveKey());

		currentPerspective.getLeafs().clear();

		final Node model = this.dockingPane.getMultiSplitLayout().getModel();
		final Set<Leaf> leafs = JXDockingPane.getAllLeaf(model);

		for (Leaf leaf : leafs) {
			final AbstractDock dock = this.dockingPane.getDock(leaf.getName());
			final PerspectiveLeaf pl = new PerspectiveLeaf(leaf.getName());
			currentPerspective.getLeafs().add(pl);
			for (int i = 0; i < dock.getTabCount(); i++) {
				final AbstractDockable dockable = (AbstractDockable) dock.getComponentAt(i);
				final PerspectiveView pv = new PerspectiveView(dockable.getId().toString());
				pl.getViews().add(pv);
			}
		}

		currentPerspective.setLayout(JXDockingPane.printModel(model));
	}
	
	/**
	 * Saves the perspectives to file.
	 * 
	 * @param os
	 * @throws IOException thrown if unable to save.
	 */
	public void savePerspectives(OutputStream os) throws IOException {
		final Perspectives perspectives = new Perspectives();
		for (Perspective perspective : this.perspectivesMap.values()) {
			perspectives.getPerspectives().add(perspective);
		}
		
		JAXB.marshal(perspectives, os);
		os.flush();
	}
	
	/**
	 * @param perspectives the perspectives to set.
	 */
	public void set(Perspectives perspectives) {
		for (Perspective perspective : perspectives.getPerspectives()) {
			this.perspectivesMap.put(
					perspective.getId(),
					perspective);
		}
	}
	
	/**
	 * TODO
	 */
	private void checkPerspectives() {
		if (this.perspectivesMap.isEmpty()) {
			this.log.log(
					Level.WARNING,
					"No perspectives loaded, reverting to default!");

			final Perspective defaultPerspective =
				getPerspectiveFactory().create(this, "default");
			this.perspectivesMap.put(
					defaultPerspective.getId(),
					defaultPerspective);
		}
	}

	/**
	 * 
	 * @param manager
	 * @param key
	 */
	protected void doChangePerspective(String key) {
		final Perspective perspective = getPerspective(key);
		
		// update model
		this.dockingPane.setModel(MultiSplitLayout.parseModel(
				perspective.getLayout()));

		for (PerspectiveLeaf leaf : perspective.getLeafs()) {
			for (PerspectiveView perspectiveView : leaf.getViews()) {
				final AbstractDockable viewInstance = getDockable(perspectiveView.getId(), true);
				if (perspectiveView != null) {
					dock(viewInstance, leaf.getId());
				}
			}
		}
	}
	
	
	// --- Inner Classes ---
	
	/**
	 * Default implementation of {@link DockableFactory}.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since 22th November 2007
	 */
	public static class DefaultDockableFactory implements DockableFactory {
	    /**
	     * {@inheritDoc}
	     */
	    @Override
	    public AbstractDockable create(Object key) {
	        try {
	        	final Class<? extends AbstractDockable> clazz = getClass(key);
                return clazz.newInstance();
	        } catch (InstantiationException ie) {
	        	throw new IllegalArgumentException(String.format(
		        		"Unable to instantiate dockable! [key=%1$s]",
		        		key), ie);
	        } catch (IllegalAccessException iae) {
	        	throw new IllegalArgumentException(String.format(
		        		"Unable to access dockable class! [key=%1$s]",
		        		key), iae);
	        }
	    }

	    /**
	     * {@inheritDoc}
	     */
	    @Override
	    @SuppressWarnings("unchecked")
	    public Class<? extends AbstractDockable> getClass(Object key) {
	    	if (key instanceof String) {
        		try {
        			return (Class<? extends AbstractDockable>) 
        					Class.forName((String) key);
				} catch (ClassNotFoundException e) {
					// not a class... evidently!
				}
        	}
        	
            if (key instanceof Class 
                    && AbstractDockable.class.isAssignableFrom((Class<?>) key))
            {
                return (Class<? extends AbstractDockable>) key;
            }
            
            throw new UnsupportedOperationException(String.format(
            		"Unknown key! [key=%1$s]",
            		key));
	    }
	}

	/**
	 * 
	 * TODO
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since 24th July 2008
	 */
	public static class DefaultPerspectiveFactory implements PerspectiveFactory {
		/**
		 * {@inheritDoc}
		 */
		@Override
		public Perspective create(DockManager manager, String key) {
			if (!DEFAULT_PERSPECTIVE.equals(key)) {
				throw new UnsupportedOperationException("Only default perspective supported!");
			}
			
			final Perspective perspective = new Perspective("default");
			perspective.setLayout("(LEAF name=default weight=1.0)");
			final PerspectiveLeaf defaultNode = new PerspectiveLeaf("default");
			defaultNode.getViews().add(
					new PerspectiveView(manager.getDockableClass("default").getName()));
			perspective.getLeafs().add(defaultNode);
			return perspective;
		}
	}

	/**
	 * Handles changes to the perspective.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since 24th July 2008
	 */
	private static class PerspectiveChangeHandler
	implements PropertyChangeListener
	{
		/**
		 * {@inheritDoc}
		 */
		@Override
		public void propertyChange(PropertyChangeEvent evt) {
			((DockManager) evt.getSource()).doChangePerspective(
					(String) evt.getNewValue());
		}
	}
}
