package civvi.osgi.desktop.swingx.menu;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.event.EventListenerList;

import org.jdesktop.beans.AbstractBean;

import civvi.osgi.desktop.swingx.JXMenuBar;
import civvi.osgi.desktop.swingx.menu.MenuModelEvent.Type;

/**
 * Defines a model for the menu structure to be used in the
 * {@link JXMenuBar}. This is intended to be dynamic dependent on added/removed
 * paths.
 * 
 * <pre>
 * 	'root'
 * 		> 'menu_1'
 * 			> menuitem_a
 * 				> menuitem_b
 * 				> menuitem_c
 *  		> menuitem_d
 *  		...
 *  		> menuitem_$
 * 		> 'menu_2'
 * 		...
 * 		> 'menu_n'
 * 
 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
 * @since v1.0.0 [13 Jun 2010]
 */
public class MenuModel {
	private final Logger log = Logger.getLogger(getClass().getName());
	private final EventListenerList listenerList = new EventListenerList();
	private final Node root;

	/**
	 * Default constructor.
	 */
	public MenuModel() {
		this.root = new Node("root", null, null);
	}

	/**
	 * @return the root node.
	 */
	public Node getRoot() {
		return root;
	}

	public void clear() {
		final List<Node> children = new ArrayList<Node>(getRoot().children);
		for (Node child : children) {
			getRoot().remove(child);
		}
	}

	/**
	 * 
	 * TODO
	 * @param actionId
	 * @param path
	 * @return
	 */
	public Node add(String actionId, String path) {
		final int index = path != null && !path.isEmpty()? path.indexOf('?') : -1;
		final String[] parentPath = index != -1 ? path.substring(0, index).split("/") : new String[0];
		final String placement = index != -1 ? path.substring(index + 1, path.length()) : null;
		Collections.reverse(Arrays.asList(parentPath));
		return add(actionId, placement, parentPath);
	}

	/**
	 * 
	 * TODO
	 * @param actionId
	 * @param placement
	 * @param path
	 * @return
	 */
	public Node add(String actionId, String placement, String... path) {
		if (this.log.isLoggable(Level.INFO)) {
			this.log.info(String.format(
					"Adding action. [actionId=%s,placement=%s,path=%s]",
					actionId,
					placement,
					Arrays.toString(path)));
		}
		
		final String id = actionId.substring(actionId.lastIndexOf('.') + 1, actionId.length());
		final Node parent = path.length == 0 ? this.root : get(path);
		if (parent == null) {
			throw new IllegalStateException(String.format(
					"Cannot add child to a non-existing parent! [%s]",
					Arrays.toString(path)));
		}

		if (log.isLoggable(Level.FINE)) {
			log.fine(String.format(
					"Adding action. [parentNode=%s,actionId=%s]",
					parent,
					actionId));
		}

		final Node child = new Node(id, actionId, placement);
		parent.add(child);
		return child;
	}

	/**
	 * Return the menu item at the given path.
	 * <p/>
	 * Paths are described as comma separated lists with a colon separating
	 * section and identifier.  
	 * 
	 * <pre>
	 * 	&lt;parent_section&gt;:&lt;parent_id&gt;,&lt;child_section&gt;:&lt;child_id&gt;
	 * </pre>
	 * 
	 * @param path
	 * @return
	 */
	public Node get(String... path) {
		return get(root, path);
	}
	
	/**
	 * 
	 * TODO 
	 * @param provider
	 * @param parent
	 * @param path
	 * @return
	 */
	private Node get(Node parent, String... path) {
		final String token = path[path.length - 1];
		
		final Node child = parent.get(token);
		if (child == null || path.length - 1 == 0) {
			return child;
		}
		
		final String[] remainder = new String[path.length - 1];
		System.arraycopy(path, 0, remainder, 0, remainder.length);
		return get(child, remainder);
	}

	public void addListener(MenuModelListener listener) {
		this.listenerList.add(MenuModelListener.class, listener);
	}

	public void removeListener(MenuModelListener listener) {
		this.listenerList.remove(MenuModelListener.class, listener);
	}

	/**
	 * Fires a {@link MenuModelEvent}.
	 * 
	 * @param node the node that changed.
	 * @param type the type of event.
	 */
	private void fire(MenuModelEvent.Type type, Node node, String... path) {
		final MenuModelListener[] listeners = this.listenerList.getListeners(MenuModelListener.class);
		final MenuModelEvent evt = new MenuModelEvent(this, type, node, path);
		for (int i = listeners.length -  1; i >= 0; i--) {
			listeners[i].changed(evt);
		}
	}


	// --- Inner classes ---

	/**
	 * Defines a node within the {@link MenuModel}.
	 * 
	 * TODO section ordering: Allow sections to always after/before others (optional)
	 * TODO child ordering: Allow a child to always appear after/before others (optional)
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [13 Jun 2010]
	 */
	public class Node extends AbstractBean {
		private final String id;
		private final String actionId;
		private final String placement;
		private Node parent;
		private final List<Node> children = new ArrayList<Node>();

		Node(String id, String actionId, String placement) {
			if (id == null || id.trim().isEmpty()) {
				throw new IllegalArgumentException("Id cannnot be null!");
			}
			this.id = id;
			this.actionId = actionId;
			this.placement = placement;
		}

		public String getId() {
			return this.id;
		}
		
		public String getActionId() {
			return this.actionId;
		}

		public String getPlacement() {
			return placement;
		}

		public Node getParent() {
			return parent;
		}

		public void add(Node child) {
			this.children.add(child);
			child.parent = this;
			fire(Type.ADDED, child, getPath());
		}
	
		public void remove(Node child) {
			final String[] path = child.getPath();
			final int index = this.children.indexOf(child);
			this.children.remove(index);
			child.parent = null;
			fire(Type.REMOVED, child, path);
		}
		
		public String[] getPath() {
			final List<String> path = new ArrayList<String>();
			Node current = this;
			while (current != null) {
				path.add(current.id);
				current = current.parent;
			}
			Collections.reverse(path);
			return path.toArray(new String[path.size()]);
		}

		public Node get(String childId) {
			for (Node child : this.children) {
				if (child.getId().equals(childId)) {
					return child;
				}
			}
			return null;
		}

		public Node getPrev() {
			final int index = this.parent.children.indexOf(this);
			if (index <= 0)
				return null;
			return this.parent.children.get(index - 1);
		}
		
		public Node getNext() {
			final int index = this.parent.children.indexOf(this);
			if (index < 0 || index >= this.parent.children.size())
				return null;
			return this.parent.children.get(index + 1);
		}
		
		public String getSection() {
			return getPlacementValue("section");
		}
		
		private String getPlacementValue(String key) {
			if ((placement != null || !placement.isEmpty()) && placement.contains(key + "=")) {
				int start = placement.indexOf(key + "=");
				int end = placement.indexOf("&", start) != -1 ? placement.indexOf("&", start) : placement.length();
				return placement.substring(start + 8, end);
			}
			return null;
		}
		
		@Override
		public String toString() {
			return super.toString() + " [id=" + getId() + "]";
		}
	}
}
