package civvi.osgi.desktop.swingx.nullable;

import javax.swing.AbstractListModel;
import javax.swing.ComboBoxModel;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;

/**
 * A {@link ComboBoxModel} used to wrap an existing {@link ComboBoxModel} but
 * with an added {@code null} item at index 0.
 * <p/>
 * <strong>NOTE</strong> {@link #getSelectedItem()} and
 * {@link #setSelectedItem(Object)} just delegate to the underlying model. If
 * the underlying model does not allow null to be selected, you may see some
 * odd behaviour.
 *
 * @author <a href="mailto:dansiviter@gmail.com">Dan Siviter</a>
 * @since 7 Jul 2006
 */
public class NullableComboBoxModel extends AbstractListModel
implements ComboBoxModel
{
	private static final long serialVersionUID = 5391912215887191687L;
	
	private ComboBoxModel delegate;
	private ListDataListener comboBoxModelListener;


	/**
	 * Constructs a nullable {@link ComboBoxModel}.
	 * 
	 * @param delegate
	 */
	NullableComboBoxModel(ComboBoxModel delegate) {
		super();
		setDelegate(delegate);
	}

	/**
	 * @return Returns the delegare {@link ComboBoxModel}.
	 */
	public ComboBoxModel getDelegate() {
		return delegate;
	}

	/**
	 * @param delegate the delegate {@link ComboBoxModel} to set.
	 */
	public void setDelegate(ComboBoxModel delegate) {
		if (comboBoxModelListener == null) {
			comboBoxModelListener = new ListDataHandler();
		}

		if (this.delegate != null) {
			this.delegate.removeListDataListener(comboBoxModelListener);
		}

		this.delegate = delegate;

		if (this.delegate != null) {
			this.delegate.addListDataListener(comboBoxModelListener);
		}
	}


	// --- Overriden/Implemented Methods ---

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object getSelectedItem() {
		return getDelegate().getSelectedItem();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setSelectedItem(Object anItem) {
		getDelegate().setSelectedItem(anItem);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getSize() {
		return 1 + getDelegate().getSize();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Object getElementAt(int index) {
		return (index == 0 ? null :
			getDelegate().getElementAt(viewToModel(index)));
	}

	/**
	 * Converts an index number in this ComboBoxModel to an index number
	 * in the underlying comboBoxModel
	 * 
	 * @param index the view index to convert.
	 * @return the index in the model.
	 */
	protected int viewToModel(int index) {
		return index <= 0 ? index : --index;
	}

	/**
	 * Converts an index number in the underlying comboBoxModel to an index
	 * number in this ComboBoxModel
	 * 
	 * @param index the model index to convert.
	 * @return the index int the view.
	 */
	protected int modelToView(int index) {
		return index < 0 ? index : ++index;
	}


	// --- Inner Classes ---    

	/**
	 * Captures events from the underlying comboBoxModel and refires them
	 * with the modified index, and this as the source
	 *
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since 7 Jul 2006
	 */
	class ListDataHandler implements ListDataListener {
		/**
		 * {@inheritDoc}
		 */
		@Override
		public void contentsChanged(ListDataEvent e) {
			fireContentsChanged(
					this,
					modelToView(e.getIndex0()),
					modelToView(e.getIndex1()));
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public void intervalAdded(ListDataEvent e) {
			fireIntervalAdded(
					this,
					modelToView(e.getIndex0()),
					modelToView(e.getIndex1()));
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public void intervalRemoved(ListDataEvent e) {
			fireIntervalRemoved(
					this,
					modelToView(e.getIndex0()),
					modelToView(e.getIndex1()));
		}
	}
}