package civvi.osgi.desktop.swingx.nullable;

import java.awt.Component;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import javax.swing.JList;
import javax.swing.ListCellRenderer;

/**
 * A decorator {@link ListCellRenderer} that will display an empty string for
 * a {@code null} value.
 *
 * @author <a href="mailto:dansiviter@gmail.com">Dan Siviter</a>
 * @since 7 Jul 2006
 */
public class NullableListCellRenderer implements ListCellRenderer {
	private final ListCellRenderer delegate;
	private final String nullText;
	
	/**
	 * Constructs a nullable {@link ListCellRenderer}.
	 *
	 * @param delegate the renderer to decorate.
	 * @param nullText the text to display in the {@code null} position. If
	 * {@code null} then an empty string is used.
	 */
	NullableListCellRenderer(
			final ListCellRenderer delegate,
			final String nullText)
	{
		this.delegate = delegate;
		this.nullText = nullText != null ? nullText : " ";
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Component getListCellRendererComponent(
			JList list,
			Object value,
			int index,
			boolean isSelected,
			boolean cellHasFocus)
	{
		final Component component = this.delegate.getListCellRendererComponent(
				list,
				value,
				index,
				isSelected,
				cellHasFocus);
		
		if (value == null) {
			decorateNull(component);
		}
		
		return component;
	}

	/**
	 * Decorates the component with null state.
	 *
	 * @param component
	 */
	private void decorateNull(Component component)  {
		try {
			// should work for JLabel and others
			final Method setTextMethod = component.getClass().getMethod("setText", String.class);
			setTextMethod.invoke(component, this.nullText);
		} catch (NoSuchMethodException nsme) {
			throw new UnsupportedOperationException(String.format(
					"Unsupported component! [component=%s]",
					component), nsme);
		} catch (IllegalAccessException iae) {
			throw new UnsupportedOperationException(String.format(
					"Unsupported component! [component=%s]",
					component), iae);
		} catch (InvocationTargetException ite) {
			throw new UnsupportedOperationException(String.format(
					"Unsupported component! [component=%s]",
					component), ite);
		}
	}
}
