package civvi.osgi.desktop.swingx.painter;

import java.awt.BasicStroke;
import java.awt.Graphics2D;
import java.awt.Insets;
import java.awt.Paint;
import java.awt.RadialGradientPaint;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.awt.geom.RectangularShape;
import java.awt.geom.RoundRectangle2D;

import javax.swing.JComponent;

import org.jdesktop.swingx.painter.AbstractAreaPainter;
import org.jdesktop.swingx.painter.effects.AreaEffect;
import org.jdesktop.swingx.util.PaintUtils;

/**
 * A Gradient based painter used for painting "multi-stop" radial gradients. These are
 * gradients that imploys more than 2 colors, where each color is defined along
 * with a float value between 0 and 1 indicating at what point along the gradient
 * the new color is used.
 * <p/>
 * As with BasicGradienPainter and mentioned in AbstractGradientPainter, the values
 * given to the centerPoint, radius, and focusPoint of the RadialGradientPainter are crucial. They
 * represent what distance from the origin the gradient should begin and end at,
 * depending on the size of the component. That is, they must be specified as values between
 * 0 and 1, where 0 means "all the way on the left/top" and 1 means "all the way on the
 * right/bottom".
 * <p/>
 * In addition, the resize behavior of the radius is specified in the resizeRadius
 * property. If HORIZONTAL, then the width of the component is used to calculate
 * the new radius. If VERTICAL then the height of the component is used. If BOTH,
 * then the Math.min(width, height) is used. If NONE, then no resize occurs for
 * the radius.
 * <p/>
 * <strong>NOTE: RadialGradientPainter relies on LinearGradientPaint, which is
 * included in the optional jar MultipleGradientPaint.jar. Be sure to have this
 * jar on your classpath if you use this class</strong>
 * 
 * @author <a href="mailto:dansiviter@gmail.com">Dan Siviter</a>
 * @since 11 Dec 2007
 */
public class RadialGradientPainter extends AbstractAreaPainter<JComponent> {
	public enum Resize {
		HORIZONTAL,
		VERTICAL,
		BOTH,
		NONE
	}
	
	private boolean rounded = false;
    //private Insets insets = new Insets(0,0,0,0);
    private int roundWidth = 20;
    private int roundHeight = 20;
    private int width = -1;
    private int height = -1;
	
//	private RadialGradientPaint paint;
	private Resize resizeRadius = Resize.HORIZONTAL;
	

	/** Creates a new instance of RadialGradientPainter */
	public RadialGradientPainter() {
	}

	/** 
	 * Creates a new instance of RadialGradientPainter 
	 * with the given RadialGradientPaint
	 *
	 * @param paint the RadialGradientPaint to use
	 */
	public RadialGradientPainter(RadialGradientPaint paint) {
		setGradientPaint(paint);
	}

	/**
	 * Set the gradient paint to use. This may be null. If null, nothing is painted
	 *
	 * @param paint the RadialGradientPaint to use
	 */
	public void setGradientPaint(RadialGradientPaint paint) {
		setFillPaint(paint);
	}

	/**
	 * @return the RadialGradientPaint used for painting. This may be null
	 */
	public RadialGradientPaint getGradientPaint() {
		return (RadialGradientPaint) getFillPaint();
	}

	@Override
	public void setFillPaint(Paint p) {
		super.setFillPaint(p);
	}
	
	/**
	 * Specifies the resize behavior for the radius of the RadialGradientPaint.
	 * If HORIZONTAL, then the width of the component is used to calculate
	 * the new radius. If VERTICAL then the height of the component is used. If BOTH,
	 * then the Math.min(width, height) is used. If NONE, then no resize occurs for
	 * the radius.
	 *
	 * @param r the Resize behavior for the radius
	 */
	public void setResizeRadius(Resize r) {
		Resize old = getResizeRadius();
		this.resizeRadius = r;
		firePropertyChange("resizeRadius", old, getResizeRadius());
	}

	/**
	 * @return the resize behavior for the radius
	 */
	public Resize getResizeRadius() {
		return resizeRadius;
	}

	/**
     * Indicates if the rectangle is rounded
     * @return if the rectangle is rounded
     */
    public boolean isRounded() {
        return rounded;
    }

    /**
     * sets if the rectangle should be rounded
     * @param rounded if the rectangle should be rounded
     */
    public void setRounded(boolean rounded) {
        boolean oldRounded = isRounded();
        this.rounded = rounded;
        setDirty(true);
        firePropertyChange("rounded",oldRounded,rounded);
    }

    /**
     * gets the round width of the rectangle
     * @return the current round width
     */
    public int getRoundWidth() {
        return roundWidth;
    }

    /**
     * sets the round width of the rectangle
     * @param roundWidth a new round width
     */
    public void setRoundWidth(int roundWidth) {
        int oldRoundWidth = getRoundWidth();
        this.roundWidth = roundWidth;
        setDirty(true);
        firePropertyChange("roundWidth",oldRoundWidth,roundWidth);
    }

    /**
     * gets the round height of the rectangle
     * @return the current round height
     */
    public int getRoundHeight() {
        return roundHeight;
    }

    /**
     * sets the round height of the rectangle
     * @param roundHeight a new round height
     */
    public void setRoundHeight(int roundHeight) {
        int oldRoundHeight = getRoundHeight();
        this.roundHeight = roundHeight;
        setDirty(true);
        firePropertyChange("roundHeight",oldRoundHeight,roundHeight);
    }
	
	/**
	 * @inheritDoc
	 */
	protected Paint calculateSizedPaint(int width, int height) {
		RadialGradientPaint paint = getGradientPaint();
		if (paint == null) {
			return null;
		}

		Point2D centerPoint = paint.getCenterPoint();
		Point2D focusPoint = paint.getFocusPoint();

		centerPoint = new Point2D.Double(centerPoint.getX() * width, centerPoint.getY() * height);
		focusPoint = new Point2D.Double(focusPoint.getX() * width, focusPoint.getY() * height);

		float radius = paint.getRadius();
		Resize r = getResizeRadius();
		r = r == null ? Resize.BOTH : r;
		switch (r) {
		case HORIZONTAL:
			radius = radius * width;
			break;
		case VERTICAL:
			radius = radius * height;
			break;
		case BOTH:
			radius = radius * Math.min(width, height);
			break;
		case NONE:
			break;
		default:
			throw new AssertionError("Cannot happen");
		}

		return new RadialGradientPaint(
				centerPoint,
				radius,
				focusPoint,
				paint.getFractions(),
				paint.getColors(),
				paint.getCycleMethod(),
				paint.getColorSpace(),
				paint.getTransform());
	}

	 /* ======== drawing code ============ */
    protected RectangularShape calculateShape(int width, int height) {
        Insets insets = getInsets();
        int x = insets.left;
        int y = insets.top;

        // use the position calcs from the super class
        Rectangle bounds = calculateLayout(this.width, this.height, width, height);
        if(this.width != -1 && !isFillHorizontal()) {
            width = this.width;
            x = bounds.x;
        }
        if(this.height != -1 && !isFillVertical()) {
            height = this.height;
            y = bounds.y;
        }

        if(isFillHorizontal()) {
            width = width - insets.left - insets.right;
        }
        if(isFillVertical()) {
            height = height - insets.top - insets.bottom;
        }


        RectangularShape shape = new Rectangle2D.Double(x, y, width, height);
        if(rounded) {
            shape = new RoundRectangle2D.Double(x, y, width, height, roundWidth, roundHeight);
        }
        return shape;
    }
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public Shape provideShape(Graphics2D g, JComponent comp, int width, int height) {
		return calculateShape(width,height);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void doPaint(Graphics2D g, JComponent object, int width, int height) {
	        RectangularShape shape = calculateShape(width, height);
	        switch (getStyle()) {
	        case BOTH:
	            drawBackground(g,shape,width,height);
	            drawBorder(g,shape,width,height);
	            break;
	        case FILLED:
	            drawBackground(g,shape,width,height);
	            break;
	        case OUTLINE:
	            drawBorder(g,shape,width,height);
	            break;
	        case NONE:
	            break;
	        }

	        // background
	        // border
	        // leave the clip to support masking other painters
//	        PaintUtils.setMergedClip(g,shape);
	        /*
	        Area area = new Area(g.getClip());
	        area.intersect(new Area(shape));//new Rectangle(0,0,width,height)));
	        g.setClip(area);*/
	        //g.setClip(shape);
	}
	
	private void drawBackground(Graphics2D g, Shape shape, int width, int height) {
        Paint p = calculateSizedPaint(width, height);
        if(isPaintStretched()) {
            p = PaintUtils.resizeGradient(p, width, height);
        }

        g.setPaint(p);

        g.fill(shape);
        if(getAreaEffects() != null) {
            for(AreaEffect ef : getAreaEffects()) {
                ef.apply(g, shape, width, height);
            }
        }
    }
	
	private void drawBorder(Graphics2D g, RectangularShape shape, int width, int height) {
        Paint p = getBorderPaint();
        if(isPaintStretched()) {
            p = PaintUtils.resizeGradient(p, width, height);
        }

        g.setPaint(p);

        g.setStroke(new BasicStroke(getBorderWidth()));
        // shrink the border by 1 px
        if(shape instanceof Rectangle2D) {
            g.draw(new Rectangle2D.Double(shape.getX(), shape.getY(),
                    shape.getWidth()-1, shape.getHeight()-1));
        } else if(shape instanceof RoundRectangle2D) {
            g.draw(new RoundRectangle2D.Double(shape.getX(), shape.getY(),
                    shape.getWidth()-1, shape.getHeight()-1,
                    ((RoundRectangle2D)shape).getArcWidth(),
                    ((RoundRectangle2D)shape).getArcHeight()));

        } else {
            g.draw(shape);
        }
    }
}