package civvi.osgi.desktop.swingx.plaf.basic;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.FlowLayout;
import java.awt.Insets;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.Action;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JToolBar;
import javax.swing.plaf.ComponentUI;

import civvi.osgi.desktop.swingx.JXDockingPane;
import civvi.osgi.desktop.swingx.docking.AbstractDock;
import civvi.osgi.desktop.swingx.docking.AbstractDockable;
import civvi.osgi.desktop.swingx.plaf.DockingPaneUI;

/**
 * Default/basic UI for {@link JXDockingPane}.
 * 
 * @author <a href="mailto:dansiviter@gmail.com">Dan Siviter</a>
 * @since v1.0.0 [20 Nov 2007]
 */
public class BasicDockingPaneUI extends DockingPaneUI {
	private final static DockingPaneUI dockingPaneUI = new BasicDockingPaneUI();

	@Override
	public void installUI(JComponent c) {
		super.installUI(c);

		final JXDockingPane dockingPane = (JXDockingPane) c;
		dockingPane.setBorder(BorderFactory.createEmptyBorder(0, 2, 0, 0));
		dockingPane.setDividerSize(3);
	}

	@Override
	public void installUI(AbstractDock dock) {
//		// Override to increase right inset for tab area,
//	    // in order to reserve space for the connect toggle.
//		final Insets insets = (Insets)UIManager.getLookAndFeelDefaults().get("TabbedPane.tabAreaInsets");
//		final Insets newInsets = (Insets) insets.clone();
//		newInsets.right += 50;
//		UIManager.put("TabbedPane.tabAreaInsets", newInsets);
//		dock.updateUI();
//		UIManager.put("TabbedPane.tabAreaInsets", insets);
	}

	@Override
	public void uninstallUI(AbstractDock dock) { }
	
	@Override
	public void installUI(AbstractDockable dockable) {
		if (dockable.getTabComponent() == null) { // had component changed?
			JLabel tabLabel = new JLabel(dockable.getTitle(), dockable.getSmallIcon(), JLabel.LEADING);
			dockable.addPropertyChangeListener(new PropertyChangeHandler(tabLabel));
			JPanel tabPanel = new JPanel(new FlowLayout(FlowLayout.LEFT, 0, 0));
			tabPanel.setOpaque(false);
			tabPanel.setBorder(BorderFactory.createEmptyBorder());
			tabPanel.add(tabLabel);
			tabPanel.add(Box.createHorizontalStrut(3));
			tabPanel.add(new RolloverButton(dockable.getActionMap().get(AbstractDockable.CLOSE_ACTION_KEY)));
			dockable.setTabComponent(tabPanel);

			if (dockable.getToolbarActionKeys().length > 0) {
				JToolBar toolBar = new JToolBar();
				toolBar.setFloatable(false);
				toolBar.add(Box.createGlue());
				toolBar.setOpaque(false);

				for (Object actionKey : dockable.getToolbarActionKeys()) {
					final Action action = dockable.getActionMap().get(actionKey);
					JButton button = toolBar.add(action);
					button.setBorderPainted(false);
					button.setContentAreaFilled(false);
					button.setOpaque(false);
					button.addMouseListener(new RolloverMouseHandler());

					// attempt to use smaller icons
					if (action.getValue(Action.SMALL_ICON) != null)
						button.setIcon((Icon) action.getValue(Action.SMALL_ICON));
				}	    	

				dockable.add(toolBar, BorderLayout.NORTH);
			}
		}
	}

	@Override
	public void uninstallUI(AbstractDockable dockable) {
		dockable.setTabComponent(null);
	}


	// --- Static Methods ---

	/**
	 * Create and return instance of this UI.
	 * 
	 * @param c the component to attach to.
	 * @return an instance of this UI.
	 * @see ComponentUI#createUI(JComponent)
	 */
	public static ComponentUI createUI(JComponent c) {
		return dockingPaneUI;
	}


	// --- Inner Classes ---

	/**
	 * Handles changes to properties to ensure they are kept in sync. with the 
	 * parent {@link Dock}.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [20 Nov 2007]
	 */
	private static class PropertyChangeHandler implements PropertyChangeListener {
		private final JLabel tabLabel;

		public PropertyChangeHandler(JLabel tabLabel) {
			this.tabLabel = tabLabel;
		}

		@Override
		public void propertyChange(PropertyChangeEvent evt) {
			if ("title".equals(evt.getPropertyName())) {
				this.tabLabel.setText((String) evt.getNewValue());
			} else if ("tooltip".equals(evt.getPropertyName())) {
				this.tabLabel.setToolTipText((String) evt.getNewValue());
			} else if ("smallIcon".equals(evt.getPropertyName())) {
				this.tabLabel.setIcon((Icon) evt.getNewValue());
			}
		}
	}

	/**
	 * A {@link JButton} that is borderless unless the mouse rolls over.
	 *
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [19 Dec 2007]
	 */
	private class RolloverButton extends JButton {
		private static final long serialVersionUID = 8489953893693698932L;

		public RolloverButton(Action action) {
			super(action);
			setMargin(new Insets(3, 3, 3, 3));
			setBorderPainted(false);
			setContentAreaFilled(false);
			setPreferredSize(new Dimension(18, 18));
			addMouseListener(new RolloverMouseHandler());
		}
	}

	/**
	 * Handles mouse rollover events to allow opaque effects.
	 *
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [5 Jan 2008]
	 */
	private static class RolloverMouseHandler extends MouseAdapter {
		@Override
		public void mouseEntered(MouseEvent e) {
			final JButton button = (JButton) e.getSource();
			if (button.isEnabled()) {
				button.setBorderPainted(true);
				button.setContentAreaFilled(true);
			}
		}

		@Override
		public void mouseExited(MouseEvent e) {
			final JButton button = (JButton) e.getSource();
			if (button.isEnabled()) {
				button.setBorderPainted(false);
				button.setContentAreaFilled(false);
			}
		}
	}
}
