package civvi.osgi.desktop.view.log;

import java.awt.Dimension;
import java.util.Collections;
import java.util.List;

import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import org.jdesktop.application.Action;
import org.jdesktop.swingx.JXList;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.decorator.ShadingColorHighlighter;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.service.log.LogEntry;
import org.osgi.service.log.LogService;

import civvi.osgi.desktop.OSGiDesktop;
import civvi.osgi.desktop.log.DesktopLogEntry;
import civvi.osgi.desktop.log.DesktopLogReaderService;
import civvi.osgi.desktop.swingx.GenericListModel;
import civvi.osgi.desktop.swingx.SwingUtil;
import civvi.osgi.desktop.swingx.event.ListDataAdaptor;
import civvi.osgi.desktop.view.AbstractView;

/**
 * 
 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
 * @since v1.0.0 [24 Sep 2010]
 */
public class LogEntriesView extends AbstractView {
	private static final long serialVersionUID = -5256086138833593650L;

	private final JXList list;
	private final GenericListModel<LogEntry> model;
	private final DesktopLogReaderService readerService;

	private LogEntryEditor logEntryEditor;
	
	private boolean entriesSelected;

	/**
	 * Default contructor.
	 */
	public LogEntriesView() {
		super("logEntries");
		this.list = new JXList();
		this.list.setPreferredSize(new Dimension(350, 50));
		this.list.setCellRenderer(new LogEntryCellRenderer());
		this.list.setModel(this.model = new GenericListModel<LogEntry>());
		this.list.setPrototypeCellValue(new PrototypeLogEntry(
				null,
				null,
				LogService.LOG_WARNING,
				"Test Message",
				null));
		this.list.addListSelectionListener(new ListSelectionHandler());
		this.model.addListDataListener(new ListDataHandler());
		this.list.setHighlighters(
				new ShadingColorHighlighter(
						new HighlightPredicate.RowGroupHighlightPredicate(1)));
		setContent(new JScrollPane(this.list));

		setToobar(SwingUtil.createTabToolbar(
				null,
				this.support.getAction("showEntry"),
				null,
				this.support.getAction("removeEntry"),
				this.support.getAction("removeAllEntries")));

		final OSGiDesktop desktop = (OSGiDesktop) this.support.getApplication();
		final BundleContext context = desktop.getActivator().getContext();
		final ServiceReference ref = context.getServiceReference(
				DesktopLogReaderService.class.getName());
		this.readerService = (DesktopLogReaderService) context.getService(ref);
		this.model.set(this.readerService.getEntries());
	}

	public boolean isEntriesSelected() {
		return entriesSelected;
	}

	public void setEntriesSelected(boolean entriesSelected) {
		final boolean oldValue = this.entriesSelected;
		this.entriesSelected = entriesSelected;
		firePropertyChange("entriesSelected", oldValue, this.entriesSelected);
	}
	
	@Override
	@SuppressWarnings({ "unchecked", "rawtypes" })
	protected void close() {
		super.close();
		this.model.set((List) Collections.emptyList());
	}


	// --- Actions ---

	/**
	 * Cancels the selected tasks.
	 */
	@Action(enabledProperty = "entriesSelected")
	public void removeEntry() {
		final int result = JOptionPane.showConfirmDialog(
				getDock().getDockingPane(),
				this.support.getString("confirmRemove.message"),
				this.support.getString("confirmRemove.title"),
				JOptionPane.OK_CANCEL_OPTION,
				JOptionPane.QUESTION_MESSAGE);

		if (result != JOptionPane.OK_OPTION) {
			return;
		}

		final int[] viewIndices = this.list.getSelectedIndices();

		for (int i = 0; i < viewIndices.length; i++) {
			final LogEntry entry = this.model.get(this.list.convertIndexToModel(viewIndices[i]));
			this.readerService.remove(entry);
		}
	}

	/**
	 * Cancels the selected tasks.
	 */
	@Action
	public void removeAllEntries() {
		final int result = JOptionPane.showConfirmDialog(
				getDock().getDockingPane(),
				this.support.getString("confirmRemoveAll.message"),
				this.support.getString("confirmRemoveAll.title"),
				JOptionPane.OK_CANCEL_OPTION,
				JOptionPane.QUESTION_MESSAGE);

		if (result != JOptionPane.OK_OPTION) {
			return;
		}

		this.readerService.removeAll();
	}

	/**
	 * Displays the selected log entry.
	 */
	@Action(enabledProperty = "entriesSelected")
	public void showEntry() {
		if (this.logEntryEditor == null)
			this.logEntryEditor = new LogEntryEditor();

		final int viewIndex = this.list.getSelectedIndex();
		final LogEntry entry = this.model.get(this.list.convertIndexToModel(viewIndex));
		this.logEntryEditor.setEditableBean(entry);

		SwingUtil.showEditorDialog(
				this,
				this.logEntryEditor,
				this.support.getString("logEntryEditor.title"));
	}


	// --- Inner Classes ---

	/**
	 * Handles changes to the list.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [23 Mar 2009]
	 */
	private class ListDataHandler extends ListDataAdaptor {
		@Override
		public void contentsChanged(ListDataEvent e) {
			support.getAction("removeAllEntries").setEnabled(!model.isEmpty());
		}
	}

	/**
	 * Handles selections on the list.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [23 Mar 2009]
	 */
	private class ListSelectionHandler implements ListSelectionListener {
		/**
		 * {@inheritDoc}
		 */
		@Override
		public void valueChanged(ListSelectionEvent e) {
			if (e.getValueIsAdjusting())
				return;

			final int[] viewIndices = list.getSelectedIndices();

			setEntriesSelected(viewIndices.length > 0);
		}
	}

	/**
	 * A prototype {@link LogEntry} to ensure the size of the list items is
	 * correct.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [24 Sep 2010]
	 */
	private static class PrototypeLogEntry extends DesktopLogEntry {
		public PrototypeLogEntry(
				Bundle bundle,
				ServiceReference ref,
				int level,
				String message,
				Throwable t)
		{
			super(bundle, ref, level, message, t);
		}
	}
}
