package civvi.osgi.desktop.view.task;

import java.awt.Dimension;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.List;

import javax.swing.JOptionPane;
import javax.swing.JScrollPane;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import org.jdesktop.application.Action;
import org.jdesktop.application.Task;
import org.jdesktop.application.TaskMonitor;
import org.jdesktop.swingx.JXList;
import org.jdesktop.swingx.decorator.HighlightPredicate;
import org.jdesktop.swingx.decorator.ShadingColorHighlighter;

import civvi.osgi.desktop.swingx.AbstractTask;
import civvi.osgi.desktop.swingx.GenericListModel;
import civvi.osgi.desktop.swingx.SwingUtil;
import civvi.osgi.desktop.swingx.event.ListDataAdaptor;
import civvi.osgi.desktop.view.AbstractView;

/**
 * A view for monitoring and cancelling {@link Task}s.
 * 
 * @author <a href="mailto=dansiviter@gmail.com">Daniel Siviter</a>
 * @since v1.0.0 [23 Mar 2009]
 */
public class TasksView extends AbstractView {
	private static final long serialVersionUID = -5256086138833593650L;

	private final JXList list;
	private final GenericListModel<Task<?, ?>> model;
	private boolean selectedCancellable;


	/**
	 * Default contructor.
	 */
	public TasksView() {
		super("tasks");
		this.list = new JXList();
		this.list.setPreferredSize(new Dimension(350, 50));
		this.list.setCellRenderer(new TaskCellRenderer());
		this.list.setModel(this.model = new GenericListModel<Task<?, ?>>());
		this.list.setPrototypeCellValue(new PrototypeTask(
				"Title",
				"Message"));
		this.list.addListSelectionListener(new ListSelectionHandler());
		this.model.addListDataListener(new ListDataHandler());
		this.list.setHighlighters(
				new ShadingColorHighlighter(
						new HighlightPredicate.RowGroupHighlightPredicate(1)));
		setContent(new JScrollPane(this.list));

		final TaskMonitor monitor = this.support.getContext().getTaskMonitor();
		monitor.addPropertyChangeListener("tasks", new TasksChangeHandler(this.model));

		setToobar(SwingUtil.createTabToolbar(null, this.support.getAction("cancelTasks")));
	}

	public void setSelectedCancellable(boolean selectedCancellable) {
		final boolean oldValue = this.selectedCancellable;
		this.selectedCancellable = selectedCancellable;
		firePropertyChange("selectedCancellable", oldValue, this.selectedCancellable);
	}
	
	public boolean isSelectedCancellable() {
		return selectedCancellable;
	}
	
	/**
	 * Cancels the selected tasks.
	 */
	@Action(enabledProperty = "selectedCancellable")
	public void cancelTasks() {
		final int result = JOptionPane.showConfirmDialog(
				getDock().getDockingPane(),
				this.support.getString("confirm.message"),
				this.support.getString("confirm.title"),
				JOptionPane.OK_CANCEL_OPTION,
				JOptionPane.QUESTION_MESSAGE);

		if (result != JOptionPane.OK_OPTION) {
			return;
		}

		final int[] viewIndices = this.list.getSelectedIndices();

		for (int i = 0; i < viewIndices.length; i++) {
			final Task<?, ?> task = this.model.get(this.list.convertIndexToModel(viewIndices[i]));
			if (task.getUserCanCancel() && !task.isCancelled()) {
				task.cancel(true);
			}
		}
	}


	// --- Inner Classes ---

	/**
	 * Handles changes to the known tasks.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [23 Mar 2009]
	 */
	private static class TasksChangeHandler implements PropertyChangeListener {
		private final GenericListModel<Task<?, ?>> model;

		public TasksChangeHandler(GenericListModel<Task<?, ?>> model) {
			this.model = model;
		}

		@Override
		public void propertyChange(PropertyChangeEvent evt) {
			@SuppressWarnings("unchecked")
			final List<Task<?, ?>> tasks = (List<Task<?, ?>>) evt.getNewValue();
			this.model.set(tasks);
		}
	}

	/**
	 * Handles changes to the list.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [23 Mar 2009]
	 */
	private class ListDataHandler extends ListDataAdaptor {
		@Override
		public void contentsChanged(ListDataEvent e) {
			if (model.isEmpty()) {
				setSelectedCancellable(false);
			}
		}
	}

	/**
	 * Handles selections on the list.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [23 Mar 2009]
	 */
	private class ListSelectionHandler implements ListSelectionListener {
		/**
		 * {@inheritDoc}
		 */
		@Override
		public void valueChanged(ListSelectionEvent e) {
			if (e.getValueIsAdjusting())
				return;

			final int[] viewIndices = list.getSelectedIndices();

			boolean canCancel = viewIndices.length > 0;

			for (int i = 0; i < viewIndices.length; i++) {
				final Task<?, ?> task = model.get(list.convertIndexToModel(viewIndices[i]));
				if (!task.getUserCanCancel() || task.isCancelled()) {
					canCancel = false;
				}
			}
			setSelectedCancellable(canCancel);
		}
	}

	/**
	 * A prototype task to ensure the size of the list items is correct.
	 * 
	 * @author <a href="mailto:dansiviter@gmail.com">Daniel Siviter</a>
	 * @since v1.0.0 [23 Mar 2009]
	 */
	private static class PrototypeTask extends AbstractTask<Object, Object> {
		public PrototypeTask(String title, String description) {
			setTitle(title);
			setDescription(description);
		}

		@Override
		protected Object doInBackground() throws Exception {
			throw new UnsupportedOperationException();
		}	
	}
}
