package civvi.osgi.desktop.tail.business;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.swing.event.EventListenerList;

import civvi.osgi.desktop.tail.business.TailEvent.Type;

/**
 * A log file tailer is designed to monitor a log file and send notifications
 * when new lines are added to the log file. This class has a notification
 * strategy similar to a SAX parser: implement the LogFileTailerListener interface,
 * create a LogFileTailer to tail your log file, add yourself as a listener, and
 * start the LogFileTailer. It is your job to interpret the results, build meaningful
 * sets of data, etc. This tailer simply fires notifications containing new log file lines, 
 * one at a time.
 * 
 * @author <a href="mailto:dan.siviter@civvi.com">Dan Siviter</a>
 * @since 19 Mar 2009
 */
public class FileTailer {
	private static final Timer timer = new Timer("Tail Timer");
	
	private final Logger log = Logger.getLogger(getClass().getName());
	private final EventListenerList listeners = new EventListenerList();
	private final Charset charset;
	private final CharsetDecoder decoder;

	private TimeUnit timeUnit;
	private int sampleDuration;
	private File file;
	private boolean startAtBeginning;
	private RandomAccessFile raf;
	private FileChannel channel;
	private long filePointer = 0;
	private TimerTask timerTask;

	private long maxBuffer = 1024 * 1024 * 1; 
	
	/**
	 * 
	 * TODO
	 *
	 */
	public FileTailer() {
		this(Charset.defaultCharset(), true, TimeUnit.MILLISECONDS, 250);
	}
	
	/**
	 * 
	 * TODO
	 * 
	 * @param charset
	 * @param startAtBeginning
	 * @param timeUnit
	 * @param sampleDuration
	 */
	public FileTailer(
			Charset charset,
			boolean startAtBeginning,
			TimeUnit timeUnit,
			int sampleDuration)
	{
		this.charset = charset;
		this.decoder = this.charset.newDecoder();
		this.startAtBeginning = startAtBeginning;
		this.timeUnit = timeUnit;
		this.sampleDuration = sampleDuration;
	}
	
	/**
	 * 
	 * TODO
	 *
	 * @param file
	 */
	public void setFile(File file) {
		if (file == null) {
			throw new IllegalArgumentException("File cannot be null!");
		}
		if (this.file != null && this.file.equals(file)) {
			return;
		}

		if (isRunning() && !file.equals(file)) {
			throw new IllegalStateException("Stop tailer before setting new file!");
		}
		this.file = file;
	}

	/**
	 * @param l the listener to add.
	 */
	public void addListener(TailListener l) {
		this.listeners.add(TailListener.class, l);
	}

	/**
	 * @param l the listener to remove.
	 */
	public void removeListener(TailListener l) {
		this.listeners.remove(TailListener.class, l);
	}

	/**
	 * 
	 * TODO
	 *
	 * @param line
	 * @param type
	 */
	protected void fireTextAdded(String line, Type type) {
		final TailEvent event = new TailEvent(this, line, type);
		
		final TailListener[] listeners = this.listeners.getListeners(
				TailListener.class);
		for (int i = listeners.length - 1; i >= 0; i--) {
			listeners[i].newLine(event);
		}
	}

	/**
	 * 
	 * TODO
	 *
	 * @return
	 */
	public boolean cancel() {
		try {
			final boolean result = this.timerTask.cancel();
			this.raf.close();
			this.channel = null;
			this.raf = null;
			return result;
		} catch (IOException ioe) {
			throw new RuntimeException(ioe);
		}
	}

	/**
	 * 
	 * TODO
	 *
	 */
	public void run() {
		try {
			Type type = Type.Increase;
			// Compare the length of the file to the file pointer
			long fileLength = this.file.length();
			
			if (fileLength < this.filePointer) {
				// Log file must have been rotated or deleted; 
				// reopen the file and reset the file pointer
				this.raf = new RandomAccessFile(this.file, "r");
				this.filePointer = 0;
				type = Type.Decrease;
			}

			if (fileLength > filePointer) {
				
				if (fileLength > maxBuffer) {
					filePointer = fileLength - maxBuffer;
					// There is data to read
					raf.seek(filePointer);
				}
				final ByteBuffer buffer = ByteBuffer.allocate(
						(int) (fileLength - filePointer));

				this.channel.read(buffer, this.filePointer);
				buffer.flip();

				CharBuffer charBuffer = this.decoder.decode(buffer);
				fireTextAdded(charBuffer.toString(), type);

				this.filePointer += buffer.capacity();
			}
		} catch (FileNotFoundException fnfe) {
			if (this.log.isLoggable(Level.WARNING)) {
				this.log.log(Level.WARNING, fnfe.getMessage(), fnfe);
			}
		} catch (IOException ie) {
			if (this.log.isLoggable(Level.WARNING)) {
				this.log.log(Level.WARNING, ie.getMessage(), ie);
			}
		}
	}

	/**
	 * 
	 * TODO
	 * @throws FileNotFoundException 
	 *
	 */
	public void start() throws FileNotFoundException {
		if (isRunning()) {
			throw new IllegalStateException(
					"Tail thread is executing. Call #stop() before starting.");
		}

		this.raf = new RandomAccessFile(file, "r");
		this.channel = this.raf.getChannel();

		this.filePointer = this.startAtBeginning ? 0 : this.file.length();
		
		final long duration = TimeUnit.MILLISECONDS.convert(
				this.sampleDuration,
				this.timeUnit);
		timer.schedule(this.timerTask = new PollFileTask(), 0, duration);
	}

	/**
	 * 
	 * TODO
	 *
	 * @return
	 */
	public boolean isRunning() {
		return this.channel != null;
	}
	
	/**
	 * 
	 * TODO
	 * 
	 * @author Dan Siviter
	 * @since 20 Mar 2009
	 */
	private class PollFileTask extends TimerTask {
		@Override
		public void run() {
			FileTailer.this.run();
		}
	}
}