package civvi.osgi.desktop.tail.view;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.LinearGradientPaint;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.geom.Point2D;
import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.logging.Level;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

import javax.swing.ImageIcon;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.JTextPane;
import javax.swing.JToolBar;
import javax.swing.SwingUtilities;
import javax.swing.Timer;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.BadLocationException;
import javax.swing.text.DefaultHighlighter;
import javax.swing.text.Document;
import javax.swing.text.Highlighter;

import org.jdesktop.application.Action;
import org.jdesktop.application.Task;
import org.jdesktop.beansbinding.AutoBinding.UpdateStrategy;
import org.jdesktop.beansbinding.BeanProperty;
import org.jdesktop.beansbinding.Bindings;
import org.jdesktop.swingx.painter.MattePainter;

import civvi.osgi.desktop.common.ApplicationUtil;
import civvi.osgi.desktop.swingx.AbstractTask;
import civvi.osgi.desktop.swingx.SwingUtil;
import civvi.osgi.desktop.tail.TailEditable;
import civvi.osgi.desktop.tail.business.FileTailer;
import civvi.osgi.desktop.tail.business.TailEvent;
import civvi.osgi.desktop.tail.business.TailEvent.Type;
import civvi.osgi.desktop.tail.business.TailListener;
import civvi.osgi.desktop.view.AbstractEditorView;

/**
 * A view for tailing a file.
 * 
 * TODO Word-wrapping needs some work!
 * 
 * @author <a href="mailto:dan.siviter@civvi.com">Dan Siviter</a>
 * @since 19 Mar 2009
 */
public class TailView extends AbstractEditorView {
	private static final long serialVersionUID = 7611463052412761135L;

	private final FileTailer tailer;

	private JXTextPane textArea;
	private JScrollPane scrollPane;
	private JTextField findField;

	private Timer findTimer;

	private int buffer = 50 * 1000;
	private boolean scrollLock;
	private boolean wordWrap;
	private int lockedCaretPosition = 0;
	
	private TailEditable tail;

	/**
	 * Creates a tail view for the given file.
	 * 
	 * @param tail
	 */
	public TailView(TailEditable tail) {
		super("tail" + tail.toString());
		try { // FIXME don't like this, can we resource load
			setSmallIcon(new ImageIcon(
					Introspector.getBeanInfo(TailEditable.class).getIcon(
							BeanInfo.ICON_COLOR_16x16)));
		} catch (IntrospectionException ie) {
			throw new RuntimeException(ie);
		}

		this.tailer = new FileTailer();
		this.tailer.addListener(new TailHandler());

		final Color colour1 = SwingUtil.alpha(getBackground(), 0);
		final Color colour2 = getBackground();

		final LinearGradientPaint gradientPaint =
			new LinearGradientPaint(
					new Point2D.Double(0.0d, 0.0d),
					new Point2D.Double(0.0d, 1.0d),
					new float[]{0.0f, 1.0f},
					new Color[]{colour1, colour2});
		setBackgroundPainter(new MattePainter(gradientPaint, true));

		setContent(createContent());
		setToobar(createToolBar());
		setTitle(this.support.getString(NAME, ""));
		setTooltip(this.support.getString(NAME, ""));

		addPropertyChangeListener("tail", new TailChangedHandler());
		
		setTail(tail);
		
		Bindings.createAutoBinding(
				UpdateStrategy.READ,
				this,
				BeanProperty.create("wordWrap"),
				this.textArea,
				BeanProperty.create("wordWrap")).bind();
	}

	/**
	 * @return the content component.
	 */
	private JComponent createContent() {
		this.textArea = new JXTextPane();
		this.textArea .setPreferredSize(new Dimension(350, 350));
		this.textArea.setEditable(false);
		this.textArea.setFont(this.support.getResourceMap().getFont("text.font"));
		this.scrollPane = new JScrollPane(this.textArea);
		return this.scrollPane;
	}

	/**
	 * @return the created toolbar.
	 */
	private JToolBar createToolBar() {
		final JLabel findLabel = new JLabel(this.support.getString("find.label"));
		findLabel.setLabelFor(this.findField = new JTextField());
		this.findField.getDocument().addDocumentListener(new FindChangedHandler());

		return SwingUtil.createTabToolbar(
				getActionMap(),
				findLabel,
				this.findField,
				new JToolBar.Separator(),
				"toggleScrollLock",
				"toggleWordWrap");
	}

	/**
	 * @return the scrollLock
	 */
	public boolean isScrollLock() {
		return this.scrollLock;
	}

	/**
	 * @param scrollLock the scrollLock to set
	 */
	public void setScrollLock(boolean scrollLock) {
		final boolean oldValue = isScrollLock();
		this.scrollLock = scrollLock;
		firePropertyChange("scrollLock", oldValue, isScrollLock());
	}

	/**
	 * @return the wordWrap.
	 */
	public boolean isWordWrap() {
		return wordWrap;
	}

	/**
	 * @param wordWrap the wordWrap to set.
	 */
	public void setWordWrap(boolean wordWrap) {
		final boolean oldValue = isWordWrap();
		this.wordWrap = wordWrap;
		firePropertyChange("wordWrap", oldValue, isWordWrap());
	}

	/**
	 * @return the tail.
	 */
	public TailEditable getTail() {
		return this.tail;
	}

	/**
	 * @param tail the tail to set.
	 */
	public void setTail(TailEditable tail) {
		final TailEditable oldValue = getTail();
		this.tail = tail;
		firePropertyChange("tail", oldValue, getTail());
	}

	/**
	 * Updates the text using the given event.
	 *
	 * @param event
	 */
	private void updateText(TailEvent event) {
		assert SwingUtilities.isEventDispatchThread() : "Use EDT!";

		try {
			final Document doc = this.textArea.getDocument();

			synchronized(doc) {
				if (event.getType() == Type.Decrease) {
					doc.remove(0, doc.getLength());
					this.lockedCaretPosition = 0;
				}

				doc.insertString(doc.getLength(), event.getText(), null);
				if (doc.getLength() > this.buffer) {
					doc.remove(0, doc.getLength() - buffer);
				}

				if (this.scrollLock) {
					this.textArea.setCaretPosition(this.lockedCaretPosition);
				} else {
					this.textArea.setCaretPosition(doc.getLength());
				}
			}
		} catch (BadLocationException ble) {
			if (this.log.isLoggable(Level.WARNING)) {
				this.log.log(Level.WARNING, "Unable to inset text!", ble);
			}
		}

		refreshHighlights();
	}

	/**
	 * Displays dialog for selecting a file to tail.
	 *
	 * @return a task to open the file.
	 */
	public Task<Object, Object> openFile(TailEditable tail) {
		this.textArea.setText("");
		this.lockedCaretPosition = 0;
		setTitle(this.support.getString(
				NAME, "- " + 
				SwingUtil.truncateFilename(tail.getFile(), 25)));
		setTooltip(tail.getFile().getAbsolutePath());
		
		return new OpenTailTask(this.tailer, tail);
	}

	/**
	 * Toggle the the scroll lock.
	 */
	@Action(selectedProperty = "scrollLock")
	public void toggleScrollLock() {
		if (this.log.isLoggable(Level.FINE)) {
			this.log.fine("'Toggle Scroll Lock' action invoked.");
		}
		this.lockedCaretPosition = textArea.viewToModel(SwingUtilities.convertPoint(scrollPane.getViewport(), 0, scrollPane.getViewport().getHeight()/2, textArea));
	}
	
	/**
	 * Toggle word-wrapping.
	 */
	@Action(selectedProperty = "wordWrap")
	public void toggleWordWrap() {
		if (this.log.isLoggable(Level.FINE)) {
			this.log.fine("'Toggle Word-Wrap' action invoked.");
		}
	}

	/**
	 * Refreshes the highlighted text. This is a timed event which will wait
	 * ~1 second before applying the highlight.
	 */
	private void refreshHighlights() {
		if (this.findTimer != null && this.findTimer.isRunning()) {
			this.findTimer.stop();
		}

		this.findTimer = new Timer(1000, new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				doRefreshHighlight();
				findTimer.stop();
				findTimer = null;
			}
		});
		this.findTimer.setDelay(1000);
		this.findTimer.setRepeats(false);
		this.findTimer.start();
	}

	/**
	 * TODO this will apply to the whole document; this is inefficient. We
	 * should be monitoring input and only applying highlights to appended
	 * text. 
	 */
	private void doRefreshHighlight() {
		final Highlighter highlighter = textArea.getHighlighter();
		highlighter.removeAllHighlights();

		final String regex = this.findField.getText();
		final Document document = this.textArea.getDocument();
		
		try {
			final String text = document.getText(0, document.getLength());

			if (regex == null || regex.trim().isEmpty() ||
					text == null || text.trim().isEmpty())
			{
				return;
			}

			final Pattern pattern = Pattern.compile(findField.getText());
			final Matcher matcher = pattern.matcher(text);

			while (matcher.find()) {
				highlighter.addHighlight(
						matcher.start(),
						matcher.end(),
						DefaultHighlighter.DefaultPainter);
			}
		} catch (BadLocationException ble) {
			if (this.log.isLoggable(Level.WARNING)) {
				this.log.log(Level.WARNING, "Unable to apply highlights!", ble);
			}
		} catch (PatternSyntaxException pse) {
			if (this.log.isLoggable(Level.FINE)) {
				this.log.log(Level.FINE, "Unable to apply highlights!", pse);
			}
		}
	}


	// --- Inner Classes ---

	/**
	 * Task to open a file.
	 * 
	 * @author <a href="mailto:dan.siviter@civvi.com">Dan Siviter</a>
	 * @since 20 Mar 2009
	 */
	private class OpenTailTask extends AbstractTask<Object, Object> {
		private final FileTailer tailer;
		private final TailEditable tail;

		/**
		 * Creates a task for opening a file.
		 * 
		 * @param application
		 * @param tailer
		 * @param file
		 */
		public OpenTailTask(
				FileTailer tailer,
				TailEditable tail)
		{
			this.tailer = tailer;
			this.tail = tail;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		protected Object doInBackground() throws Exception {
			if (this.tailer.isRunning()) {
				this.tailer.cancel();
			}
			setTitle(getResourceMap().getString(NAME, "- " + this.tail.getFile().getAbsolutePath()));

			if (this.tailer.isRunning()) {
				throw new IllegalStateException("Tailer is already running.");
			}
			this.tailer.setFile(this.tail.getFile());
			this.tailer.start();
			return null;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		protected void failed(Throwable cause) {
			super.failed(cause);
			ApplicationUtil.showError(getContext(), cause);
		}
	}


	/**
	 * Handles {@link TailEvent}s.
	 * 
	 * @author <a href="mailto:dan.siviter@civvi.com">Dan Siviter</a>
	 * @since 19 Mar 2009
	 */
	private class TailHandler implements TailListener {
		/**
		 * {@inheritDoc}
		 */
		@Override
		public void newLine(final TailEvent event) {
			SwingUtilities.invokeLater(new Runnable() {
				@Override
				public void run() {
					updateText(event);
				}
			});
		}
	}
	
	/**
	 * Handles changes to the regular expression value.
	 * 
	 * @author <a href="mailto:dan.siviter@civvi.com">Dan Siviter</a>
	 * @since 19 Mar 2009
	 */
	private class FindChangedHandler implements DocumentListener {
		/**
		 * {@inheritDoc}
		 */
		@Override
		public void changedUpdate(DocumentEvent e) {
			refreshHighlights();
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public void insertUpdate(DocumentEvent e) {
			refreshHighlights();
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public void removeUpdate(DocumentEvent e) {
			refreshHighlights();
		}
	}

	/**
	 * Handles changes to 'tail' property.
	 * 
	 * @author <a href="mailto:dan.siviter@civvi.com">Dan Siviter</a>
	 * @since 19 Mar 2009
	 */
	private class TailChangedHandler implements PropertyChangeListener {
		@Override
		public void propertyChange(PropertyChangeEvent evt) {
			support.fire(openFile((TailEditable) evt.getNewValue()));
		}
	}

	/**
	 * 
	 * @author <a href="mailto:dan.siviter@civvi.com">Dan Siviter</a>
	 * @since v?.?.? [15 April 2008]
	 */
	public class JXTextPane extends JTextPane {
		private static final long serialVersionUID = 4867575036971832793L;

		private boolean wordWrap;
		
		/**
		 * @return the wordWrap.
		 */
		public boolean isWordWrap() {
			return wordWrap;
		}

		/**
		 * @param wordWrap the wordWrap to set.
		 */
		public void setWordWrap(boolean wordWrap) {
			final boolean oldValue = isWordWrap();
			this.wordWrap = wordWrap;
			firePropertyChange("wordWrap", oldValue, isWordWrap());
		}
		
		@Override
		public void setSize(Dimension d) {
			if (wordWrap) {
				super.setSize(d);
			}
			
			if (d.width < getParent().getSize().width) {
				d.width = getParent().getSize().width;
			}

			super.setSize(d);
		}

		@Override
		public boolean getScrollableTracksViewportWidth() {
			if (wordWrap) {
				super.getScrollableTracksViewportWidth();
			}
			return false;
		}
	}
}
